/* ====================================================
 * CUTEst interface for cg_descent       May 1, 2018
 *
 * W. Hager
 *
 * (Based on CUTEr gencma.c of D. Orban, Feb 3, 2003)
 * (CUTEst evolution, Nick Gould, Apr 2, 2014)
 * ====================================================
 */

/* macros */
#define CG_DESCENTMA

#define MAXLINE 256

#ifdef __cplusplus
extern "C" {   /* To prevent C++ compilers from mangling symbols */
#endif

#include "cutest.h"
#include "cg_descent.h"

#ifdef Isg95
#define MAINENTRY MAIN_
#else
#define MAINENTRY main
#endif

void cg_value
(
    CGFLOAT *f,
    CGFLOAT *x,
    CGINT    n
) ;

void cg_grad
(
    CGFLOAT  *g,
    CGFLOAT  *x,
    CGINT     n
) ;

void cg_valgrad
(
    CGFLOAT  *f,
    CGFLOAT  *g,
    CGFLOAT  *x,
    CGINT     n
) ;

/* global variables */
    integer CUTEst_nvar;        /* number of variables */
    integer CUTEst_ncon;        /* number of constraints */
    logical cute_true = TRUE_ ;
    logical cute_false = FALSE_ ;

/* main program */
int MAINENTRY( void )
{
    double walltime ; /* wall time  */
    char *fname = "OUTSDIF.d"; /* CUTEst data file */
    integer funit = 42;        /* FORTRAN unit number for OUTSDIF.d */
    integer io_buffer = 11;    /* FORTRAN unit for internal i/o */
    integer iout = 6;          /* FORTRAN unit number for error output */
    integer ierr;              /* Exit flag from OPEN and CLOSE */
    integer status;            /* Exit flag from CUTEst tools */
    char fullpathname [1024] ;
    char *cutest_dir ;

    integer     v_order, nnzh, H_anz, ncol, *h_row, *h_col ;
    CGFLOAT     *x, *lo, *hi, *zeros, *Hx, *H_val ;
    char        qpprob [MAXLINE+1], uprob [MAXLINE+1], *uname, *qpname, *pname ;

    doublereal  calls[7], cpu[2] ;
    integer     ExitCode;
    int         QP, status_cg, U ;
    CGFLOAT     s, t ;
    CGINT       Ncol, i,
                *H_row, *H_col, *Hp, *Hi ;

    FILE *QPfile, *Ufile ;

    /* for an LP or QP, fadjust stores constant term in objective */
    CGFLOAT fadjust = CGZERO ;

    CGFLOAT *c = NULL ; /* linear cost vector */

    /* Open problem description file OUTSDIF.d */
    ierr = 0;
    FORTRAN_open( &funit, fname, &ierr ) ;
    if( ierr )
    {
        printf("Error opening file OUTSDIF.d.\nAborting.\n") ;
        exit(1) ;
    }

    /* --- Setup for unconstrained problem ---------- */
    printf ("Initializing CGdata structure.\n") ;
    CGdata *cgdata = cg_setup () ;
    if ( cgdata == NULL )
    {
            cg_error (-1, __FILE__, __LINE__, "cg_setup failed") ;
    }
    else
    {
        printf ("Successfully initialized CGdata structure.\n") ;
    }
    CGparm *cgparm = cgdata->Parm ;
    CGstat *Stat = cgdata->Stat ;

    /* Determine problem size */
    CUTEST_udimen( &status, &funit, &CUTEst_nvar) ;
    if (status)
    {
        printf("** CUTEst error, status = %ld, aborting\n", (LONG) status);
        exit(status);
    }
    ncol = CUTEst_nvar ; /* integer */
    Ncol = ncol ;        /* CGINT */

    /* v_order determines order of lin and nonlin variables */
    /* if v_order  = 1 then lin come before nonlin */
    /* if v_order  = 2 then nonlin come before lin */
    /* if v_order != 1 or 2 then order does not matter */
    v_order = 2 ;

    /* Reserve memory for variables, bounds, and multipliers */
    /* and call appropriate initialization routine for CUTEst */
    x =      (CGFLOAT *) malloc (Ncol*sizeof (CGFLOAT)) ;
    lo =     (CGFLOAT *) malloc (Ncol*sizeof (CGFLOAT)) ;
    hi =     (CGFLOAT *) malloc (Ncol*sizeof (CGFLOAT)) ;
    CUTEST_usetup ( &status, &funit, &iout, &io_buffer, &ncol, x, lo, hi ) ;

    if (status)
    {
        printf("** CUTEst error, status = %ld, aborting\n", (LONG) status);
        exit(status);
    }

    /* Store zero in an array named zeros, it is used below. */
    zeros = (CGFLOAT *) malloc (Ncol*sizeof (CGFLOAT)) ;
    cg_initx (zeros, CGZERO, Ncol) ;

    /* Get problem name */
    pname = (char *) malloc ((FSTRING_LEN+1)*sizeof (char)) ;
    CUTEST_probname( &status, pname ) ;
    if (status)
    {
        printf("** CUTEst error, status = %ld, aborting\n", (LONG) status) ;
        exit(status) ;
    }

    /* Make sure to null-terminate problem name */
    pname[FSTRING_LEN] = '\0';
    i = FSTRING_LEN - 1;
    while(i-- > 0 && pname[i] == ' ')
    {
        pname[i] = '\0';
    }

    /* Print problem name */
    printf ("\n Problem: %s (n = %ld)\n", pname, (LONG) Ncol ) ;

    /* read the full path to the pasa/CUTEst directory and store it in the
       variable cutest_dir */
#include ".cutest_location"

    /* See if the problem is in the list of unconstrained problem.
       If not, then terminate with an error message. */
    strcpy (fullpathname, cutest_dir) ;
    Ufile = fopen (strcat (fullpathname, "/classU"), "r") ;
    U = FALSE ;
    while (fgets (uprob, MAXLINE, Ufile) != (char *) NULL)
    {
        for (uname = uprob; *uname; uname++)
        {
            if (isspace (*uname)) *uname = '\0' ;
        }
        uname = uprob ;
        if ( strcmp (uname, pname) == 0 )
        {
            U = TRUE ;
            break ;
        }
    }

    if ( U == FALSE )
    {
        printf ("Problem %s was not found in the list of "
                "unconstrained problems in the file %s\n",
                pname, strcat (fullpathname, "/classU")) ;
        printf ("Either the problem is constrained, or the file should be "
                "updated to include this new unconstrained problem.\n") ;
        cg_error (-1, __FILE__, __LINE__, "STOP") ;
    }

    /* see if the problem has a quadratic objective by comparing the name
       pname of the test problem to the names qpprob of the unconstrained
       QPs contained in the file classQPU */
    strcpy (fullpathname, cutest_dir) ;
    QPfile = fopen (strcat (fullpathname, "/classUQP"), "r") ;
    QP = FALSE ;
    while (fgets (qpprob, MAXLINE, QPfile) != (char *) NULL)
    {
        for (qpname = qpprob; *qpname; qpname++)
        {
            if (isspace (*qpname)) *qpname = '\0' ;
        }
        qpname = qpprob ;
        if ( strcmp (qpname, pname) == 0 )
        {
            QP = TRUE ;
            break ;
        }
    }

    /* initial cgdata structure setup */
    cgdata->x = x ;
    cgdata->n = Ncol ;

    /* If the problem has a quadratic objective, then extract the
       Hessian matrix and the linear term in the objective. */
    if ( QP == TRUE )
    {
        printf ("the problem has a quadratic objective\n") ;
        /* nnzh = the number of nonzeros required to store the sparse Hessian 
           matrix in coordinate format */
        if (status)
        {
            printf("** CUTEst error, status = %ld, aborting\n", (LONG) status) ;
            exit(status);
        }

        /* Reserve memory for H_val, H_row, H_col */
        CUTEST_udimsh ( &status, &nnzh );
        H_val = (CGFLOAT *) malloc (nnzh*sizeof (CGFLOAT)) ;
        H_row = (CGINT *)   malloc (nnzh*sizeof (CGINT)) ;
        H_col = (CGINT *)   malloc (nnzh*sizeof (CGINT)) ;
        h_row = (integer *) malloc (nnzh*sizeof (integer)) ;
        h_col = (integer *) malloc (nnzh*sizeof (integer)) ;

        /* Determine the nonzero values in the Hessian */
        CUTEST_ush ( &status, &ncol, zeros, &H_anz, &nnzh, H_val, h_row, h_col);

        if (status)
        {
            printf("** CUTEst error, status = %ld, aborting\n", (LONG) status) ;
            exit(status) ;
        }
        if ( nnzh != H_anz )
        {
            printf ("nnzh (%ld) != H_anz (%ld)\n", (LONG) nnzh, (LONG) H_anz) ;
            cg_error (-1, __FILE__, __LINE__, "STOP") ;
        }

        /* convert from fortran to c indexing, subtract 1 */
        for (i = 0; i < H_anz; i++)
        {
            H_row [i] = h_row [i] - 1 ;
            H_col [i] = h_col [i] - 1 ;
        }

        status = cg_convert_triple_to_sparse (&Hp, &Hi, &Hx, &Ncol, &Ncol,
                                      H_row, H_col, H_val, H_anz, TRUE, TRUE) ;

        if ( status == SOPT_OUT_OF_MEMORY )
        {
            printf ("In cg_main.c, there was not enough memory to\n"
                    "convert the CUTEst Hessian of a quadratic to the sparse\n"
                    "matrix format used by cg_descent.\n") ;
            cg_error (-1, __FILE__, __LINE__, "execution halted") ;
        }
        else if ( status == SOPT_ERROR_IN_INPUT_MATRIX )
        {
            printf ("In cg_main.c, there appears to be an error in the\n"
                    "triples describing the CUTEst Hessian of a quadratic.\n") ;
            cg_error (-1, __FILE__, __LINE__, "execution halted") ;
        }
        /* if a vanishing sparse matrix element was discovered, it was
           ignored when building the sparse matrix for pasa */

        /* Determine c in objective function */
        c = (CGFLOAT *) malloc (Ncol*sizeof (CGFLOAT)) ;
        CUTEST_uofg (&status, &ncol, zeros, &fadjust, c, &cute_true);
        /* store the linear cost vector and the sparse Hessian matrix in
           the cgdata structure */
        cgdata->c = c ;
        cgdata->Hp = Hp ;
        cgdata->Hi = Hi ;
        cgdata->Hx = Hx ;
    }
    else /* the problem is not quadratic */
    {
        cgdata->value = cg_value ;
        cgdata->grad = cg_grad ;
        cgdata->valgrad = cg_valgrad ;
    }

    /* set new parameter values for CG here, otherwise default values are used.
       For example:

       cgparm->grad_tol = 1.e-8 ; */ /* default is 1.e-6 */ ;

    /* Call the optimizer */

    /* time run using wall clock, call several times to exclude startup cost */
    walltime = cg_timer () ;
    walltime = cg_timer () ;

    status_cg = cg_descent (cgdata) ;

    walltime = cg_timer () - walltime ;

    ExitCode = 0;

    /* Get CUTEst statistics */
    /* CUTEST_ureport( &status, calls, cpu) ;*/
    if (status)
    {
        printf("** CUTEst error, status = %ld, aborting\n", (LONG) status) ;
        exit(status) ;
    }

    /* print unformatted cg_descent statistics */
    /* printf ("!!%10s %6ld %7ld %7ld %7ld %5i %16.7e %16.7e %11.6f\n",
             pname, (LONG) Ncol, (LONG) Stat->iter, (LONG) Stat->nfunc,
             (LONG) Stat->ngrad, status_cg, Stat->err, Stat->f+fadjust,
             walltime) ;*/

    printf("\n\n *********************** CG statistics **************"
           "**********\n\n") ;
    printf("Code used                 : cg_descent\n") ;
    printf("Problem                   : %-s\n", pname) ;
    printf("# variables               = %-10ld\n\n",
          (LONG) Ncol) ;
    printf("# cg iterations           = %-10ld\n\n",
          (LONG) Stat->iter) ;
    printf("# cg function evals       = %-10ld\n\n",
          (LONG) Stat->nfunc) ;
    printf("# cg gradient evals       = %-10ld\n\n",
          (LONG) Stat->ngrad) ;
    printf("|| g ||                   = %-16.7e\n", Stat->err) ;
    printf("Final f                   = %-16.7e\n", Stat->f+fadjust) ;
    cg_value (&t, x, CUTEst_nvar) ;
    printf("Function value at final x = %-16.7e\n", t) ;
    printf("Solve time                = %-11.6f seconds\n", walltime) ;
    printf("\n ***********************************************************"
           "*******\n\n") ;

    /* Print lines to clearly separate problems */

    printf (" ====================================================\n\n" ) ;

    ierr = 0;
    FORTRAN_close( &funit, &ierr ) ;
    if ( ierr )
    {
        printf( "Error closing %s on unit %ld.\n", fname, (LONG) funit ) ;
        printf( "Trying not to abort.\n" ) ;
    }

    /* Free workspace */
    free (zeros) ;
    free (pname) ;
    free (x) ;
    free (lo) ;
    free (hi) ;
    if ( QP == TRUE )
    {
        free (Hx) ;
        free (Hi) ;
        free (Hp) ;
        free (c) ;
        free (h_row) ;
        free (h_col) ;
        free (H_val) ;
        free (H_row) ;
        free (H_col) ;
    }

    cg_terminate (&cgdata) ;

    /* end cutest */
    CUTEST_uterminate( &status ) ;

    /* close files */
    fclose(QPfile) ;
    fclose(Ufile) ;

    return 0;
}

#ifdef __cplusplus
}    /* Closing brace for  extern "C"  block */
#endif
void cg_value
(
    CGFLOAT *f,
    CGFLOAT *x,
    CGINT N
)
{
    integer status, n ;

    n = N ;
    CUTEST_ufn( &status, &n, x, f) ;
    if ((status == 1) || (status == 2))
    {
        printf("** CUTEst error, status = %ld, aborting\n", (LONG) status) ;
        exit(status) ;
    }
/*printf ("function value: %e\n", *f) ;*/
    return ;
}

void cg_grad
(
    CGFLOAT *g,
    CGFLOAT *x,
    CGINT N
)
{
    integer n, status;
    n = N ;
    CUTEST_ugr ( &status, &n, x, g) ;
    if ((status == 1) || (status == 2))
    {
        printf("** CUTEst error, status = %ld, aborting\n", (LONG) status);
        exit(status);
    }
}

void cg_valgrad
(
    CGFLOAT *f,
    CGFLOAT *g,
    CGFLOAT *x,
    CGINT N
)
{
    integer n, status;
    n = N ;
    CUTEST_uofg( &status, &n, x, f, g, &cute_true) ;
    if ((status == 1) || (status == 2))
    {
        printf("** CUTEst error, status = %ld, aborting\n", (LONG) status);
        exit(status);
    }
    return ;
}
