/* -------------------------- Problem Details -------------------------- 
    This is a demo for solving the problem:
    
                min     0.5 sum_{i=1}^n exp (x_i) - sqrt(i)*x_i

    The routines for evaluating the objective and its gradient appear near
    the end of this file. */

#include "cg_descent.h"

/* prototypes */
void value
(
    CGFLOAT *f,
    CGFLOAT *x,
    CGINT    n
) ;

void grad
(
    CGFLOAT *g,
    CGFLOAT *x,
    CGINT    n
) ;

void valgrad
(
    CGFLOAT *f,
    CGFLOAT *g,
    CGFLOAT *x,
    CGINT    n
) ;

int main (void)
{
    CGINT i, n ;

    CGdata *cgdata = cg_setup () ; /* initialize a CGdata structure */

    n = cgdata->n = 10000 ; /* set problem dimension to 1000000 */
    /* cgdata->x can be malloc'd and a starting guess provided.
       If there is no assignment to cgdata->x, then cg_descent will malloc
       cgdata->x and set the starting guess to x = zero. */

    /* the routines for evaluating the objective and its gradient appear at
       the end of this file */
    cgdata->value = value ;
    cgdata->grad  = grad ;

    /* run the code and time it */
    CGFLOAT walltime = cg_timer () ; /* call it once to exclude startup cost */
    walltime = cg_timer () ;

    cg_descent (cgdata) ;

    walltime = cg_timer () - walltime ;
    printf ("Time of Run 1: %e\n", walltime) ;
    printf ("Print first 4 components of x:\n") ;
    for (i = 0; i < 4; i++) printf ("  %10.7f\n", cgdata->x [i]) ;
    printf ("\n") ;

    /* By default, cg_descent does not print statistics for the run.
       If you want the code to print the run statistics, then the
       parameter PrintStat, whose default value is FALSE, needs to be
       changed to TRUE:
    cgdata->Parm->PrintStat = TRUE ;

    Alternatively, after the run is complete, the statistics can be printed
    using the cg_print_stat routine as we show below: */

    printf ("cg statistics for run 1:\n") ;
    cg_print_stat (cgdata) ;

    /* With a loss in speed, the valgrad argument of cg_descent has been
       omitted so far. Below, we also utilize the valgrad routine which
       evaluates the objective value and its gradient simultaneously.
       If the cg_descent has to evaluate both the objective and its gradient,
       it is usually faster to evaluate them simultaneously rather than
       evaluate first one and then the other. The running time comparison
       on a Dell T7610 was 0.00212 s for Run 2 versus 0.001476 s for Run 3.
       The number of iterations and function/gradient evaluations were
       identical. */
    cgdata->valgrad = valgrad ;
    /* since we want to reproduce run 1 but with different evaluation routines,
       we must set the starting guess to zero, the same starting guess
       used in run 1. */
    cg_initx (cgdata->x, CGZERO, n) ;

    /* Note that the run status message can be eliminated as adjusting the
       status parameter as shown below: */
    cgdata->Parm->PrintStatus = FALSE ;
    walltime = cg_timer () ;

    cg_descent (cgdata) ;

    walltime = cg_timer () - walltime ;
    printf ("Time of Run 2 (using valgrad): %e\n", walltime) ;
    printf ("Print first 4 components of x:\n") ;
    for (i = 0; i < 4; i++) printf ("  %10.7f\n", cgdata->x [i]) ;
    printf ("\n") ;

    /* Note that the statistics are also returned in the structure
       cgdata->Stat. Below we print the number of gradient evaluations
       using the statistics structure. Note that cgdata->Stat contains
       the statistics structure even when cgdata->Parm->Stat = FALSE. */
    printf ("Number of Function Evaluations from the Stat structure: %i\n",
            cgdata->Stat->nfunc) ;
    printf ("Number of Gradient Evaluations from the Stat structure: %i\n",
            cgdata->Stat->ngrad) ;

    /* Although we turned off the automatic printing of the run status,
       we can still access the run status in several different ways.
       For example, it is returned by the cg_descent routine; that is,

       status = cg_descent (cgdata);

       second, the run status is printed by the routine cg_print_status
       as we show below: */

    printf ("\nPrint status of run 2:\n") ;
    cg_print_status (cgdata) ;

    /* Finally, we free the workspace created by cg_setup, including the memory
       allocated by cg_descent for cgdata->x */
    cg_terminate (&cgdata) ;
}

void value
(
    CGFLOAT *f,
    CGFLOAT *x,
    CGINT    n
)
{
    CGFLOAT t, s ;
    CGINT i ;
    s = 0. ;
    for (i = 0; i < n; i++)
    {
        t = i+1 ;
        t = sqrt (t) ;
        s += exp (x [i]) - t*x [i] ;
    }
    *f = s ;
}

void grad
(
    CGFLOAT *g,
    CGFLOAT *x,
    CGINT    n
)
{
    CGFLOAT t ;
    CGINT i ;
    for (i = 0; i < n; i++)
    {
        t = i + 1 ;
        t = sqrt (t) ;
        g [i] = exp (x [i]) -  t ;
    }
}

void valgrad
(
    CGFLOAT *f,
    CGFLOAT *g,
    CGFLOAT *x,
    CGINT    n
)
{
    CGFLOAT s ;
    CGINT i ;
    s = 0. ;
    for (i = 0; i < n; i++)
    {
        CGFLOAT const  t = sqrt (i+1) ;
        CGFLOAT const xi = x [i] ;
        CGFLOAT const ex = exp (xi) ;
        s += ex - t*xi ;
        g [i] = ex -  t ;
    }
    *f = s ;
}
