/* -------------------------- Problem Details -------------------------- 
    This is a demo for solving the problem:
    
                min     0.5 sum_{i=1}^n  i*x_i^2 + x_i

    The primary purpose of this demo is to show the user how to utilize
    cg_descent's specialized routines for handling a quadratic objective. */

#include "cg_descent.h"

/* prototypes */
void hprod
(
    CGFLOAT   *val,
    CGFLOAT     *x,
    CGINT        n
) ;

int main (void)
{
    int status ;
    CGINT i, n, *HTi, *HTj ;
    CGFLOAT *HTx ;

    CGdata *cgdata = cg_setup () ; /* initialize a CGdata structure */
    if ( cgdata == NULL)
    {
        cg_error (-1, __FILE__, __LINE__, "cg_setup failed!") ;
    }

    n = cgdata->n = 100 ; /* set problem dimension to 100 */
    /* since cgdata->x is not specified, the starting guess is x = 0 */

    /* Since the objective is a quadratic, we will provide the objective's
       linear cost vector and the Hessian matrix */
    /* linear cost vector space */
    cgdata->c = cg_malloc (&status, n, sizeof (CGFLOAT)) ;
    if ( status == SOPT_OUT_OF_MEMORY )
    {
        printf ("In demoQP, ran out of memory when allocating the linear\n"
                "cost vector c for a quadratic\n") ;
    }
    /* cg_descent has a special routine for setting all components of a real
       vector to the same value */
    cg_initx (cgdata->c, 1, n) ;

    /* Input the Hessian of the quadratic using triples form; that is,
       using arrays Ti, Tj, and Tx associated with the row indices,
       column indices, and the nonzero matrix elements. */
    
    HTi = (CGINT *)   cg_malloc (&status, n, sizeof (CGINT)) ;
    HTj = (CGINT *)   cg_malloc (&status, n, sizeof (CGINT)) ;
    HTx = (CGFLOAT *) cg_malloc (&status, n, sizeof (CGFLOAT)) ;
    /* cg_descent is coded in C where the array indices start at 0.
       Hence, the i-th diagonal element of the Hessian is i+1. */
    for (i = 0; i < n; i++)
    {
        HTi [i] = i ; HTj [i] = i ; HTx [i] = i + 1 ;
    }
    cgdata->HTi = HTi ; /* row indices of nonzeros in Hessian */
    cgdata->HTj = HTj ; /* col indices of nonzeros in Hessian */
    cgdata->HTx = HTx ; /* nonzero elements of the Hessian */
    cgdata->Hnz = n ;   /* number of nonzeros in Hessian */

    /* by default, the run statistics are not printed. Set PrintStat to TRUE */
    cgdata->Parm->PrintStat = TRUE ;
    printf ("\nRun 1: print run statistics and first 4 components of x.\n") ;

    cg_descent (cgdata) ;

    printf ("x [1:4] =\n") ;
    for (i = 0; i < 4; i++) printf ("    %10.7f\n", cgdata->x [i]) ;

    /* do not print the complete statistics, only print selected statistics
       from the statistics structure returned in cgdata */
    printf ("\nRun 2: use the Stat structure to print the number\n"
            "        of function and gradient evaluations. The problem is\n"
            "        solved quickly since the starting guess in cgdata->x\n"
            "        is the solution computed in Run 1.\n") ;
     
    /* Set PrintStat to FALSE since we only want to print selected statistics.*/
    cgdata->Parm->PrintStat = FALSE ;

    cg_descent (cgdata) ;

    /* access the number of function and gradient evaluations from the
       Stat structure found in cgdata */
    printf ("Number of Function Evaluations: %i\n", cgdata->Stat->nfunc) ;
    printf ("Number of Gradient Evaluations: %i\n", cgdata->Stat->ngrad) ;

    /* free the memory allocated by the user for c */
    free (cgdata->c) ;

    /* free the workspace created by cg_setup, including the memory
       allocated by cg_descent for cgdata->x */
    cg_terminate (&cgdata) ;
}
