/* --- Routines required by more than one mex function --- */
/* This file contains function definitions associated with 
   calls in multiple MATLAB MEX functions found within SuiteOPT */

/* ---------- Include header files ---------- */
#include "suiteopt_matlab.h"

/* --- Wrappers for function handles to pass to C functions ----------------- */
#ifdef SUITEOPT_MATLAB_OBJ
/* Wrapper to evaluate user's objective function */
/* The arguments are converted to MATLAB's format, the user's function stored 
   in a global variable, suiteopt_matlab_value, and is evaluated using the  
   mxCallMATLAB statement. The output is converted back to correct type. */
void suiteopt_matlab_value
(
    SOPTFLOAT *val, /* Final (scalar) function value stored in val */
    SOPTFLOAT *x,   /* Evaluate objective function at x */
    SOPTINT n       /* Length of x */
)
{
    /* Initialize variables */
    mxArray *F, *X, *ppFevalRhs [2] ;

    #if DEBUG_CGDESCENT_MEX
    mexPrintf("Evaluating suiteopt_matlab_value\n") ;
    #endif

    /* Set initial values */
    X = mxCreateDoubleMatrix (0, 0, mxREAL) ;
    mxFree (mxGetPr (X)) ;
    mxSetPr(X, x) ;
    mxSetN(X, 1) ;
    mxSetM(X, n) ;
   
    /* Evaluate function in MATLAB */
    ppFevalRhs[0] = suiteopt_value ;
    ppFevalRhs[1] = X ;
    /* This statement uses MATLAB's feval to evaluate cg_value (X) and
       return the single output in F */
    mexCallMATLAB(1, &F, 2, ppFevalRhs, "feval") ;
    mxSetPr(X, NULL) ;

    /* Copy F into val */ 
    memcpy(val, mxGetPr(F), sizeof(double)) ;

    /* Free memory */
    mxDestroyArray (F) ;

    /* Exit function */
    return ; 
}

/* Wrapper to evaluate user's gradient function */
void suiteopt_matlab_grad
(
    SOPTFLOAT *g, /* Final gradient stored in g */
    SOPTFLOAT *x, /* Evaluate gradient at x */
    SOPTINT    n  /* Length of x */
)
{
    /* Initialize variables */
    mxArray *G, *ppFevalRhs [2], *X ;

    #if DEBUG_CGDESCENT_MEX
    mexPrintf("Evaluating suiteopt_matlab_grad\n") ;
    #endif

    /* Set initial values */
    X = mxCreateDoubleMatrix(0, 0, mxREAL) ;
    mxFree(mxGetPr(X)) ;
    mxSetPr(X, x) ;
    mxSetN(X, 1) ;
    mxSetM(X, n) ;
   
    /* Evaluate function in MATLAB */
    ppFevalRhs[0] = suiteopt_grad ;
    ppFevalRhs[1] = X ;
    mexCallMATLAB(1, &G, 2, ppFevalRhs, "feval") ;
    mxSetPr(X, NULL) ;

    /* Copy values from G into g */ 
    memcpy(g, mxGetPr(G), sizeof(SOPTFLOAT)*n) ;

    /* Free memory */
    mxDestroyArray (G) ;

    /* Exit function */
    return ; 
}

/* Wrapper to evaluate user's valgrad function */
void suiteopt_matlab_valgrad
(
    SOPTFLOAT *val, /* Final (scalar) function value stored in val */
    SOPTFLOAT *g,   /* Final gradient stored in g */
    SOPTFLOAT *x,   /* Evaluate objective function at x */
    SOPTINT n       /* Length of x */
)
{
    /* Initialize variables */
    mxArray *X, *FG[2], *ppFevalRhs[2] ;

    #if DEBUG_CGDESCENT_MEX
    mexPrintf("Evaluating suiteopt_matlab_valgrad\n") ;
    #endif

    /* Convert x to MATLAB's mxArray format */
    X = mxCreateDoubleMatrix(0,0,mxREAL) ;
    mxFree(mxGetPr(X)) ;
    mxSetPr(X, x) ;
    mxSetN(X, 1) ;
    mxSetM(X, n) ;

    /* Evaluate function */
    ppFevalRhs[0] = suiteopt_valgrad ;
    ppFevalRhs[1] = X ;
    mexCallMATLAB(2, FG, 2, ppFevalRhs, "feval") ; 
    mxSetPr(X, NULL) ;

    /* Copy values from FG into val and g */
    memcpy(val, mxGetPr(FG[0]), sizeof(double)) ;
    memcpy(g, mxGetPr(FG[1]), sizeof(double)*n) ;

    /* Free memory */
    mxDestroyArray (FG [0]) ;
    mxDestroyArray (FG [1]) ;

    /* Exit function */
    return ;
}
#endif

#ifdef SUITEOPT_MATLAB_HPROD
/* Routine for computing hessian (with free indices) times vector (for PASA) */
void suiteopt_matlab_hprod
(
    SOPTFLOAT         *p, /* Product Hx stored in p */
    SOPTFLOAT         *x, /* Multiply hessian times x */
    SOPTINT const *ifree, /* Indices of free components */
    SOPTINT const      n, /* Length of x */
    SOPTINT const     nf  /* Number of free components */
)
{
    /* Initialize variables */
    mxArray *I, *P, *X, *ppFevalRhs[3] ;
    double *nzIndices ;
    long i ;

    #if DEBUG_SUITEOPT_MEX
    mexPrintf("Evaluating suiteopt_matlab_hprod in matlab\n") ;
    #endif

    /* Set initial values */
    X = mxCreateDoubleMatrix(0, 0, mxREAL) ;
    mxFree(mxGetPr(X)) ;
    mxSetPr(X, x) ;
    mxSetN(X, 1) ;
    mxSetM(X, n) ;

    /* Initialize nzIndices */
    I = mxCreateDoubleMatrix(0, 0, mxREAL) ;
    mxFree(mxGetPr(I)) ;
    mxSetN(I, 1) ;
    if ( ifree == NULL ) /* means all element of x are nonzero */
    {
        nzIndices = mxMalloc (n * sizeof(SOPTINT)) ;
        mxSetPr(I, nzIndices) ;
        mxSetM(I,n) ;
        for (i = 0; i < n; i++)
        {
            nzIndices [i] = i + 1 ; /* +1 to convert C indices to MATLAB */
        }
    }
    else
    {
        nzIndices = mxMalloc (nf * sizeof(SOPTINT)) ;
        mxSetPr(I, nzIndices) ;
        mxSetM(I,nf) ;
        for (i = 0; i < nf; i++)
        {
            /* +1 to convert C indices to MATLAB */
            nzIndices [i] = ifree [i] + 1 ;
        }
    }

    /* Set initial values */
    ppFevalRhs [0] = suiteopt_hprod ;
    ppFevalRhs [1] = X ;
    ppFevalRhs [2] = I ;

    /* Evaluate function */
    mexCallMATLAB(1, &P, 3, ppFevalRhs, "feval") ; 

    /* Copy values from P into p */ 
    memcpy(p, mxGetPr(P), sizeof(SOPTFLOAT)*n) ;

    /* Free memory */
    mxSetPr(X, NULL) ;
    mxSetPr(I, NULL) ;
    mxFree(nzIndices) ;

    #if DEBUG_SUITEOPT_MEX
    mexPrintf("Finished evaluating suiteopt_matlab_hprod in matlab\n") ;
    #endif

    /* Exit function */
    return ; 
}
#endif

#ifdef SUITEOPT_MATLAB_CGHPROD
/* Routine for computing CGDESCENT hessian times vector */
void suiteopt_matlab_cghprod
(
    SOPTFLOAT *p, /* Product Hx stored in p */
    SOPTFLOAT *x, /* Multiply hessian times x */
    SOPTINT    n  /* Length of x */
)
{
    /* Initialize variables */
    mxArray *P, *X, *ppFevalRhs[3] ;
    long i ;

    #if DEBUG_CGDESCENT_MEX
    mexPrintf("Evaluating suiteopt_matlab_cghprod\n") ;
    #endif

    /* Set initial values */
    X = mxCreateDoubleMatrix(0, 0, mxREAL) ;
    mxFree(mxGetPr(X)) ;
    mxSetPr(X, x) ;
    mxSetN(X, 1) ;
    mxSetM(X, n) ;

    /* Set initial values */
    ppFevalRhs [0] = cg_hprod ;
    ppFevalRhs [1] = X ;

    /* Evaluate function */
    mexCallMATLAB(1, &P, 2, ppFevalRhs, "feval") ; 

    /* Copy values from P into p */ 
    memcpy(p, mxGetPr(P), sizeof(SOPTFLOAT)*n) ;

    /* Free memory */
    mxSetPr(X, NULL) ;

    /* Exit function */
    return ; 
}
#endif

/* --- Routines for importing problem data to MEX functions ----------------- */
/* Returns pointer to float with data_name if found in user struct,
   Otherwise returns NULL */
SOPTFLOAT * suiteopt_matlab_get_float
(
    const mxArray *options, /* User provided struct to mex function */
    LONG                *n, /* Pointer to current problem dimension */
    char        *data_name  /* String for name of element to set pointer for */
)
{
    /* Initialize variables */
    LONG len ;
    mxArray *field ;

    if ((field = mxGetField(options, 0, data_name)) != NULL)
    {
        if (mxIsClass(field, "double"))
        {
            #ifdef DEBUG_SUITEOPT_MEX
            mexPrintf("%s::User provided data for %s.\n", 
                      SUITEOPT_MATLAB_SOLVER, data_name) ;
            #endif

            /* Number of components of data */
            len = SUITEOPTMAX(mxGetM(field), mxGetN(field)) ;
            /* Check if problem dimension was EMPTY */
            if (*n <= 0)
            {
                #ifdef DEBUG_SUITEOPT_MEX
                mexPrintf("%s::Length of %s is %d and problem dimension"
                          " is %d.\n",
                          SUITEOPT_MATLAB_SOLVER, data_name, len, *n) ;
                mexPrintf("%s::Setting problem dimension equal to "
                          "%d.\n", SUITEOPT_MATLAB_SOLVER, len) ;
                #endif
                /* Set value of n to length of data */
                *n = len ;
            }
            /* dimension was provided previously but does not match
               the dimension of the vector, then halt with error */
            else if (len != *n)
            {
                mexPrintf("%s::Length of %s is %ld while previously\n"
                          "the dimension was implied to be %ld.\n",
                          SUITEOPT_MATLAB_SOLVER,  data_name, len, *n) ;
                mexErrMsgTxt("Terminating execution.") ;
            }
            /* Return pointer to data */
            return mxGetPr (field) ;
        }
        else /* data must be of type double, Print error message */
        {
            mexPrintf("%s::Input data provided for %s must be of type double.\n"
                      ,SUITEOPT_MATLAB_SOLVER, data_name) ;
            mexErrMsgTxt("Terminating execution.\n") ;
        }
    }
    else /* data not provided to solver */
    {
        #ifdef DEBUG_SUITEOPT_MEX
        mexPrintf("%s::Input data for %s not provided.\n", 
                  SUITEOPT_MATLAB_SOLVER, data_name) ;
        #endif
    }
    /* Data not provided. Return NULL pointer */
    return NULL ;
}

/* Sets scalar with data_name to output[0] if found in user struct */
void suiteopt_matlab_get_scalar
(
    const mxArray *options, /* User provided struct to mex function */
    SOPTFLOAT *output,      /* Ptr to var to set equal to usr data (if given) */
    char *data_name         /* String for name of element to set pointer for */
)
{
    /* Initialize variables */
    mxArray *field ;
    int status ;

    /* see if data_name appears in the options list */
    if ((field = mxGetField(options, 0, data_name)) != NULL)
    {
        #ifdef DEBUG_SUITEOPT_MEX
        mexPrintf("%s::User provided data for %s.\n", 
                  SUITEOPT_MATLAB_SOLVER, data_name) ;
        #endif

        /* check that the associated field is double */
        if (mxIsClass(field, "double"))
        {
            /* Copy value for user data */
            *output = mxGetScalar (field) ;  

            #ifdef DEBUG_SUITEOPT_MEX
            mexPrintf("%s::Successfully stored value of %f for %s.\n", 
                      SUITEOPT_MATLAB_SOLVER, *output, data_name) ;
            #endif
        }
        else /* field corresponding to data_name not of type double */
        {
            /* print error message */
            mexPrintf("%s::Input data provided for %s must be of type double\n",
                      SUITEOPT_MATLAB_SOLVER, data_name) ;
            mexErrMsgTxt("Terminating program.\n") ;
        }
    }
    else /* data_name not present */
    {
        #ifdef DEBUG_SUITEOPT_MEX
        mexPrintf("%s::Input data for %s not provided.\n", 
                  SUITEOPT_MATLAB_SOLVER, data_name) ;
        #endif
    }
}

/* Sets int with data_name to output[0] if found in user struct */
void suiteopt_matlab_get_int
(
    const mxArray *options, /* User provided struct to mex function */
    SOPTINT *output,    /* Ptr to var to set equal to usr data (if given) */
    char *data_name         /* String for name of element to set pointer for */
)
{
    /* Initialize variables */
    mxArray *field ;

    if ((field = mxGetField(options, 0, data_name)) != NULL)
    {/* Copy value from field to output */
        #ifdef DEBUG_SUITEOPT_MEX
        mexPrintf("%s::User provided data for %s.\n", 
                  SUITEOPT_MATLAB_SOLVER, data_name) ;
        #endif

        if (mxIsClass(field, "double"))
        {
            /* Copy value for user data */
            output[0] = (SOPTINT) mxGetScalar (field) ;  

            #ifdef DEBUG_SUITEOPT_MEX
            mexPrintf("%s::Successfully stored value of %d for %s.\n", 
                      SUITEOPT_MATLAB_SOLVER, output[0], data_name) ;
            #endif
        }
        else /* data must be of type double */
        {
            /* Print error message */
            mexPrintf("%s::Input data provided for %s must be of type double.\n"
                      ,SUITEOPT_MATLAB_SOLVER, data_name) ;
            mexErrMsgTxt("Terminating program.\n") ;
        }
    }
    else /* Print error message */
    {
        #ifdef DEBUG_SUITEOPT_MEX
        mexPrintf("%s::Input data for %s not provided.\n", 
                  SUITEOPT_MATLAB_SOLVER, data_name) ;
        #endif
    }
}


/* Import problem dimension (n) */
void suiteopt_matlab_get_problem_dimension
(
    const mxArray *options, /* User provided struct to mex function */
    SOPTINT         *n  /* Pointer to problem dimension variable */
)
{
    #if defined(NAPHEAP_MATLAB) || defined(CGDESCENT_MATLAB)
    /* n: Problem dimension (if provided) */
    suiteopt_matlab_get_int (options, n, "n") ;
    #else
    /* ncol: Problem dimension (if provided) */
    suiteopt_matlab_get_int (options, n, "ncol") ;
    #endif

    /* Exit function */
    return ;
}


#ifdef SUITEOPT_MATLAB_BOUND_CONSTRAINTS
/* Sets data associated w/ bound constraints of the form lo <= x <= hi */
/* Imports lo and hi */
void suiteopt_matlab_get_bound_constraints
(
    const mxArray *options   /* User provided struct to mex function */
#ifdef NAPHEAP_MATLAB
    ,NAPdata *data           /* Struct in which to store pproj problem data */
#elif PPROJ_MATLAB
    ,PPdata *data            /* Struct in which to store pproj problem data */
#elif PASA_MATLAB
    ,PASAdata *data          /* Struct in which to store pasa problem data */
#endif
)
{
    /* Initialize variables */
    mxArray *field ;
    LONG n = data->SUITEOPT_MATLAB_N ;
    
    /* lo: Bound constraint lo <= x */
    data->lo = suiteopt_matlab_get_float (options, &n, "lo") ;
    /* Check if lo not provided */
    if (data->lo == NULL)
    {
        /* Set parameter loExists to FALSE */
        #ifdef PASA_MATLAB
        data->Parms->pasa->loExists = FALSE ;
        #else
        data->Parm->loExists = FALSE ;
        #endif
    }

    /* hi: Bound constraint x <= hi */
    data->hi = suiteopt_matlab_get_float (options, &n, "hi") ;
    /* Check if hi not provided */
    if (data->hi == NULL)
    {
        /* Set parameter hiExists to FALSE */
        #ifdef PASA_MATLAB
        data->Parms->pasa->hiExists = FALSE ;
        #else
        data->Parm->hiExists = FALSE ;
        #endif
    }
    /* if the value for n was gotten from the size of lo or hi, then save it */
    data->SUITEOPT_MATLAB_N = n ;
}
#endif


#ifdef SUITEOPT_MATLAB_POLYHEDRAL_CONSTRAINTS
/* Sets data associated w/ polyhedral constraints of the form bl <= Ax <= bu */
/* Imports nrow, Ap, Ai, Ax, bl, and bu */
/* If PPROJ: Also imports nsing, row_sing singlo, singhi, singc */
void suiteopt_matlab_get_polyhedral_constraints
(
    const mxArray *options  /* User provided struct to mex function */
#ifdef PPROJ_MATLAB
    ,PPdata * data          /* Struct in which to store pproj problem data */
#elif PASA_MATLAB
    ,PASAdata * data        /* Struct in which to store pasa problem data */
#endif
)
{
    /* Initialize variables */
    mxArray *field ;
    int status ;
    LONG Ancol, Anrow, ncol, nrow ;


    /* save the user input for problem dimension and number of rows in A */
    ncol = data->ncol ;
    nrow = data->nrow ;
    /* Verify that user provided polyhedral constraint matrix A */
    if ((field = mxGetField(options, 0, "A")) != NULL)
    {
        if (mxIsSparse(field)) 
        {
            #if DEBUG_SUITEOPT_MEX
            mexPrintf("%s::About to load sparse matrix A\n",
                SUITEOPT_MATLAB_SOLVER) ;
            #endif

            /* Number of rows in A */
            Anrow = mxGetM (field) ;
            /* Number of cols in A */
            Ancol = mxGetN (field) ;
            /* Column pointers for A */
            data->Ap = mxGetJc (field) ; 
            /* Row indices for A */
            data->Ai = mxGetIr (field) ; 
            /* Matrix values for A */
            data->Ax = mxGetPr (field) ; 
            
            #if DEBUG_SUITEOPT_MEX
            mexPrintf("%s::Loaded sparse A successfully\n",
                       SUITEOPT_MATLAB_SOLVER) ;
            #endif
        }
        else if (mxGetNumberOfDimensions(field) == 2)
        {
            #if DEBUG_SUITEOPT_MEX
            mexPrintf("%s::About to load full matrix A\n",
                       SUITEOPT_MATLAB_SOLVER) ;
            #endif

            /* Number of rows in A */
            Anrow = mxGetM (field) ;
            /* Number of cols in A */
            Ancol = mxGetN (field) ;
            /* Matrix elements */
            data->A_by_cols = mxGetPr(field) ;

            /* Convert dense matrix, stored by columns, to sparse format
               for SuiteOPT */
            status = sopt_convert_dense_to_sparse (&data->Ap, &data->Ai,
                    &data->Ax, data->A_by_cols, Anrow, Ancol, FALSE) ;
            /* Check if error occurred when converting constraint matrix */
            if ( status == SOPT_OUT_OF_MEMORY )
            {
                mexPrintf("%s::Out of memory when converting dense "
                          "constraint matrix to sparse matrix.\n",
                           SUITEOPT_MATLAB_SOLVER) ;
                mexErrMsgTxt("Terminating execution.\n") ;
            }
            
            #if DEBUG_SUITEOPT_MEX
            mexPrintf("%s::Loaded full A successfully\n",
                                                       SUITEOPT_MATLAB_SOLVER) ;
            #endif
        }
        else
        {
            /* Print error message */
            mexPrintf("%s::Matrix for polyhedral constraint set provided must "
                      "be in sparse or full format.\n", SUITEOPT_MATLAB_SOLVER);
            mexErrMsgTxt("Terminating execution.\n") ;
        }

        /* if ncol <= 0, then ncol has not yet been assigned by the user */
        if ( ncol <= 0 )
        {
            ncol = Ancol ;
        }
        /* if the user provided ncol, and ncol > Ancol, then expand
           the sparse matrix by adding columns of zeros */
        else if ( ncol > Ancol )
        {
            LONG j, *ap ;
            ap = mxMalloc ((ncol+1)*sizeof (LONG)) ;
            LONG const l = data->ncol ;
            for (j = 0; j <= l; j++)
            {
                ap [j] = data->Ap [j] ;
            }
            mxFree (data->Ap) ;
            data->Ap = ap ;
            LONG const k = ap [l] ;
            for (; j <= ncol; j++)
            {
                ap [j] = k ;
            }
        }
        else if ( ncol < Ancol ) /* print error message */
        {
            mexPrintf("%s::Matrix for polyhedral constraint set provided but\n"
                      "the number of columns %ld in the matrix was greater\n"
                      "than the problem dimension %ld specified by the user.\n",
                      SUITEOPT_MATLAB_SOLVER, Ancol, ncol) ;
            mexErrMsgTxt("Terminating execution.\n") ;
        }

        /* if nrow was not assigned by the user, or if the value of nrow >=
           the number of rows in the matrix, then set nrow = the number of
           rows in A */
        if ( (nrow <= 0) || (nrow > Anrow) )
        {
            nrow = Anrow ;
        }
        else if ( nrow < Anrow )
        {
            mexPrintf("%s::Matrix for polyhedral constraint set provided but\n"
                      "the number of rows %ld in the matrix was greater\n"
                      "than the number of rows %ld specified by the user.\n",
                      SUITEOPT_MATLAB_SOLVER, Anrow, nrow) ;
            mexErrMsgTxt("Terminating execution.\n") ;
        }
        /* Since A provided, user must provide at least one of bl and bu. */
        /* bl: Polyhedral lower bound bl <= Ax */
        LONG Nrow = nrow ;
        data->bl = suiteopt_matlab_get_float (options, &Nrow, "bl");
        /* bu: Polyhedral upper bound Ax <= bu */
        data->bu = suiteopt_matlab_get_float (options, &Nrow, "bu");
        /* Check if user provided at least one of bl and bu */
        if ((data->bl == NULL) && (data->bu == NULL))
        {
            /* Print error message and terminate */
            mexPrintf("%s::Matrix for polyhedral constraint set provided but\n"
                      "both bl and bu missing. A linear constraint must have\n"
                      "either a right side or a left side\n",
                       SUITEOPT_MATLAB_SOLVER) ;
            mexErrMsgTxt("Terminating execution.\n") ;
        }

        /* Print error message if length of right and left side does not
           match the number of rows in the matrix */
        if ( nrow != Nrow )
        {
            mexPrintf("%s::Matrix for polyhedral constraint has %ld rows\n"
                      "but bl or bu has dimension %ld\n",
                       SUITEOPT_MATLAB_SOLVER, nrow, Nrow);
            mexErrMsgTxt("Terminating execution.\n") ;
        }

        /* --- Additional polyhedral data associated exclusively with PPROJ */
        #ifdef PPROJ_MATLAB
            /* Since A provided, user may provide nsing. Check and import */
            if ((field = mxGetField(options, 0, "nsing")) != NULL)
            {/* Copy pointer from field to nsing */
                #if DEBUG_SUITEOPT_MEX
                mexPrintf("pproj::Value provided for nsing.\n") ;
                #endif
    
                data->nsing = (long) mxGetScalar (field) ;  
    
                #if DEBUG_SUITEOPT_MEX
                mexPrintf("pproj::Successfully imported value for nsing.\n") ;
                #endif
            }
    
            /* Since A provided, user may provide row_sing. Check and import */
            if ((field = mxGetField(options, 0, "row_sing")) != NULL)
            {/* Copy pointer from field to row_sing */
                #if DEBUG_SUITEOPT_MEX
                mexPrintf("pproj::User provided row_sing.\n") ;
                #endif
    
                data->row_sing = mxGetIr (field) ;  
    
                #if DEBUG_SUITEOPT_MEX
                mexPrintf("pproj::Successfully imported row_sing.\n") ;
                #endif
            }
    
            /* Since A provided, user may provide singlo. Check and import */
            if ((field = mxGetField(options, 0, "singlo")) != NULL)
            {/* Copy pointer from field to singlo */
                #if DEBUG_SUITEOPT_MEX
                mexPrintf("pproj::User provided singlo.\n") ;
                #endif
    
                data->singlo = mxGetPr (field) ;  
    
                #if DEBUG_SUITEOPT_MEX
                mexPrintf("pproj::Successfully imported singlo.\n") ;
                #endif
            }
    
            /* Since A provided, user may provide singhi. Check and import */
            if ((field = mxGetField(options, 0, "singhi")) != NULL)
            {/* Copy pointer from field to singhi */
                #if DEBUG_SUITEOPT_MEX
                mexPrintf("pproj::User provided singhi.\n") ;
                #endif
    
                data->singhi = mxGetPr (field) ;  
    
                #if DEBUG_SUITEOPT_MEX
                mexPrintf("pproj::Successfully imported singhi.\n") ;
                #endif
            }
    
            /* Since A provided, user may provide singc. Check and import */
            if ((field = mxGetField(options, 0, "singc")) != NULL)
            {/* Copy pointer from field to singc */
                #if DEBUG_SUITEOPT_MEX
                mexPrintf("pproj::User provided singc.\n") ;
                #endif
    
                data->singc = mxGetPr (field) ;  
    
                #if DEBUG_SUITEOPT_MEX
                mexPrintf("pproj::Successfully imported singc.\n") ;
                #endif
            }
        #endif
    }
    #ifdef PASA_MATLAB
    else if ((field = mxGetField(options, 0, "a")) != NULL)
    {   /* User provided napheap vector a instead of matrix */
        /* Setting nrow = 1 */
        nrow = 1 ;
        /* a: Linear constraint vector */
        Ancol = -1 ;
        data->a = suiteopt_matlab_get_float (options, &Ancol, "a");

        if ( ncol <= 0 )
        {
            ncol = Ancol ;
        }
        else if ( ncol < Ancol )
        {
            mexPrintf("%s::The row vector in the knapsack constraint has\n"
                      "%ld components but the problem dimension is %ld\n",
                       SUITEOPT_MATLAB_SOLVER, Ancol, ncol);
            mexErrMsgTxt("Terminating execution.\n") ;
        }
        else if ( ncol > Ancol ) /* pad "a" vector with zeros */
        {
            LONG j ;
            SOPTFLOAT *ax ;
            ax = mxMalloc (ncol*sizeof (SOPTFLOAT)) ;
            for (j = 0; j < Ancol; j++)
            {
                ax [j] = data->a [j] ;
            }
            mxFree (data->a) ;
            data->a = ax ;
            for (; j < ncol; j++)
            {
                ax [j] = SOPTZERO ;
            }
        }

        data->bl = suiteopt_matlab_get_float (options, &nrow, "bl");
        /* bu: Polyhedral upper bound Ax <= bu */
        data->bu = suiteopt_matlab_get_float (options, &nrow, "bu");

        /* If user did not provide bl and bu, then terminate execution. */ 
        if ((data->bl == NULL) && (data->bu == NULL))
        {
            /* Print error message */
            mexPrintf("%s::Vector a for napsack constraints provided\n"
                      "but both bl and bu missing. A linear constraint\n"
                      "must have either a right side or a left side.\n",
                       SUITEOPT_MATLAB_SOLVER) ;
            mexErrMsgTxt("Terminating execution.\n") ;
        }
    }
    #endif
    else /* Neither A nor a provided. Set Ap, Ai, Ax, a, bl, and bu to NULL */
    {
        /* Setting nrow = 0 */
        nrow = 0 ;

        /* Print warning message */
        #if DEBUG_SUITEOPT_MEX
            #ifdef PPROJ_MATLAB
            mexPrintf("%s::Matrix A not provided. Ap, Ai, Ax, bl, bu, "
                      "set to NULL.\n", SUITEOPT_MATLAB_SOLVER) ;
            mexPrintf("pproj::row_sing, singlo, "
                      "singhi, and singc set to NULL.\n") ;
            mexPrintf("pproj::ni and nsing set to default values.\n") ;
            #elif PASA_MATLAB
            mexPrintf("%s::Matrix A and vector a not provided. Ap, Ai, Ax, a, "
                      "bl, bu, set to NULL.\n", SUITEOPT_MATLAB_SOLVER) ;
            mexPrintf("%s::Set nrow = 0.\n", SUITEOPT_MATLAB_SOLVER) ;
            #endif
        #endif
    }
    data->nrow = nrow ;
    data->ncol = ncol ;
}
#endif

/* solvers that apply to QPs */
#ifdef SUITEOPT_MATLAB_QP
/* Imports a sparse Hessian matrix stored into Hp, Hi, Hx, and ncol */
void suiteopt_matlab_get_Hessian_matrix
(
    const mxArray *options  /* User provided struct to mex function */
#ifdef PASA_MATLAB
    ,PASAdata * data        /* Struct in which to store pasa problem data */
#elif CGDESCENT_MATLAB
    ,CGdata * data          /* Struct in which to store cg problem data */
#endif
)
{
    /* Initialize variables */
    mxArray *field ;
    int status ;
    LONG j, k, Hncol, Hnrow, ncol ;

    ncol = data->SUITEOPT_MATLAB_N ;
    /* Verify that user provided sparse Hessian matrix H */
    if ((field = mxGetField(options, 0, "H")) != NULL)
    {
        if (mxIsSparse(field)) 
        {
            #if DEBUG_SUITEOPT_MEX
            mexPrintf("%s::About to load sparse Hessian matrix H\n",
                SUITEOPT_MATLAB_SOLVER) ;
            #endif

            /* Number of rows in H */
            Hnrow = mxGetM (field) ;
            /* Number of cols in H */
            Hncol = mxGetN (field) ;
            /* Column pointers for H */
            data->Hp = mxGetJc (field) ; 
            /* Row indices for H */
            data->Hi = mxGetIr (field) ; 
            /* Matrix values for H */
            data->Hx = mxGetPr (field) ; 
            
            #if DEBUG_SUITEOPT_MEX
            mexPrintf("%s::Loaded sparse H successfully\n",
                       SUITEOPT_MATLAB_SOLVER) ;
            #endif
        }
        else if (mxGetNumberOfDimensions(field) == 2)
        {
            #if DEBUG_SUITEOPT_MEX
            mexPrintf("%s::About to load full matrix H\n",
                       SUITEOPT_MATLAB_SOLVER) ;
            #endif

            /* Number of rows in H */
            Hnrow = mxGetM (field) ;

            /* Number of cols in H */
            Hncol = mxGetN (field) ;

            if ( (Hnrow == Hncol) && ((Hncol <= ncol) || (ncol <= 0)) )
            {
                /* Convert dense matrix to sparse format for SuiteOPT */
                status = sopt_convert_dense_to_sparse (&data->Hp, &data->Hi,
                               &data->Hx, mxGetPr(field), Hncol, Hncol, FALSE) ;
                if ( status == SOPT_OUT_OF_MEMORY )
                {
                    mexPrintf("%s::Out of memory when converting dense "
                              "Hessian to sparse matrix.\n",
                               SUITEOPT_MATLAB_SOLVER) ;
                    mexErrMsgTxt("Terminating program.\n") ;
                }
            }
        }
        else
        {
            /* Print error message */
            mexPrintf("%s::User's Hessian matrix must "
                      "be in sparse or full format.\n", SUITEOPT_MATLAB_SOLVER);
            mexErrMsgTxt("Terminating program.\n") ;
        }

        if ( Hnrow != Hncol )
        {
            mexPrintf ("%s::The user's Hessian matrix is not symmetric since\n"
                       "the number of rows %ld is not equal to the number\n"
                       "of column %ld\n",
                       SUITEOPT_MATLAB_SOLVER, Hnrow, Hncol) ;
            mexErrMsgTxt("Terminating program.\n") ;
        }

        /* if the problem dimension has not yet been assigned, then use Hncol */
        if ( ncol <= 0 )
        {
            data->SUITEOPT_MATLAB_N = ncol = Hncol ;
        }
        else if ( Hncol > ncol )
        {
            mexPrintf ("The user's Hessian matrix has more columns %ld than\n"
                       "the problem dimension %ld\n", Hncol, ncol);
            mexErrMsgTxt("Terminating program.\n") ;
        }

        /* if the problem dimension is larger than that of the Hessian matrix,
           then expand the Hessian with columns of zeros */
        else if ( ncol > Hncol )
        {
            SOPTINT *hp ;
            hp = mxMalloc ((ncol+1)*sizeof (SOPTINT)) ;
            for (j = 0; j <= Hncol; j++)
            {
                hp [j] = data->Hp [j] ;
            }
            mxFree (data->Hp) ;
            data->Hp = hp ;
            k = hp [Hncol] ;
            for (; j <= ncol; j++)
            {
                hp [j] = k ;
            }
        }
            
        #if DEBUG_SUITEOPT_MEX
        mexPrintf("%s::Loaded full H successfully\n", SUITEOPT_MATLAB_SOLVER) ;
        #endif
    }
}
#endif

/* Returns pointer to function handle with data_name if found in user struct,
   Otherwise returns NULL pointer */
mxArray* suiteopt_matlab_get_func
(
    const mxArray *options, /* User provided struct to mex function */
    int inputs,             /* Correct number of input args for function */
    int outputs,            /* Correct number of output args for function */
    char *data_name         /* String for name of element to set pointer for */
)
{
    /* --- Initialize variables --- */
    int nargsin, nargsout ;
    mxArray *lhs[1], *rhs, *field ;

    /* --- Function handle for objective --- */
    if ((field = mxGetField(options, 0, data_name)) != NULL)
    {
        if (mxIsClass(field, "function_handle"))
        {
            /* ----- Check number of input and output arguments ----- */
            /* Copy prhs to non const mxArray */
            rhs = (mxArray*) field ;
            /* Call MATLAB function nargin and store output in lhs[0] */
            mexCallMATLAB(1, lhs, 1, &rhs, "nargin") ;
            /* Cast lhs[0] to integer nargsin */
            nargsin = (int) mxGetScalar (lhs[0]) ;
            /* Call MATLAB function nargout and store output in lhs[0] */
            mexCallMATLAB(1, lhs, 1, &rhs, "nargout") ;
            /* Cast lhs[0] to integer nargsout */
            nargsout = (int) mxGetScalar (lhs[0]) ;

            #ifdef DEBUG_SUITEOPT_MEX
            mexPrintf("%s::Number of input arguments for %s: %i\n", 
                      SUITEOPT_MATLAB_SOLVER, data_name, nargsin) ;
            mexPrintf("%s::Number of output arguments for %s: %i\n", 
                      SUITEOPT_MATLAB_SOLVER, data_name, nargsout) ;
            #endif

            if ((nargsin == inputs) && 
                ((nargsout == outputs) || (nargsout == -outputs))) 
            {
                #ifdef DEBUG_SUITEOPT_MEX
                mexPrintf("%s::Successfully imported function handle '%s'.\n",
                          SUITEOPT_MATLAB_SOLVER, data_name) ;
                #endif

                /* Return handle for function evaluation */
                return (mxArray*) field ;
            }
            else {/* Incorrect number of input and output args */
                mexPrintf("%s::Function handle for %s provided to pasa "
                          "has incorrect number of arguments.\n", 
                          SUITEOPT_MATLAB_SOLVER, data_name) ;
                mexPrintf("%s::Function handle for %s must have %d input "
                          "argument(s) and %d output argument(s).\n",
                          SUITEOPT_MATLAB_SOLVER, data_name, inputs, outputs) ;
                mexErrMsgTxt("Terminating program.") ;
            }
        }
        else {/* objective must be a function handle */
            /* Print error message */
            mexPrintf("%s::Function handle for %s must have %d input "
                      "argument(s) and %d output argument(s).\n",
                      SUITEOPT_MATLAB_SOLVER, data_name, inputs, outputs) ;
            mexErrMsgTxt("Terminating cg_descent.") ;
        }
    }
    else {/* objective not provided to pasa */
        #ifdef DEBUG_SUITEOPT_MEX
        mexPrintf("%s::Function handle for %s not provided. Returning NULL "
                  "pointer.\n", SUITEOPT_MATLAB_SOLVER, data_name) ;
        #endif
    }

    /* Function handle not found. Returning NULL */
    return NULL ;
}

/* --- Utility functions used in SuiteOPT MEX functions --- */
/* Copy vector x into vector y */
void suiteopt_matlab_copy_arr
(
    SOPTFLOAT *y, /* output of copy */
    SOPTFLOAT *x, /* input of copy */
    int            n  /* length of vectors */
)
{
    int i, n5 ;
    n5 = n % 5 ;
    for (i = 0; i < n5; i++) y [i] = x [i] ;
    for (; i < n; )
    {
        y [i] = x [i] ;
        i++ ;
        y [i] = x [i] ;
        i++ ;
        y [i] = x [i] ;
        i++ ;
        y [i] = x [i] ;
        i++ ;
        y [i] = x [i] ;
        i++ ;
    }
    return ;
}
