/* ====================================================
 * CUTEst interface for pasa             August 1, 2016
 *
 * W. Hager
 *
 * (Based on CUTEr gencma.c of D. Orban, Feb 3, 2003)
 * (CUTEst evolution, Nick Gould, Apr 2, 2014)
 * ====================================================
 */

/* macros */
#define PASAMA

#define MAXLINE 256

#ifdef __cplusplus
extern "C" {   /* To prevent C++ compilers from mangling symbols */
#endif

#include "cutest.h"
#include "pasa.h"

#ifdef Isg95
#define MAINENTRY MAIN_
#else
#define MAINENTRY main
#endif

/* evaluation routine prototypes for unconstrained problems */

void cg_value
(
    PASAFLOAT *f,
    PASAFLOAT *x,
    PASAINT    n
) ;

void cg_grad
(
    PASAFLOAT  *g,
    PASAFLOAT  *x,
    PASAINT     n
) ;

void cg_valgrad
(
    PASAFLOAT  *f,
    PASAFLOAT  *g,
    PASAFLOAT  *x,
    PASAINT     n
) ;

/* evaluation routine prototypes for constrained problems */

void pasa_value
(
    PASAFLOAT *f,
    PASAFLOAT *x,
    PASAINT    n
) ;

void pasa_grad
(
    PASAFLOAT  *g,
    PASAFLOAT  *x,
    PASAINT     n
) ;

void pasa_valgrad
(
    PASAFLOAT  *f,
    PASAFLOAT  *g,
    PASAFLOAT  *x,
    PASAINT     n
) ;

/* global variables */
    logical cute_true = TRUE_ ;
    logical cute_false = FALSE_ ;

/* main program */
int MAINENTRY( void )
{

    /* wall clock: */
    double walltime ;

    char *fname = "OUTSDIF.d"; /* CUTEst data file */
    integer funit = 42;        /* FORTRAN unit number for OUTSDIF.d */
    integer io_buffer = 11;    /* FORTRAN unit for internal i/o */
    integer iout = 6;          /* FORTRAN unit number for error output */
    integer ierr;              /* Exit flag from OPEN and CLOSE */
    integer status;            /* Exit flag from CUTEst tools */
    char fullpathname [1024] ;
    char *cutest_dir ;

    VarTypes vtypes;

    integer     e_order, l_order, v_order, nnzj, nnzh, anz, lj1, lj2, H_anz,
                nrow, ncol,
                *cutest_J_row, *cutest_J_col, *cutest_H_row, *cutest_H_col ;
    integer     iprob = 0 ;
    PASAFLOAT   *x, *xcopy, *lo, *hi, *lambda, *offset, *c, *bl, *bu, *Ax, *ATx,
                *J_val, *zeros ;
    char        pbname [MAXLINE+1], *uname, *name ;
    logical     *equatn, *linear ;

    doublereal  calls[7], cpu[2] ;
    integer     ExitCode ;
    int         loExists, hiExists, LP, QP, UB, status_pasa ;
    PASAFLOAT   s, t, u, *HTx, *Hx, *H_val ;
    PASAINT     Anz, Hnz, Nrow, Ncol, row, col, free_cols, free_rows,
                i, j, k, l, p, nchols,
                *J_row, *J_col, *H_row, *H_col,
                *Ap, *Ai, *ATp, *ATi, *Hp, *Hi,
                *drop_col, *drop_row, *ifree,
                *HTi, *HTp ;

    integer CUTEst_nvar;        /* number of variables */
    integer CUTEst_ncon;        /* number of constraints */
    integer CUTEst_nnzh ;       /* number of nonzeros in Hessian */

    FILE *spec, *file ;
    PASAstat  *pasastat ;  /* statistics for PASA */
    CGstat      *cgstat ;  /* statistics for CG */
    PPstat   *pprojstat ;  /* statistics for PPROJ */
    NAPstat    *napstat ;  /* statistics for NAPHEAP */
    PASAparm  *pasaparm ;  /* parameters for PASA */
    CGparm      *cgparm ;  /* parameters for CG */
    NAPparm    *napparm ;  /* parameters for NAPHEAP */
    PPparm   *pprojparm ;  /* parameters for PPROJ */
    PASAdata  *pasadata ;  /* data structure for pasa input */

    /* --- Initialize problem types ------ */
    LP = FALSE ;
    QP = FALSE ;
    UB = FALSE ;

    /* for an LP or QP, fadjust stores constant term in objective */
    PASAFLOAT fadjust = PASAZERO ;

    /* initialize arrays for sparse Hessian of QP,
       for linear cost term, and for Jacobia of constraints */
    c     = NULL ;
    Ap    = NULL ;
    Ai    = NULL ;
    Ax    = NULL ;
    Hp    = NULL ;
    Hi    = NULL ;
    Hx    = NULL ;
    lo    = NULL ;
    hi    = NULL ;
    bl    = NULL ;
    bu    = NULL ;
    H_val = NULL ;
    H_row = NULL ;
    H_col = NULL ;
    Nrow  = 0 ;

    /* create the default pasa data structure */
    printf ("Initializing PASAdata structure.\n") ;
    pasadata = pasa_setup () ;
    if ( pasadata == NULL )
    {
            pasa_error (-1, __FILE__, __LINE__, "pasa_setup failed") ;
    }
    else
    {
        printf ("Successfully initialized PASAdata structure.\n") ;
    }

    /* Open problem description file OUTSDIF.d */
    ierr = 0;
    FORTRAN_open( &funit, fname, &ierr ) ;
    if( ierr )
    {
        printf("Error opening file OUTSDIF.d.\nAborting.\n") ;
        exit(1) ;
    }

    /* Get the name of the user's problem */
    uname = (char *) malloc ((FSTRING_LEN+1)*sizeof (char)) ;
    CUTEST_pname( &status, &funit, uname ) ;
    if (status)
    {
        printf("** CUTEst error, status = %ld, aborting\n", (LONG) status) ;
        exit(status) ;
    }

    /* Make sure to null-terminate problem name */
    uname[FSTRING_LEN] = '\0';
    i = FSTRING_LEN - 1;
    while(i-- > 0 && uname[i] == ' ')
    {
        uname[i] = '\0';
    }

    /* Print problem name */
    printf ("\nProblem: %s\n", uname) ;

    /* read the full path to the pasa/CUTEst directory and store it in the
       variable cutest_dir */
#include ".cutest_location"

    /* CUTEst stores the unconstrained and bound constrained problems in a
       different format from the general constrained problems, and there are
       different tools for handling each format. If the problem name is
       in the file classUB, then the problem is unconstrained or bound
       constrained. */
    strcpy (fullpathname, cutest_dir) ;
    file = fopen (strcat (fullpathname, "/classUB"), "r") ;
    while (fgets (pbname, MAXLINE, file) != (char *) NULL)
    {
        for (name = pbname; *name; name++)
        {
            if (isspace (*name)) *name = '\0' ;
        }
        name = pbname ;

        /* check if the name of the problem in the classUB file matches the
           user's problem name */
        if ( strcmp (name, uname) == 0 )
        {
            /* Indicate problem type */
            UB = TRUE ;
            printf ("The problem is unconstrained or bound constrained\n") ;
            /* Exit while loop */
            break ;
        }
    }
    fclose(file) ;

    /* see if the problem has a linear objective by comparing the name
       pname of the test problem to the names lpprob of the LPs contained
       in the file classLP */
    strcpy (fullpathname, cutest_dir) ;
    file = fopen (strcat (fullpathname, "/classLP"), "r") ;
    while (fgets (pbname, MAXLINE, file) != (char *) NULL)
    {
        for (name = pbname; *name; name++)
        {
            if (isspace (*name)) *name = '\0' ;
        }
        name = pbname ;
        if ( strcmp (name, uname) == 0 )
        {
            pasadata->LP = LP = TRUE ;
            printf ("The problem is an LP\n") ;
            break ;
        }
    }
    fclose(file) ;

    /* see if the problem has a quadratic objective by comparing the name
       of the test problem to the names of the QPs contained
       in the file classQP */
    if ( !LP )
    {
        strcpy (fullpathname, cutest_dir) ;
        file = fopen (strcat (fullpathname, "/classQP"), "r") ;
        while (fgets (pbname, MAXLINE, file) != (char *) NULL)
        {
            for (name = pbname; *name; name++)
            {
                if (isspace (*name)) *name = '\0' ;
            }
            name = pbname ;
            if ( strcmp (name, uname) == 0 )
            {
                QP = TRUE ;
                printf ("The problem is a QP\n") ;
                break ;
            }
        }
        fclose(file) ;
    }

    /* Determine the problem dimension and number of constraints */
    if ( UB )
    {
        CUTEST_udimen( &status, &funit, &CUTEst_nvar) ;
    }
    else
    {
        CUTEST_cdimen( &status, &funit, &CUTEst_nvar, &CUTEst_ncon) ;
        nrow = CUTEst_ncon ; /* integer */
        Nrow = nrow ;        /* PASAINT */
    }

    ncol = CUTEst_nvar ; /* integer */
    Ncol = ncol ;        /* PASAINT */

    /* Allocate memory for variables and bounds */
    x  = (PASAFLOAT *) malloc (Ncol*sizeof (PASAFLOAT)) ;
    lo = (PASAFLOAT *) malloc (Ncol*sizeof (PASAFLOAT)) ;
    hi = (PASAFLOAT *) malloc (Ncol*sizeof (PASAFLOAT)) ;
    pasadata->x = x ;

    /* Constants in linear terms are obtained by evaluation at zero */
    zeros = (PASAFLOAT *) malloc (Ncol*sizeof (PASAFLOAT)) ;
    pasa_initx (zeros, PASAZERO, Ncol) ;

    if ( UB )
    {
        /* starting guess, bounds */
        CUTEST_usetup (&status, &funit, &iout, &io_buffer, &ncol, x, lo, hi) ;
    }
    else /* get the linear constraints */
    {
        /* e_order determines order of eqn and ineq in list of constraints */
        /* if e_order  = 1 then equations come before inequalities */
        /* if e_order  = 2 then inequalities come before equalities */
        /* if e_order != 1 or 2 then order does not matter */
        e_order = 1 ;

        /* l_order determines order of lin and nonlin constraints */
        /* if l_order  = 1 then lin come before nonlin */
        /* if l_order  = 2 then nonlin come before lin */
        /* if l_order != 1 or 2 then order does not matter */
        l_order = 1 ;

        /* v_order determines order of lin and nonlin variables */
        v_order = 2 ;

        /* Reserve memory for linear inequality bounds and CUTE logical
           variables  */
        if ( Nrow )
        {
            pasadata->bl = bl = (PASAFLOAT *) malloc (Nrow*sizeof (PASAFLOAT)) ;
            pasadata->bu = bu = (PASAFLOAT *) malloc (Nrow*sizeof (PASAFLOAT)) ;
            /* lambda = initial guess for the constraint multipliers */
            pasadata->lambda = lambda
                             = (PASAFLOAT *) malloc (Nrow*sizeof (PASAFLOAT)) ;
            MALLOC(equatn, nrow, logical) ;
            MALLOC(linear, nrow, logical) ;
            /* offset = constant terms in the constraint */
            offset = (PASAFLOAT *) malloc (Nrow*sizeof (PASAFLOAT)) ;
        }

        /* Read the problem data */
        CUTEST_csetup ( &status, &funit, &iout, &io_buffer, &ncol,
                        &nrow, x, lo, hi, lambda, bl, bu, equatn, 
                        linear, &e_order, &l_order, &v_order ) ;
        if ( Nrow )
        {
            free (equatn) ;
            free (linear) ;

            /* CUTE uses 1e20/1e21 for infinite bounds, convert to PASAINF */
            for (i = 0; i < nrow; i++)
            {
                if ( (bl [i] == -1e20) || (bl [i] == -1e21) ) bl [i] = -PASAINF;
            }

            for (i = 0; i < nrow; i++)
            {
                if ( (bu [i] ==  1e20) || (bl [i] ==  1e21) ) bu [i] =  PASAINF;
            }

            /* Adjust bl and bu for any constants in the constraints.
               We obtain the constants by evaluating constraints at x = 0. */
            CUTEST_ccfg ( &status, &ncol, &nrow, zeros, offset, 
                          &cute_false, &lj1, &lj2, J_val, &cute_false ) ;

            for (i = 0; i < nrow; i++)
            {
                bl [i] -= offset [i] ;
                bu [i] -= offset [i] ;
            }

            /* nnzj is the number of nonzeros in the objective function gradient
               plus the number of nonzeros in the constraint gradients */
            CUTEST_cdimsj ( &status, &nnzj ) ;

            /* Create arrays for Jacobian triple: J_val, J_col, J_row */
            J_val = (PASAFLOAT *) malloc (nnzj*sizeof (PASAFLOAT)) ;
            cutest_J_col = (integer *) malloc (nnzj*sizeof (integer)) ;
            cutest_J_row = (integer *) malloc (nnzj*sizeof (integer)) ;

            /* Evaluate constraint Jacobian at zero */
            CUTEST_csgr ( &status, &ncol, &nrow, zeros, lambda, &cute_false,
                      &anz, &nnzj, J_val, cutest_J_col, cutest_J_row ) ;
            /* nonzeros in the constraint Jacobian = nnzj - ncol since the
               gradient of the objective is also stored in J_val */
            Anz = nnzj - ncol ;

            if (status)
            {
                printf("** CUTEst error, status = %ld, aborting\n",
                        (LONG) status) ;
                exit(status) ;
            }

            /* allocate for the pasa J_col and J_row */
            J_col = (PASAINT *) malloc (Anz*sizeof (PASAINT)) ;
            J_row = (PASAINT *) malloc (Anz*sizeof (PASAINT)) ;

            /* Delete the row number zero from the triples since that entry
               corresponds to the objective gradient; also, adjust row and
               column numbers by 1 to correspond to indexing in the
               C programming language. */
            k = 0 ;
            for (i = 0; i < nnzj; i++)
            {
                if ( cutest_J_row [i] && J_val [i] )
                {
                    J_row [k] = cutest_J_row [i] - 1 ;
                    J_col [k] = cutest_J_col [i] - 1 ;
                    J_val [k] = J_val [i] ;
                    k++ ;
                }
            }
            free (cutest_J_row) ;
            free (cutest_J_col) ;
            free (offset) ;
            if ( Anz != k )
            {
                printf ("There seems to be a problem with the Jacobian of the\n"
                        "constraints since the number of nonzeros %ld implied\n"
                        "by the routine CUTEST_cdimsj did not match the\n"
                        "number of nonzeros %ld when evaluating the\n"
                        "constraint Jacobian using CUTEST_csgr\n",
                        (LONG) Anz, (LONG) k) ;
                pasa_error (-1, __FILE__, __LINE__, "pasa_setup failed") ;
            }
            printf ("nonzeros in A: %ld\n", (LONG) Anz) ;
            printf ("largest nonzero in A: %e\n", pasa_sup_normx (J_val, Anz)) ;

            status = sopt_convert_triple_to_sparse (&Ap, &Ai, &Ax, &Nrow, &Ncol,
                                         J_row, J_col, J_val, Anz, TRUE, FALSE);
            if ( status == SOPT_OUT_OF_MEMORY )
            {
                printf ("In pasa_main.c, there was not enough memory to\n"
                        "convert the CUTEst constraint matrix to the sparse\n"
                        "matrix format used by PASA.\n") ;
                pasa_error (-1, __FILE__, __LINE__, "execution halted") ;
            }
            else if ( status == SOPT_ERROR_IN_INPUT_MATRIX )
            {
                printf ("In pasa_main.c, there appears to be an error in the\n"
                        "triples describing the CUTEst linear constraint\n"
                        "matrix.\n") ;
                pasa_error (-1, __FILE__, __LINE__, "execution halted") ;
            }
            /* if a vanishing sparse matrix element was discovered, it was
               ignored when building the sparse matrix for pasa */
            pasadata->Ap = Ap ;
            pasadata->Ai = Ai ;
            pasadata->Ax = Ax ;
            free (J_row) ;
            free (J_col) ;
            free (J_val) ;
        }
    }
    if (status)
    {
        printf("** CUTEst error, status = %ld, aborting\n", (LONG) status);
        exit(status);
    }

    /* CUTE uses 1e20 or 1e21 to denote infinite bounds, convert to PASAINF */
    loExists = FALSE ;
    for (i = 0; i < ncol; i++)
    {
        if ( (lo [i] == -1e+20) || (lo [i] == -1e+21) ) lo [i]   = -PASAINF ;
        else                                            loExists = TRUE ;
    }
    if ( loExists ) pasadata->lo = lo ;

    /* Convert infinite bounds for hi to PASAINF */
    hiExists = FALSE ;
    for (i = 0; i < ncol; i++)
    {
        if ( (hi [i] == 1e+20) || (hi [i] == 1e+21) ) hi [i] = PASAINF ;
        else                                        hiExists = TRUE ;
    }
    if ( hiExists ) pasadata->hi = hi ;

    /* for LPs or QPs, there is a linear term in the objective */
    if ( LP || QP )
    {
        /* If the problem has a linear objective, then extract the
           linear term in the objective. */
        pasadata->c = c = (PASAFLOAT *) malloc (Ncol*sizeof (PASAFLOAT)) ;

        /* Evaluate nnz of Hessian and any constant term in the objective */
        if ( UB )
        {
            /* number of nonzeros in Hessian */
            CUTEST_udimsh ( &status, &nnzh );
            /* fadjust = constant term in cost */
            CUTEST_uofg (&status, &ncol, zeros, &fadjust, c, &cute_true);
        }
        else
        {
            /* number of nonzeros in the Hessian of the objective */
            CUTEST_cdimsh ( &status, &nnzh );
            /* fadjust = constant term in cost */
            CUTEST_cofg (&status, &ncol, zeros, &fadjust, c, &cute_true);
        }
        if ( LP )
        {
            printf ("cost adjustment in LP: %e "
                    "(ignored to obtain published netlib cost)\n", fadjust) ;
            fadjust = PASAZERO ;
        }
    }

    /* Extract the Hessian of a QP */
    if ( QP )
    {
        /* Memory for H_val, cutest_H_row, cutest_H_col */
        H_val = (PASAFLOAT *) malloc (nnzh*sizeof (PASAFLOAT)) ;
        cutest_H_row = (integer *) malloc (nnzh*sizeof (integer)) ;
        cutest_H_col = (integer *) malloc (nnzh*sizeof (integer)) ;

        /* Get the nonzero values in the Hessian */
        if ( UB )
        {
            CUTEST_ush (&status, &ncol, zeros, &H_anz, &nnzh, 
                         H_val, cutest_H_row, cutest_H_col) ;
        }
        else
        {
            CUTEST_cish (&status, &ncol, zeros, &iprob, &anz, &nnzh,
                          H_val, cutest_H_row, cutest_H_col) ;
        }
        /* convert CUTE integer to PASAINT */
        Hnz = nnzh ;
 
        if (status)
        {
            printf("** CUTEst error, status = %ld, aborting\n", 
                   (LONG) status) ;
            exit(status) ;
        }

        /* subtract 1 from the row and column to convert to indexing
           in the C programming languate and store in PASAINT arrays */
        H_row = (PASAINT *) malloc (Hnz*sizeof (PASAINT)) ;
        H_col = (PASAINT *) malloc (Hnz*sizeof (PASAINT)) ;
        for (i = 0; i < Hnz; i++)
        {
            H_col [i] = cutest_H_col [i] - 1 ;
            H_row [i] = cutest_H_row [i] - 1 ;
        }
        free (cutest_H_col) ;
        free (cutest_H_row) ;

        /* convert triple format to sparse matrix format */
        status = sopt_convert_triple_to_sparse (&Hp, &Hi, &Hx, &Ncol, &Ncol,
                                         H_row, H_col, H_val, Hnz, TRUE, TRUE) ;

        if ( status == SOPT_OUT_OF_MEMORY )
        {
            printf ("In pasa_main.c, there was not enough memory to\n"
                    "convert the CUTEst Hessian of a quadratic to the sparse\n"
                    "matrix format used by PASA.\n") ;
            pasa_error (-1, __FILE__, __LINE__, "execution halted") ;
        }
        else if ( status == SOPT_ERROR_IN_INPUT_MATRIX )
        {
            printf ("In pasa_main.c, there appears to be an error in the\n"
                    "triples describing the CUTEst Hessian of a quadratic.\n") ;
            pasa_error (-1, __FILE__, __LINE__, "execution halted") ;
        }
        /* if a vanishing sparse matrix element was discovered, it was
           ignored when building the sparse matrix for pasa */
        free (H_row) ;
        free (H_col) ;
        free (H_val) ;

        /* if the problem has fixed variables, then remove them from problem */
        if ( loExists || hiExists )
        {
            drop_col = (int *) malloc (Ncol*sizeof (int)) ;
            /* drop_col [i] = 1 if i-th column is dropped
                            = 0 if i-th column is kept */
            for (j = 0; j < ncol; j++)
            {
                if ( lo [j] == hi [j] )
                {
                    drop_col [j] = 1 ;
                    x [j] = lo [j] ;
                }
                else
                {
                    drop_col [j] = 0 ;
                }
            }

            /* find any fixed variables connected with the linear constraints */
            PASAINT *tempW ;
            tempW = (PASAINT *) malloc (PASAMAX(Nrow, Ncol)*sizeof (PASAINT)) ;
            if ( Nrow )
            {
                /* compute transpose of A */
                ATp = (PASAINT *) malloc ((Nrow+1)*sizeof (PASAINT)) ;
                ATi = (PASAINT *) malloc (Anz*sizeof (PASAINT)) ;
                ATx = (PASAFLOAT *) malloc (Anz*sizeof (PASAFLOAT)) ;
                sopt_transpose (ATp, ATi, ATx, Ap, Ai, Ax, Nrow, Ncol, tempW) ;

                drop_row = (int *) malloc (Nrow*sizeof (int)) ;
                /* drop_row [i] = 1 if i-th row is dropped
                                = 0 if i-th row is kept */
                l = 0 ;
                for (i = 0; i < nrow; i++)
                {
                    drop_row [i] = 0 ;
                    /* find equality constraints with one variable */
                    if ( (ATp [i+1] - ATp [i] == 1) && (bl [i] == bu [i]) )
                    {
                        k = ATp [i] ;
                        j = ATi [k] ;
                        x [j] = bl [i]/ATx [k] ; /* set x to bound */
                        drop_col [j] = 1 ; /* column   can be removed */
                        drop_row [i] = 1 ; /* equation can be removed */
                        l++ ;
                    }
                }
            }
            xcopy = (PASAFLOAT *) malloc (Ncol*sizeof (PASAFLOAT)) ;
            pasa_copyx (xcopy, x, Ncol) ;

            /* cost adjustment due to fixed variables is 0.5*xb'*H*xb + c'*xb
               c adjustment is xb'*H_{bf} */
            s = PASAZERO ;
            for (j = 0; j < ncol; j++)
            {
                PASAINT const q = Hp [j+1] ;
                if ( drop_col [j] == 1 )
                {
                    t = x [j] ;
                    fadjust += t*c [j] ;
                    for (p = Hp [j]; p < q; p++)
                    {
                        if ( drop_col [Hi [p]] == 1 )
                        {
                            s += t*Hx [p]*x [Hi [p]] ;
                        }
                    }
                    /* adjust bl and bu for the removed fixed variables */
                    if ( Nrow )
                    {
                        PASAINT const q1 = Ap [j+1] ;
                        for (p = Ap [j]; p < q1; p++)
                        {
                            i = Ai [p] ;
                            bl [i] -= t*Ax [p] ;
                            bu [i] -= t*Ax [p] ;
                        }
                    }
                }
                else /* variable is free */
                {
                    t = PASAZERO ;
                    for (p = Hp [j]; p < q; p++)
                    {
                        if ( drop_col [Hi [p]] == 1 )
                        {
                            t += Hx [p]*x [Hi [p]] ;
                        }
                    }
                    c [j] += t ;
                }
            }
            fadjust += 0.5*s ;

            /* compress c, lo, and hi */
            free_cols = 0 ;
            l = 0 ;
            ifree = (PASAINT *) malloc (Ncol*sizeof (PASAINT)) ;
            loExists = FALSE ;
            hiExists = FALSE ;
            for (j = 0; j < ncol; j++)
            {
                if ( drop_col [j] == 0 ) /* free variable */
                {
                    c [free_cols] = c [j] ;   /* compress c */
                    lo [free_cols] = lo [j] ; /* compress lo */
                    x [free_cols] = x [j] ;   /* compress starting guess */
                    if ( lo [j] > -PASAINF )
                    {
                        loExists = TRUE ;
                    }
                    hi [free_cols] = hi [j] ; /* compress hi */
                    if ( hi [j] < PASAINF )
                    {
                        hiExists = TRUE ;
                    }
                    ifree [free_cols] = j ;   /* map to prior indices */
                    free_cols++ ;
                }
            }
            printf ("largest element of linear term: %e\n",
                    pasa_sup_normx (c, free_cols)) ;
            if ( !loExists )
            {
                free (lo) ;
                pasadata->lo = lo = NULL ;
            }
            if ( !hiExists )
            {
                free (hi) ;
                pasadata->hi = hi = NULL ;
            }

            /* compress bl and bu */
            free_rows = 0 ;
            if ( Nrow )
            {
                l = 0 ;
                for (i = 0; i < nrow; i++)
                {
                    if ( drop_row [i] == 0 ) /* free variable */
                    {
                        bl [free_rows] = bl [i] ; /* compress bl */
                        bu [free_rows] = bu [i] ; /* compress bu */
                        free_rows++ ;
                    }
                }
            }

            /* remove columns from H corresponding to bound variables */
            free_cols = pasa_compress_matrix (Hp, Hi, Hx, Ncol, drop_col) ;

            /* transpose the matrix with the deleted columns */
            HTp= (PASAINT *)   malloc ((Ncol+1)*sizeof (PASAINT)) ;
            k = Hp [free_cols] ; /* nnz in H */
            HTi= (PASAINT *)   malloc (k*sizeof (PASAINT)) ;
            HTx= (PASAFLOAT *) malloc (k*sizeof (PASAFLOAT)) ;
            pasa_transpose (HTp, HTi, HTx, Hp, Hi, Hx, Ncol, free_cols, tempW) ;

            /* copy HT to H */
            PASAFLOAT *Htempx ;
            Htempx = Hx ; Hx = HTx ; free (Htempx) ;
            PASAINT *Htempi ;
            Htempi = Hi ; Hi = HTi ; free (Htempi) ;
            Htempi = Hp ; Hp = HTp ; free (Htempi) ;

            /* remove deleted columns from the transpose matrix stored in H */
            free_cols = pasa_compress_matrix (Hp, Hi, Hx, Ncol, drop_col) ;

            if ( Nrow )
            {
                /* compress A by removing deleted rows */
                free_rows = pasa_compress_matrix (ATp, ATi, ATx, Nrow,drop_row);

                /* transpose AT to get A with the deleted equations removed */
                pasa_transpose (Ap, Ai, Ax, ATp, ATi, ATx, Ncol, free_rows,
                                tempW) ;

                /* compress A by removing deleted columns */
                free_cols = pasa_compress_matrix (Ap, Ai, Ax, Ncol, drop_col) ;
                Anz = Ap [Ncol] ;
                free (drop_row) ;
                free (ATp) ;
                free (ATi) ;
                free (ATx) ;
            }
            free (tempW) ;

            printf ("number of variables: %ld  ... after removing the bound "
                    "variables: %ld\n",  (LONG) Ncol,  (LONG) free_cols) ;
            Ncol = free_cols ;

            printf ("number of equations: %ld  ... after removing equalities "
                    "with 1 variable: %ld\n\n",
                    (LONG) Nrow, (LONG) free_rows) ;
            Nrow = free_rows ;

            if ( !free_rows )
            {
                if ( bl != NULL )
                {
                    free (bl) ; pasadata->bl = NULL ;
                }
                if ( bu != NULL )
                {
                    free (bu) ; pasadata->bu = NULL ;
                }
            }
            free (drop_col) ;
        }
        pasadata->Hp = Hp ;
        pasadata->Hi = Hi ;
        pasadata->Hx = Hx ;
    }
    else /* problem is not a QP */
    {
        /* use cutest evaluation routines, which depends on problem structure */
        if ( UB ) /* problem is unconstrained or bound constrained */
        {
            pasadata->value   = cg_value ;
            pasadata->grad    = cg_grad ;
            pasadata->valgrad = cg_valgrad ;
        }
        else /* the problem has more general constraints */
        {
            pasadata->value   = pasa_value ;
            pasadata->grad    = pasa_grad ;
            pasadata->valgrad = pasa_valgrad ;
        }
    }
    pasadata->ncol = Ncol ;
    pasadata->nrow = Nrow ;

    if ( Nrow ) printf ("largest nonzero in A: %e\n", pasa_sup_normx (Ax, Anz));
    printf ("fadjust: %e\n", fadjust) ;

    cgparm    = pasadata->Parms->cg ;
    napparm   = pasadata->Parms->napheap ;
    pprojparm = pasadata->Parms->pproj ;
    pasaparm  = pasadata->Parms->pasa ;

    /* Set new parameter values for PASA/CG/PPROJ/NAPHEAP here,
       otherwise default values are used. */

    /* pasaparm->PrintLevel = 3 ; 
    pprojparm->PrintLevel = 3 ; 
    cgparm->PrintLevel = 3 ; */

    /* pasaparm->grad_tol = 1.e-8 ;*/

    /* pasaparm->PrintStat = TRUE ;*/       /* default PrintStat = FALSE */

    /* The 2021 factorization-based version of PASA cannot handle a
       constraint matrix with lots of rows and one or more dense columns
       (since the dense columns cause A'*A to be dense).
       On the other hand, some of the CUTE problems with dense columns
       can be solved by the 2021 version of PASA using the gradient projection
       algorithm and an iterative implementation of PPROJ.  For these
       problems, we take pasaparm->GradProjOnly = TRUE and
       pprojparm->cholmod = FALSE. */
    strcpy (fullpathname, cutest_dir) ;
    file = fopen (strcat (fullpathname, "/classGP"), "r") ;
    while (fgets (pbname, MAXLINE, file) != (char *) NULL)
    {
        for (name = pbname; *name; name++)
        {
            if (isspace (*name)) *name = '\0' ;
        }
        name = pbname ;
        if ( strcmp (name, uname) == 0 )
        {
            printf ("The problem has at least one dense column.\n"
                    "Try the gradient projection algorithm.\n") ;
            pasaparm->GradProjOnly = TRUE ;
            pprojparm->cholmod = FALSE ;
            break ;
        }
    }
    fclose(file) ;

    /* Run time is evaluated using wall clock; to obtain precise times,
       turn off unnecessary CPU consuming processes in the background. */
    walltime = pasa_timer () ; /* timer initialization */

    /* Call the optimizer */
    walltime = pasa_timer () ;
    status_pasa = pasa (pasadata) ;
    /* sopt_printxMATLAB (pasadata->x, Ncol, "x") ;*/

    walltime = pasa_timer () - walltime ;

    /* print statistics for an unconstrained problem */
    pasastat = pasadata->Stats->pasa ;
    pprojstat = pasadata->Stats->pproj ;
    cgstat = pasadata->Stats->cg ;
    napstat = pasadata->Stats->napheap ;
    nchols = 0 ; if ( pasadata->Stats->use_pproj ) nchols = pprojstat->nchols ;
    if ( UB && !loExists && !hiExists )
    {
        printf("\n\n **************** Statistics for CG_DESCENT *********"
               "**********\n\n") ;
        printf("\n\n *********************** CG statistics **************"
               "**********\n\n") ;
        printf("Problem name                            = %-s\n", uname) ;
        printf("Number of variables                     = %-10ld\n",
              (LONG) Ncol) ;
        printf("Iterations of conjugate gradient  (CG)  = %-10ld\n",
              (LONG) cgstat->iter) ;
        printf("Function evaluations in CG              = %-10ld\n",
              (LONG) cgstat->nfunc) ;
        printf("Gradient evaluations in CG              = %-10ld\n\n",
              (LONG) cgstat->ngrad) ;
        printf("Sup-norm of gradient                    = %-16.7e\n",
               cgstat->err) ;
        printf("Final objective value                   = %-16.7e\n",
               cgstat->f+fadjust) ;
        printf("Solution wall time in seconds           = %-11.6f\n",
               walltime) ;
        printf("\n ******************************************************"
                   "********\n\n") ;
    }
    else if ( LP )
    {
        printf("\n\n ******************* PASA LP statistics *************"
               "**********\n\n") ;
        printf("Problem name                            = %-s\n", uname) ;
        printf("Number of variables                     = %-10ld\n",
              (LONG) Ncol) ;
        printf("Number of linear constraints            = %-10ld\n\n",
              (LONG) Nrow) ;
        if ( Nrow > 1 )
        {
            printf("Iterations of Dual Active Set Algorithm = %-10ld\n",
                  (LONG) pasastat->gpit) ;
            printf("Number of projections onto feasible set = %-10ld\n",
                  (LONG) pasastat->nproject) ;
            printf("Number of Cholesky factorizations       = %-10i\n", nchols);
        }
        printf("Sup-norm of projected gradient          = %-16.7e\n",
               pasastat->err) ;
        printf("Final objective value                   = %-16.7e\n",
               pasastat->f+fadjust) ;
        printf("Solution wall time in seconds           = %-11.6f\n",
               walltime) ;
        printf("\n ******************************************************"
                   "********\n\n") ;
    }
    else /* not an LP, general constraints */
    {
        printf("\n\n ***************** Statistics for PASA Run **********"
               "**********\n\n") ;
        printf("Problem name                            = %-s\n", uname) ;
        printf("Number of variables                     = %-10ld\n",
              (LONG) Ncol) ;
        printf("Number of linear constraints            = %-10ld\n\n",
              (LONG) Nrow) ;
        printf("Iterations of gradient projection (GP)  = %-10ld\n",
              (LONG) pasastat->gpit) ;
        printf("Iterations of active set GP             = %-10ld\n",
              (LONG) pasastat->agpit) ;
        printf("Iterations of conjugate gradient  (CG)  = %-10ld\n\n",
              (LONG) cgstat->iter) ;
        printf("Function evaluations in GP              = %-10ld\n",
              (LONG) pasastat->gpnf) ;
        printf("Function evaluations in active set GP   = %-10ld\n",
              (LONG) pasastat->agpnf) ;
        printf("Function evaluations in CG              = %-10ld\n\n",
              (LONG) cgstat->nfunc) ;
        printf("Gradient evaluations in GP              = %-10ld\n",
              (LONG) pasastat->gpng) ;
        printf("Gradient evaluations in active set GP   = %-10ld\n",
              (LONG) pasastat->agpng) ;
        printf("Gradient evaluations in CG              = %-10ld\n\n",
              (LONG) cgstat->ngrad) ;
        printf("Number of projections onto feasible set = %-10ld\n",
              (LONG) pasastat->nproject) ;
        printf("Number of Cholesky factorizations       = %-10i\n", nchols) ;
        printf("Sup-norm of projected gradient          = %-16.7e\n",
               pasastat->err) ;
        printf("Final objective value                   = %-16.7e\n",
               pasastat->f+fadjust) ;
        printf("Solution wall time in seconds           = %-11.6f\n",
               walltime) ;
        printf("\n ******************************************************"
                   "************\n\n") ;
    }
    printf ("Post analysis of KKT error in the computed solution:\n\n") ;
       
    PASAFLOAT abs_Al, E, normx, norml, norm_absAx, *g, abserr [4], relerr [4] ;
    /* if bound constraints exist, report the violation */
    E = PASAZERO ;
    /* g will store gradient of Lagrangian L(x, lambda) = f(x) + lambda'(b-Ax)*/
    g = (PASAFLOAT *) malloc (Ncol*sizeof (PASAFLOAT)) ;
    if ( loExists || hiExists )
    {
        if ( loExists )
        {
            for (j = 0; j < Ncol; j++)
            {
                if ( x [j] < lo [j] )
                {
                    t = lo [j] - x [j] ;
                    if ( t > E ) E = t ;
                }
            }
        }
        if ( hiExists )
        {
            for (j = 0; j < Ncol; j++)
            {
                if ( x [j] > hi [j] )
                {
                    t = x [j] - hi [j] ;
                    if ( t > E ) E = t ;
                }
            }
        }
        normx = pasa_sup_normx (x, Ncol) ;
        if ( normx == PASAZERO ) normx = PASAONE ;
        abserr [0] = E ; relerr [0] = E/normx ;
        printf ("absolute error in variable bounds:     %10.2e  "
                "relative error: %10.2e\n", E, E/normx) ;
    }
    else
    {
        E = PASAZERO ;
        abserr [0] = relerr [0] = E ;
        printf ("absolute error in variable bounds:     %10.2e  "
                "relative error: %10.2e\n", E, E) ;
    }
    /* Evaluate the gradient of the objective at x. We can compute this
       gradient using the CUTE builtin routine except for QPs since these
       were compressed when free variables were present. */
    if ( LP || QP )
    {
        if ( pasadata->c != NULL ) pasa_copyx (g, pasadata->c, Ncol) ;
        else                       pasa_initx (g, PASAZERO, Ncol) ;
        if ( QP )
        {
            p = 0 ;
            for (j = 0; j < Ncol; j++)
            {
                PASAFLOAT const T = x [j] ;
                PASAINT const q = Hp [j+1] ;
                for (; p < q; p++) g [Hi [p]] += T*Hx [p] ;
            }
        }
    }
    else /* general nonlinear function */
    {
        if ( UB ) cg_grad   (g, x, Ncol) ;
        else      pasa_grad (g, x, Ncol) ;
    }

    if ( !UB && (Nrow > 0) )
    {
        /* if linear inequalities exists, report the violation in feasibility */
        E = PASAZERO ;
        PASAFLOAT *y = (PASAFLOAT *) malloc (Nrow*sizeof (PASAFLOAT)) ;
        pasa_initx (y, PASAZERO, Nrow) ;
        PASAFLOAT *absAx = (PASAFLOAT *) malloc (Nrow*sizeof (PASAFLOAT)) ;
        pasa_initx (absAx, PASAZERO, Nrow) ;
        /* compute product of A with x */
        p = 0 ;
        for (j = 0; j < Ncol; j++)
        {
            PASAFLOAT const T = x [j] ;
            for (; p < Ap [j+1]; p++)
            {
                u = Ax [p]*T ;
                y [Ai [p]] += u ;
                absAx [Ai [p]] += fabs(u) ;
            }
        }
        for (i = 0; i < Nrow; i++)
        {
            if ( y [i] < bl [i] )
            {
                t =  bl [i] - y [i] ;
                if ( t > E ) E = t ;
            }
            else if ( y [i] > bu [i] )
            {
                t =  y [i] - bu [i] ;
                if ( t > E ) E = t ;
            }
        }
        norm_absAx = pasa_sup_normx (absAx, Nrow) ;
        if ( norm_absAx == PASAZERO ) norm_absAx = PASAONE ;
        printf ("absolute error in linear constraints:  %10.2e  "
                "relative error: %10.2e\n", E, E/norm_absAx) ;
        abserr [1] = E ; relerr [1] = E/norm_absAx ;
        E = PASAZERO ;
        norml = PASAZERO ;

        /* Check the sign of lambda. When lambda > 0, constraint should be
           at lower bound bl */
        for (i = 0; i < Nrow; i++)
        {
            if ( lambda [i] >= PASAZERO )
            {
                t = PASAMIN (lambda [i], fabs (bl [i] - y [i])) ;
                if ( t > E ) E = t ;
            }
            else
            {
                t = PASAMIN (-lambda [i], fabs (bu [i] - y [i])) ;
                if ( t > E ) E = t ;
            }
        }
        norml = pasa_sup_normx (lambda, Nrow) ;
        if ( norml == 0 ) norml = PASAONE ;
        printf ("absolute error in multiplier sign:     %10.2e  "
                "relative error: %10.2e\n", E, E/norml) ;
        abserr [2] = E ; relerr [2] = E/norml ;
        free (y) ; free (absAx) ;
        E = PASAZERO ;
        /* Now check the gradient of the Lagrangian. It should vanish for the
           free set where lo_F < x_F < hi_F. If the gradient is positive, then
           should have x_i = lo_i, if the gradient is negative, then
           should have x_i = hi_i. Subtract A'lambda from g, the gradient of
           the objective. */
        abs_Al = PASAZERO ;
        p = 0 ;
        for (j = 0; j < Ncol; j++)
        {
            PASAINT const q = Ap [j+1] ;
            t = PASAZERO ;
            s = PASAZERO ;
            for (; p < q; p++)
            {
                u = Ax [p]*lambda [Ai [p]] ;
                t -= u ;
                s += fabs (u) ;
            }
            g [j] += t ;
            if ( abs_Al < s ) abs_Al = s ;
        }
        if ( abs_Al == PASAZERO ) abs_Al = PASAONE ;
    }
    else /* problem of class UB */
    {
        E = PASAZERO ;
        abserr [1] = relerr [1] = abserr [2] = relerr [2] = E ;
        printf ("absolute error in linear constraints:  %10.2e  "
                "relative error: %10.2e\n", E, E) ;
        printf ("absolute error in multiplier sign:     %10.2e  "
                "relative error: %10.2e\n", E, E) ;
        abs_Al = PASAONE ;
    }
    for (j = 0; j < Ncol; j++)
    {
        t = g [j] ;
        if ( t > 0 )
        {
            if ( loExists && (u = fabs (lo [j] - x [j])) < t ) t = u ;
        }
        else /* t <= 0 */
        {
            t = -t ;
            if ( hiExists && (u = fabs (hi [j] - x [j])) < t ) t = u ;
        }
        if ( t > E ) E = t ;
    }
    printf ("absolute error in Lagrangian gradient: %10.2e  "
            "relative error: %10.2e\n", E, E/abs_Al) ;
    abserr [3] = E ; relerr [3] = E/abs_Al ;
    free (g) ;

    /* map x back to original coordinates for QPs */
    if ( QP && (loExists || hiExists) )
    {
        for (j = 0; j < free_cols; j++)
        {
            xcopy [ifree [j]] = x [j] ;
        }
        pasa_copyx (x, xcopy, CUTEst_nvar) ;
        free (ifree) ;
        free (xcopy) ;
    }
    if ( UB ) cg_value (&t, x, CUTEst_nvar) ;
    else      pasa_value (&t, x, CUTEst_nvar) ;
    printf("\nCUTE Function value at final x = %-25.16e\n", t) ;

    /* Print lines to clearly separate problems */

    printf ("\n ========================================================\n\n");

#if 0
    long gpit, gpnf, gpng, agpit, agpnf, agpng, cgit, cgnf, cgng, nproject ;
    PASAFLOAT err, cost ; 
    err = pasastat->err; cost = pasastat->f+fadjust ;
    printf ("dim, gp3, agp3, cg3, nproj, nchol, status, err, cost, time\n") ;
    if ( UB && !loExists && !hiExists )
    {
        gpit = 0; gpnf = 0; gpng = 0; agpit = 0; agpnf = 0; agpng = 0;
        cgit = cgstat->iter; cgnf = cgstat->nfunc; cgng = cgstat->ngrad;
        nproject = 0; err = cgstat->err; cost = cgstat->f+fadjust;
    }
    else if ( LP )
    {
        if ( Nrow > 1 )
        {
            gpit = pasastat->gpit ;
            nproject = pasastat->nproject;
        }
        else
        {
            gpit = nproject = 0 ;
        }
        gpnf = 0; gpng = 0; agpit = 0; agpnf = 0; agpng = 0;
        cgit = 0; cgnf = 0; cgng  = 0;
    }
    else
    {
        gpit = pasastat->gpit; gpnf = pasastat->gpnf; gpng = pasastat->gpng;
        agpit = pasastat->agpit; agpnf = pasastat->agpnf;
        agpng = pasastat->agpng; cgit = cgstat->iter; cgnf = cgstat->nfunc;
        cgng = cgstat->ngrad; nproject = pasastat->nproject;
    }
    printf ("!!%10s %6ld %4ld %4ld %4ld %4ld %4ld %4ld %6ld %6ld %6ld %7ld "
                "%7i %5i %16.7e %16.7e %11.6f\n\n",
                 uname, (LONG) Ncol, gpit, gpnf, gpng, agpit, agpnf, agpng,
                 cgit, cgnf, cgng, nproject, nchols, status_pasa, err, cost,
                 walltime) ;

    printf ("bounds, linear constraints, lambda sign, grad_x L\n") ;
    printf ("##%10s %10.2e %10.2e %10.2e %10.2e %10.2e %10.2e %10.2e %10.2e\n",
             uname,
             abserr[0], relerr[0], abserr[1], relerr[1], abserr[2], relerr[2],
             abserr[3], relerr[3]) ;

    printf ("\nNrow Ncol GPit AGPit CGit nfunc ngrad err cost walltime\n") ;
    printf("++ %-8s %6li %6li %5li %8li %8li %8li %8li %10.2e %14.7e %11.6f",
             uname, (LONG) Nrow, (LONG) Ncol, gpit, agpit, cgit,
             gpnf+agpnf+cgnf, gpng+agpng+cgng, err, cost, walltime) ;
    printf ("\n =========================================================\n\n");
#endif

    ierr = 0;
    FORTRAN_close( &funit, &ierr ) ;
    if ( ierr )
    {
        printf( "Error closing %s on unit %ld.\n", fname, (LONG) funit ) ;
        printf( "Trying not to abort.\n" ) ;
    }
    if ( Nrow > 0 )
    {
        free (Ap) ;
        free (Ai) ;
        free (Ax) ;
        free (lambda) ;
    }

    /* Free workspace */
    free (zeros) ;
    free (uname) ;
    free (x) ;
    if ( lo != NULL ) free (lo) ;
    if ( hi != NULL ) free (hi) ;
    if ( bl != NULL ) free (bl) ;
    if ( bu != NULL ) free (bu) ;
    if ( c != NULL ) free (c) ;
    if ( QP )
    {
        free (Hx) ;
        free (Hi) ;
        free (Hp) ;
    }

    /* free memory allocated for pasa data */
    pasa_terminate (&pasadata) ;

    /* end cutest */
    CUTEST_cterminate( &status ) ;
    fflush (stdout) ;

    return 0 ;
}

#ifdef __cplusplus
}    /* Closing brace for  extern "C"  block */
#endif

void cg_value
(
    PASAFLOAT *f,
    PASAFLOAT *x,
    PASAINT N
)
{
    integer status, n ;

    n = N ;
    CUTEST_ufn( &status, &n, x, f) ;
    if ((status == 1) || (status == 2))
    {
        printf("** CUTEst error, status = %ld, aborting\n", (LONG) status) ;
        exit(status) ;
    }
    return ;
}

void cg_grad
(
    PASAFLOAT *g,
    PASAFLOAT *x,
    PASAINT N
)
{
    integer n, status;
    n = N ;
    CUTEST_ugr( &status, &n, x, g) ;
    if ((status == 1) || (status == 2))
    {
        printf("** CUTEst error, status = %ld, aborting\n", (LONG) status);
        exit(status);
    }
}

void cg_valgrad
(
    PASAFLOAT *f,
    PASAFLOAT *g,
    PASAFLOAT *x,
    PASAINT N
)
{
    integer n, status;
    n = N ;
    CUTEST_uofg( &status, &n, x, f, g, &cute_true) ;
    if ((status == 1) || (status == 2))
    {
        printf("** CUTEst error, status = %ld, aborting\n", (LONG) status);
        exit(status);
    }
    return ;
}

void pasa_value
(
    PASAFLOAT *f,
    PASAFLOAT *x,
    PASAINT N
)
{
    PASAFLOAT *dummy ;
    integer status, n ;

    n = N ;
    CUTEST_cofg(&status, &n, x, f, dummy, &cute_false) ;
    if ((status == 1) || (status == 2))
    {
        printf("** CUTEst error, status = %ld, aborting\n", (LONG) status) ;
        exit(status) ;
    }
    return ;
}

void pasa_grad
(
    PASAFLOAT *g,
    PASAFLOAT *x,
    PASAINT N
)
{
    PASAFLOAT F ;
    integer n, status;
    n = N ;
    CUTEST_cofg( &status, &n, x, &F, g, &cute_true) ;
    if ((status == 1) || (status == 2))
    {
        printf("** CUTEst error, status = %ld, aborting\n", (LONG) status);
        exit(status);
    }
}

void pasa_valgrad
(
    PASAFLOAT *f,
    PASAFLOAT *g,
    PASAFLOAT *x,
    PASAINT N
)
{
    integer n, status;
    n = N ;
    CUTEST_cofg( &status, &n, x, f, g, &cute_true) ;
    if ((status == 1) || (status == 2))
    {
        printf("** CUTEst error, status = %ld, aborting\n", (LONG) status);
        exit(status);
    }
    return ;
}
