    /* Test problem: Hock and Schittkowski problem 112 as implemented in CUTEST

       Dimension: n = 10

         min f(x) s.t. 1.e-6 <= x, Ax = [2 1 1]'

       where

         f(x) = sum_{i = 1 to n} x_i(c_i + log (x_i/ sum_{j = 1 to 10} x_j))

              | 1  2  2  0  0  1  0  0  0  1 |
          A = | 0  0  0  1  2  1  1  0  0  0 |
              | 0  0  1  0  0  0  1  1  2  1 |

          c = [-6.089 -17.164 -34.054  -5.914 -24.721 ...
              -14.986 -24.100 -10.708 -26.662 -22.179 ]

       Starting guess: x_i = 0.1, 1 <= i <= n */

#include "pasa.h"

/* prototypes */
void value
(
    PASAFLOAT *val,
    PASAFLOAT   *x,
    PASAINT      n
) ;

void grad
(
    PASAFLOAT *g,
    PASAFLOAT *x,
    PASAINT    n
) ;

void valgrad
(
    PASAFLOAT *val,
    PASAFLOAT   *g,
    PASAFLOAT   *x,
    PASAINT      n
) ;

/* global c */
    PASAFLOAT c [] = {-6.089, -17.164, -34.054,  -5.914, -24.721,
                     -14.986, -24.100, -10.708, -26.662, -22.179 } ;

/* main function */
int main (void /*int argc, char **argv*/)
{
   /*  Initialize the pasadata structure for storing the problem description.
       pasa_setup returns a pointer to a data structure where
       all parameter values are set to default values. */
    printf ("Initializing PASAdata structure.\n") ;
    PASAdata *pasadata = pasa_setup () ; 
    if ( pasadata == NULL )
    {
        printf ("pasa_setup ran out of memory\n") ;
    }
    else
    {
        printf ("Successfully initialized PASAdata structure.\n") ;
    }

    /* Need to specify the number of columns and rows in A since
       it will be input as a dense matrix by rows and pasa needs to know
       how to extract the rows of the matrix from the array. */
                   pasadata->nrow = 3 ;
    PASAINT ncol = pasadata->ncol = 10 ;

    /* Currently, there are 3 different ways to give the elements of the
       constraint matrix A:

       (1) pasa stores A in sparse format using the arrays
             Ai - row indices of the nonzeros elements. Indices in each
                  column should be in increasing order.
             Ax - the nonzeros numerical entries corresponding to entries of Ai
             Ap - the array of column pointers (size ncol + 1).  Ap [j] is
                  location in Ai or Ax of the first nonzero associated
                  with column j. Ap [ncol] is the total number of nonzeros
                  in the matrix
       (2) a packed array containing all the elements of A, both zeros and
           nonzeros.
       (3) triples format:
             Ti  - row      indices of nonzero entries
             Tj  - column   indices of nonzero entries
             Tx  - numerical values of nonzero entries
             Tnz - number of nonzeros in A

        For this example, it is easiest to use a packed array. Storing A by
        rows, we have */

    PASAFLOAT A [] = { 1, 2, 2, 0, 0, 1, 0, 0, 0, 1,
                       0, 0, 0, 1, 2, 1, 1, 0, 0, 0,
                       0, 0, 1, 0, 0, 0, 1, 1, 2, 1 } ;
    pasadata->A_by_rows = A ;

    /* Input the lower bounds for A*x */
    PASAFLOAT bl [] = { 2, 1, 1 } ;
    pasadata->bl = bl ;
    /* pasadata->bu = bl since the upper and lower bounds are equal */
    pasadata->bu = bl ;

    /* the starting guess is the vector with all its entries 0.1 */
    PASAFLOAT *x = pasadata->x = (PASAFLOAT *) malloc (ncol*sizeof (PPFLOAT)) ;
    pasa_initx (x, 0.1, ncol) ;

    /* lower bounds is the vector with all its entries 1.e-6 */
    PASAFLOAT *lo = pasadata->lo = (PASAFLOAT *) malloc (ncol*sizeof (PPFLOAT));
    pasa_initx (lo, 1.e-6, ncol) ;
    /* nothing is stored in pasadata->hi since there are no upper bounds */

    /* The routines to evaluate the function and the gradient appear below,
       while their prototypes are above.  Since it is often easy to
       evaluate the gradient at the same time as the function value,
       we also code a routine valgrad that simultaneously evaluates
       the function and its gradient.  The value and gradient routines
       are required, while the valgrad is optional. In some applications,
       the solution time is reduced when valgrad is provided. */
    pasadata->value = value ;
    pasadata->grad = grad ;
    pasadata->valgrad = valgrad ;

    /* now solve the problem using pasa */
    pasa (pasadata) ; 

    /* By default, pasadata->Parm->print_status = TRUE, so the status of
       the run will be printed. If this parameter was FALSE, then the user
       could manually print the status by running pasa_print_status(pasadata);
       run status is also an output of pasa: int status = pasa (pasadata) ;
       The run was successful when status = 0.  If the run was successful,
       then the solution is stored in pasadata->x */
    pasa_printx (pasadata->x, ncol, "solution =") ;

    /* The pasa_print_stats routine gives the statistics for all the
       routines used in the run */
    pasa_print_stats (pasadata) ;

    /* The pasa_print_parms routine gives the parameter values of all the
       routines that were used during the solution process. */
    pasa_print_parms (pasadata) ;

    /* Specific statistics can be extracted from the pasadata structure
       as shown below. */
        PASAstat *pasastat = pasadata->Stats->pasa ;
        PPstat  *pprojstat = pasadata->Stats->pproj ;
        CGstat     *cgstat = pasadata->Stats->cg ;

        printf("\n ***************** Statistics for PASA Run **********"
               "**********\n\n") ;
        printf("Number of variables                     = %-10ld\n",
              (LONG) pasadata->ncol) ;
        printf("Number of linear constraints            = %-10ld\n\n",
              (LONG) pasadata->nrow) ;
        printf("Iterations of gradient projection (GP)  = %-10ld\n",
              (LONG) pasastat->gpit) ;
        printf("Iterations of active set GP             = %-10ld\n",
              (LONG) pasastat->agpit) ;
        printf("Iterations of conjugate gradient  (CG)  = %-10ld\n\n",
              (LONG) cgstat->iter) ;
        printf("Function evaluations in GP              = %-10ld\n",
              (LONG) pasastat->gpnf) ;
        printf("Function evaluations in active set GP   = %-10ld\n",
              (LONG) pasastat->agpnf) ;
        printf("Function evaluations in CG              = %-10ld\n\n",
              (LONG) cgstat->nfunc) ;
        printf("Gradient evaluations in GP              = %-10ld\n",
              (LONG) pasastat->gpng) ;
        printf("Gradient evaluations in active set GP   = %-10ld\n",
              (LONG) pasastat->agpng) ;
        printf("Gradient evaluations in CG              = %-10ld\n\n",
              (LONG) cgstat->ngrad) ;
        printf("Number of projections onto feasible set = %-10ld\n",
              (LONG) pasastat->nproject) ;
        printf("Number of Cholesky factorizations       = %-10i\n",
               pprojstat->nchols) ;
        printf("Sup-norm of projected gradient          = %-16.7e\n",
               pasastat->err) ;
        printf("Final objective value                   = %-16.7e\n",
               pasastat->f) ;
        printf("\n ******************************************************"
                   "************\n\n") ;
    /* Note: the default error tolerance is 1.e-6. It can be changed by
             setting a different value for pasadata->Parms->pasa->grad_tol */

    /* free memory malloc'd above */
    free (lo) ;
    free (x) ;

    /* use pasa_terminate to free the pasadata structure and any memory
       that was malloc'd by pasa. */
    pasa_terminate (&pasadata) ;

    /* exit the program */
    return (0) ;
}

void value
(
    PASAFLOAT *val,
    PASAFLOAT   *x,
    PASAINT      n
)
{
    PASAFLOAT f, logt, t ;
    PASAINT i ;

    t = f = PASAZERO ; /* initialize function value f */
    for (i = 0; i < n; i++) t += x [i] ;
    logt = log (t) ;
    for (i = 0; i < n; i++)
    {
        f += x [i]*(c [i] + log (x [i]) - logt) ;
    }
    *val = f;
}

void grad
(
    PASAFLOAT *g,
    PASAFLOAT *x,
    PASAINT    n
)
{
    PASAFLOAT t, logt ;
    PASAINT i ;

    t = PASAZERO ;
    for (i = 0; i < n; i++) t += x [i] ;
    logt = log (t) ;
    for (i = 0; i < n; i++)
    {
        g [i] = c [i] + log (x [i]) - logt ;
    }
}

void valgrad
(
    PASAFLOAT *val,
    PASAFLOAT   *g,
    PASAFLOAT   *x,
    PASAINT      n
)
{
    PASAFLOAT f, logt, s, t ;
    PASAINT i ;

    t = f = PASAZERO ; /* initialize function value f */
    for (i = 0; i < n; i++) t += x [i] ;
    logt = log (t) ;
    for (i = 0; i < n; i++)
    {
        PASAFLOAT const xi = x [i] ;
        s = c [i] + log (xi) - logt ;
        g [i] = s ;
        f += xi*s ;
    }
    *val = f ;
}
