/*  This test problem is an optimal control problem that arises in the
    treatment of the ebola virus (see *). An important feature of the problem
    is that it is singular. If the dynamics were discretized and the cost
    was optimized, then the solution oscillates wildly in the singular
    region, so the optimal control in that region would not be determined.
    The oscillations are removed using a penalty term based on the total
    variation of the optimal control. The dynamics are discretized using
    Euler's method with a constant control on each mesh interval. If u_i
    is the control on the i-th interval, then we add the constraint
    
    u_{i+1} - u_i = iota_i - zeta_i
    
    where zeta_i and iota_i >= 0. The penalty term that we add to the objective
    function is p * sum_{i = 1}^n iota_i + zeta_i, which is p times the total
    variation in the control. Besides the control u, which can be singular,
    there is another control v which is bang/bang. The optimization in this
    problem is over x = [u, zeta, iota, v] where u and v are both constrained
    to the interval [0, 1], while the only constraint on zeta and iota is the
    nonnegativity constraint. If there are n intervals in the mesh, then the
    dimension of x is 4*n - 2 since u and v both have n components while
    zeta and iota have n-1 components. For pasa, the constraints should be
    written in the form bl <= A*x <= bu and lo <= x <= hi. Thus a row of the A
    matrix corresponds to the constraint:
    
             0 <= u_i - u_{i+1} + iota_i - zeta_i <= 0,  0 <= i <= n-2
    
    The state variable in the control problem is the triple (S, I, R)
    corresponding to the susceptible, infected, and recovered individuals.
    The control u is vaccination rate while v is referred to as treatment rate.
    The controls are linear in both the dynamics and the cost, which leads
    to the singularity of the solution.

    * Optimal Control for a SIR Epidemiological Model with Time-varying
    Populations by Urszula Ledzewicz, Mahya Aghaee, and Heinz Schaettler,
    2016 IEEE Conference on Control Applications (CCA), pp. 1268-1273,
    DOI: 10.1109/CCA.2016.7587981 */

#include "pasa.h"

/* prototypes */
void value
(
    PASAFLOAT *val,
    PASAFLOAT   *x,
    PASAINT   ncol
) ;

void grad
(
    PASAFLOAT *g,
    PASAFLOAT *x,
    PASAINT ncol
) ;

void valgrad
(
    PASAFLOAT *val,
    PASAFLOAT   *g,
    PASAFLOAT   *x,
    PASAINT   ncol
) ;

void state (void) ;

void costate (void) ;

/* ------ Initialize constant parameters (global variables) ------ */
PASAINT n = 500 ;            /* Dimension of u and v, number of mesh intervals*/
PASAFLOAT h ;                /* Step size (T/n) */
PASAFLOAT T = 50 ;           /* Time horizon (weeks) */
PASAFLOAT a = 5 ;            /* Constant in cost function */
PASAFLOAT b = 50 ;           /* Constant in cost function */
PASAFLOAT c = 300 ;          /* Constant in cost function */
PASAFLOAT p = 1e-1 ;         /* penalty parameter in cost function */
PASAFLOAT mu = 0.005 ;       /* Disease induced death rate */
PASAFLOAT nu = 0.00188 ;     /* Natural death rate of the population */
PASAFLOAT eta = 0.1 ;        /* Effectiveness of treatment */
PASAFLOAT rho = 0.007 ;      /* Resensitization rate */
PASAFLOAT gamma = 0.00683 ;  /* Birth rate of the population */
PASAFLOAT beta = 0.2426 ;    /* Rate of infectiousness of the disease */
PASAFLOAT kappa = 0.3 ;      /* Effectiveness of vaccination */
PASAFLOAT alpha = 0.00002 ;  /* Rate at which disease is overcome */

PASAFLOAT S_state = 1000 ;   /* Initial state value for S */
PASAFLOAT I_state = 10 ;     /* Initial state value for I */
PASAFLOAT R_state = 0 ;      /* Initial state value for R */
PASAFLOAT S_costate = 0 ;    /* Initial costate value for S */
PASAFLOAT I_costate = 0 ;    /* Initial costate value for I */
PASAFLOAT R_costate = 0 ;    /* Initial costate value for R */

/* additional global arrays */
PASAFLOAT *u, *v, *zeta, *iota, *S, *I, *R, *lS, *lI, *lR ;

/* main function */
int main (void /*int argc, char **argv*/)
{
    PASAINT i, k ;

    /* Step size in discretization: h, T, and n are all global */
    h = T/n ;

    /* Allocate memory for global u, v, zeta, iota, S, I, R, lS, lI, and lR */
    int sI = sizeof (PASAINT) ;
    int sF = sizeof (PASAFLOAT) ;
    u    = (PASAFLOAT *) malloc (n*sF) ;
    v    = (PASAFLOAT *) malloc (n*sF) ;
    zeta = (PASAFLOAT *) malloc ((n-1)*sF) ;
    iota = (PASAFLOAT *) malloc ((n-1)*sF) ;
    S    = (PASAFLOAT *) malloc (n*sF) ;
    I    = (PASAFLOAT *) malloc (n*sF) ;
    R    = (PASAFLOAT *) malloc (n*sF) ;
    lS   = (PASAFLOAT *) malloc (n*sF) ;
    lI   = (PASAFLOAT *) malloc (n*sF) ;
    lR   = (PASAFLOAT *) malloc (n*sF) ;

    /* Initialize the pasadata structure for storing the problem description.
       pasa_setup returns a pointer to a data structure where
       all parameter values are set to default values. */
    printf ("Initializing PASAdata structure.\n") ;
    PASAdata *pasadata = pasa_setup () ;
    if ( pasadata == NULL )
    {
        printf ("pasa_setup ran out of memory\n") ;
    }
    else
    {
        printf ("Successfully initialized PASAdata structure.\n") ;
    }

    /* Input the row and column dimension of the constraint matrix A */
    PASAFLOAT nrow = pasadata->nrow = n - 1 ;
    PASAFLOAT ncol = pasadata->ncol = 4*n - 2 ;

    /* The matrix will be input as a series of triples (i, j, A[i, j])
       where A[i, j] is a nonzero in row i and column j in the matrix.
       The triples are stored in 3 vectors Ti, Tj, and Tx. In the i-th
       row of A, the nonzeros are:
       A[i, i] = 1, A[i, i+1] = -1, A[i, n+i] = -1, A[i, 2*n+i-1] = 1 */

    PASAINT   *Ti = pasadata->Ti = (PASAINT *)   malloc (4*nrow*sI) ;
    PASAINT   *Tj = pasadata->Tj = (PASAINT *)   malloc (4*nrow*sI) ;
    PASAFLOAT *Tx = pasadata->Tx = (PASAFLOAT *) malloc (4*nrow*sF) ;
    k = 0 ;
    for(i = 0; i < nrow; i++)
    {
        Ti [k] = i ;
        Tj [k] = i ;
        Tx [k] = PASAONE ;
        k++ ;
        Ti [k] = i ;
        Tj [k] = i + 1 ;
        Tx [k] = -PASAONE ;
        k++ ;
        Ti [k] = i ;
        Tj [k] = n + i ;
        Tx [k] = -PASAONE ;
        k++ ;
        Ti [k] = i ;
        Tj [k] = n + n + i - 1 ;
        Tx [k] = PASAONE ;
        k++ ;
    }
    /* store the total number of nonzeros in pasadata->Tnz */
    pasadata->Tnz = 4*nrow ;

    /* The lower and upper bounds for A*x are all zero */
    PASAFLOAT *bl = pasadata->bl = (PASAFLOAT *) malloc (nrow*sF) ;
    pasa_initx (bl, (PASAFLOAT) 0, nrow) ;
    /* No need to malloc bu since it is equal to bl */
    pasadata->bu = bl ;

    /* The ncol lower bounds for x are all 0 */
    PASAFLOAT *lo = pasadata->lo = (PASAFLOAT *) malloc (ncol*sF) ;
    pasa_initx (lo, (PASAFLOAT) 0, ncol) ;

    /* The first and last n components of x are bound by 1 while there are
       no bounds on the other components. */
    PASAFLOAT *hi = pasadata->hi = (PASAFLOAT *) malloc (ncol*sF) ;
    pasa_initx (hi, (PASAFLOAT) 1, ncol) ; /* all components set to 1 */
    pasa_initx (hi+n, PASAINF, 2*n - 2) ;  /* infinity for middle components */

    /* A starting guess for the solution x would be stored in pasadata->x.
       Be sure to first malloc the memory:
       pasadata->x = (PASAFLOAT *) malloc (ncol*sizeof (PASAFLOAT)) ;
       We do not give a starting guess, in which case pasa allocates the
       memory for x and uses the starting guess x = 0. */

    /* The routines to evaluate the function and the gradient appear below,
       while their prototypes are above.  Since it is often easy to
       evaluate the gradient at the same time as the function value,
       we also code a routine valgrad below that simultaneously evaluates
       the function and its gradient.  Both the value and gradient routines
       are required, while the valgrad routine is optional.
       By including valgrad, the solution time can be reduced for some
       problems. For this problem, the state subroutine is only called
       one time inside valgrad while it would be called twice
       if the function value and its gradient were evaluated separately. */
    pasadata->value = value ;
    pasadata->grad = grad ;
    pasadata->valgrad = valgrad ;

    /* The default error tolerance is 1.e-6. To change the tolerance to 1.e-8:*/
    pasadata->Parms->pasa->grad_tol = 1.e-8 ;

    /* call pasa to solve the problem */
    pasa (pasadata) ; 

    /* By default, pasadata->Parms->pasa->print_status = TRUE, and the status
       of the run will be printed. If this parameter was FALSE, then the
       status could be printed later by using the statement
       pasa_print_status (pasadata) ;
       The status is also returned by pasa: int status = pasa(pasadata) ;
       If status = 0, then the run was successful. */

    /* Specific statistics can be extracted from the pasadata structure
       as shown below. */
        PASAstat *pasastat = pasadata->Stats->pasa ;
        PPstat  *pprojstat = pasadata->Stats->pproj ;
        CGstat     *cgstat = pasadata->Stats->cg ;

        printf(" ***************** Statistics for PASA Run **********"
               "**********\n\n") ;
        printf("Number of variables                     = %-10ld\n",
              (LONG) ncol) ;
        printf("Number of linear constraints            = %-10ld\n\n",
              (LONG) nrow) ;
        printf("Iterations of gradient projection (GP)  = %-10ld\n",
              (LONG) pasastat->gpit) ;
        printf("Iterations of active set GP             = %-10ld\n",
              (LONG) pasastat->agpit) ;
        printf("Iterations of conjugate gradient  (CG)  = %-10ld\n\n",
              (LONG) cgstat->iter) ;
        printf("Function evaluations in GP              = %-10ld\n",
              (LONG) pasastat->gpnf) ;
        printf("Function evaluations in active set GP   = %-10ld\n",
              (LONG) pasastat->agpnf) ;
        printf("Function evaluations in CG              = %-10ld\n\n",
              (LONG) cgstat->nfunc) ;
        printf("Gradient evaluations in GP              = %-10ld\n",
              (LONG) pasastat->gpng) ;
        printf("Gradient evaluations in active set GP   = %-10ld\n",
              (LONG) pasastat->agpng) ;
        printf("Gradient evaluations in CG              = %-10ld\n\n",
              (LONG) cgstat->ngrad) ;
        printf("Number of projections onto feasible set = %-10ld\n",
              (LONG) pasastat->nproject) ;
        printf("Number of Cholesky factorizations       = %-10i\n",
               pprojstat->nchols) ;
        printf("Sup-norm of projected gradient          = %-16.7e\n",
               pasastat->err) ;
        printf("Final objective value                   = %-16.7e\n",
               pasastat->f) ; 
        printf("\n ******************************************************"
                   "************\n\n") ;

    /* The statistics and parameter values can also be printed
       during the pasa run by setting parameter values. For example,
       to print the the statistics and parameter values during the
       pasa run, insert the following statements before the
       statement: pasa (pasadata) ; */
    pasadata->Parms->pasa->PrintStat = TRUE ;
    pasadata->Parms->pasa->PrintParm = TRUE ;

    /* Even if these statements were not inserted before the call to pasa,
       the statistics and parameter can be printed after the run using
       the following statements:

           pasa_print_stats(pasadata) ;
           pasa_print_parms(pasadata) ; */

    /* Use pasa_terminate to free the pasadata structure and all the memory
       that was malloc'd within pasa. Note that if the user did not malloc
       pasadata->x and provide a starting guess, then the memory allocated
       by pasa for x is freed when pasa_terminate is executed. */
    pasa_terminate (&pasadata) ;

    /* Free additional memory malloc'd in main() */
    pasa_free (Ti) ;
    pasa_free (Tj) ;
    pasa_free (Tx) ;
    pasa_free (bl) ;
    pasa_free (lo) ;
    pasa_free (hi) ;

    /* free the global arrays */
    pasa_free (u) ;
    pasa_free (v) ;
    pasa_free (zeta) ;
    pasa_free (iota) ;
    pasa_free (S) ;
    pasa_free (I) ;
    pasa_free (R) ;
    pasa_free (lS) ;
    pasa_free (lI) ;
    pasa_free (lR) ;

    /* Note that pasa_terminate does not free any memory malloc'd by the user.
       Thus if the user malloc's pasadata->x when specifying the
       starting guess, then the user needs to free it. */

    /* exit the program */
    return (0) ;
}

void state (void)
{
    /* Initialize variables */
    int i ;
    double temp ;

    /* Initialize S, I, and R values */
    S[0] = S_state ;
    I[0] = I_state ;
    R[0] = R_state ;

    /* Compute remaining S, I, and R values */
    for (i=0; i<n-1; i++)
    {
        temp = S[i] + I[i] + R[i] ;
        S[i+1] = S[i] + h*(gamma*temp - nu*S[i] - (beta*S[i]*I[i])/temp
                      + rho*R[i] - kappa*S[i]*u[i]) ;
        I[i+1] = I[i] + h*(beta*S[i]*I[i]/temp - (nu + mu + alpha)*I[i]
                      - eta*I[i]*v[i]) ;
        R[i+1] = R[i] + h*(-nu*R[i] - rho*R[i] + kappa*S[i]*u[i] + alpha*I[i]
                      + eta*I[i]*v[i]) ;
    }
}

void costate (void)
{
    /* Initialize variables */
    PASAINT i ;
    PASAFLOAT temp ;

    /* Initialize lS, lI, and lR values */
    lS[n-1] = S_costate ;
    lI[n-1] = I_costate ;
    lR[n-1] = R_costate ;

    /* Compute remaining S, I, and R values */
    for (i=1; i<n; i++)
    {
        temp = S[n-i] + I[n-i] + R[n-i] ;
        lS[n-i-1] = lS[n-i] + h*lS[n-i]*(gamma - nu - beta*(I[n-i]/temp)
                            + beta*(S[n-i]*I[n-i]/(temp*temp)) - kappa*u[n-i])
                            + h*lI[n-i]*(beta*(I[n-i]/temp)
                            - beta*(S[n-i]*I[n-i]/(temp*temp)))
                            + h*lR[n-i]*(kappa*u[n-i]) ;
        lI[n-i-1] = lI[n-i] + h*a +h*lS[n-i]*(gamma - (beta*S[n-i])/temp 
                            + (beta*S[n-i]*I[n-i])/(temp*temp))
                            + h*lI[n-i]*(( beta*S[n-i])/temp
                            - (beta*S[n-i]*I[n-i])/(temp*temp) 
                            - (nu + mu + alpha) - eta*v[n-i])
                            + h*lR[n-i]*(alpha + eta*v[n-i]) ;
        lR[n-i-1] = lR[n-i] + h*lS[n-i]*(gamma
                            + (beta*S[n-i]*I[n-i])/(temp*temp) + rho)
                            + h* lI[n-i]*(-(beta*S[n-i]*I[n-i])/(temp*temp) ) 
                            + h* lR[n-i]*(-nu - rho) ;
    }
}

void value
(
    PASAFLOAT *val,
    PASAFLOAT   *x,
    PASAINT   ncol
)
{
    PASAFLOAT J, su, sI, sv, szi ;
    PASAINT i ;

    /* Extract u, zeta, iota, and v from x */
    pasa_copyx (u, x, n) ;
    pasa_copyx (zeta, x+n, n-1) ;
    pasa_copyx (iota, x+2*n-1, n-1) ;
    pasa_copyx (v, x+3*n-2, n) ;

    /* Compute state values for S, I, and R */
    state() ;

    /* Use state and control values to compute cost function */
    /* Initialize temp sum values */
    su  = 0 ; 
    sI  = 0 ;
    sv  = 0 ;
    szi = 0 ;

    /* Sum values in vectors */
    for (i = 0; i < n; i++)
    {
        su  += u[i] ; 
        sI  += I[i] ;
        sv  += v[i] ;
    }
    for (i = 0; i < n-1; i++)
    {
        szi += zeta[i] + iota[i] ;
    }

    /* Multiply sums by constants and add to cost function value */
    J = h*(a*sI + b*su + c*sv) ;
    J += p*szi ;

    /* Copy objective function value */
    *val = J;
}

void grad
(
    PASAFLOAT *g,
    PASAFLOAT *x,
    PASAINT ncol
)
{
    PASAINT i ;

    /* Extract u and v from x */
    pasa_copyx (u, x, n) ;
    pasa_copyx (v, x+3*n-2, n) ;

    /* Compute state values for S, I, and R */
    state() ;

    /* Compute costate values for S, I, and R */
    costate() ;

    /* Update gradient values and store in array g to return */
    for(i = 0; i < n; i++)
    {
        /* First n entries are gradient of u */
        g[i] = h*(b - lS[i]*kappa*S[i] + lR[i]*kappa*S[i]) ;
        /* Final n entries are gradient of v */
        g[3*n - 2 + i] = h*(c - lI[i]*eta*I[i] + lR[i]*eta*I[i]) ;
    }

    /* middle 2n - 2 elements of g all equal p */
    pasa_initx (g+n, p, 2*n-2) ;
}

void valgrad
(
    PASAFLOAT *val,
    PASAFLOAT   *g,
    PASAFLOAT   *x,
    PASAINT   ncol
)
{
    PASAFLOAT J, su, sI, sv, szi ;
    PASAINT i ;

    /* Extract u, zeta, iota, and v from x */
    pasa_copyx (u, x, n) ;
    pasa_copyx (zeta, x+n, n-1) ;
    pasa_copyx (iota, x+2*n-1, n-1) ;
    pasa_copyx (v, x+3*n-2, n) ;

    /* Compute state values for S, I, and R */
    state() ;

    /* Compute costate values for S, I, and R */
    costate() ;

    /* Use state and control values to compute cost function */
    /* Initialize temp sum values */
    su  = 0 ; 
    sI  = 0 ;
    sv  = 0 ;
    szi = 0 ;

    /* Loop to compute gradient values and scalars in cost function */
    for(i = 0; i < n; i++)
    {
        /* Cost function terms */
        /* Compute scalars required for cost function */
        su  += u[i] ; 
        sI  += I[i] ;
        sv  += v[i] ;
    }
    for(i = 0; i < n-1; i++)
    {
        szi += zeta[i] + iota[i] ;
    }

    /* Store gradient values in array g */
    for(i = 0; i < n; i++)
    {
        /* First n entries are gradient of u */
        g[i] = h*(b - lS[i]*kappa*S[i] + lR[i]*kappa*S[i]) ;
        /* Final n entries are gradient of v */
        g[3*n - 2 + i] = h*(c - lI[i]*eta*I[i] + lR[i]*eta*I[i]) ;
    }

    /* middle 2n - 2 elements of g all equal p */
    pasa_initx (g+n, p, 2*n-2) ;

    /* Multiply sums by constants and add to cost function value */
    J = h*(a*sI + b*su + c*sv) ;
    J += p*szi ;

    /* Copy objective function value */
    *val = J;
}
