    /* Test problem: Generalized Rosenbrock function

         min f(x)

       where

         f(x) = Sum_{i = 1 to n-1} 100 (x [i+1] - x [i]^2)^2 + (1 - x [i])^2

       We will take n = 10 to obtain a 10 dimensional generalization
       of the ordinary Rosenbrock function. The global minimum is attained
       at x = ones (n, 1). */

#include "pasa.h"

/* prototypes */
void value
(
    PASAFLOAT *val,
    PASAFLOAT   *x,
    PASAINT      n
) ;

void grad
(
    PASAFLOAT *g,
    PASAFLOAT *x,
    PASAINT    n
) ;

void valgrad
(
    PASAFLOAT *val,
    PASAFLOAT   *g,
    PASAFLOAT   *x,
    PASAINT      n
) ;

/* main function */
int main (void /*int argc, char **argv*/)
{
    /* Create the pasadata structure for storing the problem description */
    printf ("Initializing PASAdata structure.\n") ;
    PASAdata *pasadata = pasa_setup () ;

    if ( pasadata == NULL )
    {
        printf ("pasa_setup ran out of memory\n") ;
    }
    else
    {
        printf ("Successfully initialized PASAdata structure.\n") ;
    }

    /* For an unconstrained optimization problem, we need to specify the
       problem dimension and the routines for evaluating the function and
       its gradient. A starting guess for the solution can also be provided.
       If the starting guess is omitted, then it is zero by default. */
    pasadata->ncol = 10 ;

    /* The routines to evaluate the function and its gradient appear below,
       while their prototypes are above.  Since it is often easy to
       evaluate the gradient at the same time as the function value,
       we also code a routine valgrad that simultaneously evaluates
       the function and its gradient.  The value and gradient routines
       are required, while the valgrad is optional. In some applications,
       the solution time is reduced when valgrad is provided. */
    pasadata->value = value ;
    pasadata->grad = grad ;
    pasadata->valgrad = valgrad ;

    /* ---------------------- Customizing parameters ------------------------ */
    /* When pasa_setup() is called, all pasa parameters are set to their
       default values, which means that there is no printing. */

    /* call pasa */
    int status = pasa (pasadata) ; 

    /* By default, pasadata->Parm->print_status = TRUE, so the status of
       the run will be printed. If this parameter was FALSE, then the
       status is printed using pasa_print_status (pasadata) ; the run
       status is output as an integer to the variable status variable above.
       The run was successful when status = 0. If status is nonzero, then
       execution should probably be halted since the result may be
       meaningless or undefined. */
    if ( status ) exit (0) ;

    /* If the run was successful, then the solution is stored in pasadata->x */
    pasa_printx (pasadata->x, pasadata->ncol, "solution =") ;

    /* The pasa_print_stats routine gives all the statistics for the run */
    pasa_print_stats (pasadata) ;

    /* The pasa_print_parms routine gives the parameter values of all the
       routines that were used during the solution process. */
    pasa_print_parms (pasadata) ;

    /* Specific statistics can be extracted from the pasadata structure
       as shown below. For an unconstrained optimization problem, pasa
       uses the conjugate gradient algorithm cg_descent to solve the
       problem. Hence, the only statistics are those associated by
       cg_descent. */

    CGstat *stat = pasadata->Stats->cg ;
    printf("\n\n *********************** PASA statistics **************"
           "**********\n\n") ;
    printf(" Code used                   : pasa\n") ;
    printf(" Problem                     : %-s\n", "Rosenbrock Test Problem") ;
    printf(" # variables                 = %-10i\n", pasadata->ncol) ;
    printf(" # cg iterations             = %-10i\n", stat->iter) ;
    printf(" # cg function evals         = %-10i\n", stat->nfunc) ;
    printf(" # cg gradient evals         = %-10i\n", stat->ngrad) ;
    printf(" ||g||                       = %-16.7e\n", stat->err) ;
    printf(" Final f                     = %-16.7e\n", stat->f) ;

    printf("\n ***********************************************************"
           "*******\n\n") ;

    /* Note: the default error tolerance is 1.e-6. It can be changed by
             setting a different value for pasadata->Parms->pasa->grad_tol */

    /* The statistics and parameter values could have been printed
       during the pasa run by setting parameter values. For example,
       to print the the statistics and parameter values during the
       pasa run, insert the following two statements before the
       "pasa (pasadata) ;" statement. */
    pasadata->Parms->pasa->PrintStat = TRUE ;
    pasadata->Parms->pasa->PrintParm = TRUE ;

    /* use pasa_terminate to free the pasadata structure and all the memory
       that was malloc'd for the pasadata structure. */
    pasa_terminate (&pasadata) ;

    /* exit the program */
    return (0) ;
}

void value
(
    PASAFLOAT *val,
    PASAFLOAT   *x,
    PASAINT      n
)
{
    PASAFLOAT f, t1, t2, t3 ;
    PASAINT i ;

    f = 0 ; /* initialize function value f */

    for (i = 0; i < n - 1; i++)
    {
        t1 = x [i] ;
        t2 = x [i + 1] - t1 * t1 ;
        t3 = t1 - 1 ;
        f += 100 * t2 * t2 + t3 * t3 ;
    }

    *val = f;
}

void grad
(
    PASAFLOAT *g,
    PASAFLOAT *x,
    PASAINT    n
)
{
    PASAFLOAT t0, t1, t2, t3 ;
    PASAINT i ;

    t1 = x [0] ;
    t0 = 200*(x [1] - t1*t1) ;
    g [0] = 2*(t1 - 1 - t0*t1) ;
    for (i = 1; i < n - 1; i++)
    {
        t1 = x [i] ;
        t2 = 200*(x [i + 1] - t1 * t1) ;
        t3 = t1 - 1 ;
        g [i] = 2*(t3 - t2*t1)  + t0 ;
        t0 = t2 ;
    }
    g [i] = t2 ;                        /* last entry of gradient */
}

void valgrad
(
    PASAFLOAT *val,
    PASAFLOAT   *g,
    PASAFLOAT   *x,
    PASAINT      n
)
{
    PASAFLOAT f, t0, t1, t2, t3 ;
    PASAINT i ;

    t1 = x [0] ;
    t3 = t1 - 1 ;
    t0 = 200*(x [1] - t1*t1) ;
    g [0] = 2*(t3 - t0*t1) ;
    f = 0.0025*t0*t0 + t3*t3 ;

    for (i = 1; i < n - 1; i++)
    {
        t1 = x [i] ;
        t2 = 200*(x [i + 1] - t1 * t1) ;
        t3 = t1 - 1 ;
        g [i] = 2*(t3 - t2*t1)  + t0 ;
        f += 0.0025*t2*t2 + t3*t3 ;
        t0 = t2 ;
    }
    g [i] = t2 ;
    *val = f ;
}
