% This test problem is an optimal control problem that arises in the
% treatment of the ebola virus (see *). An important feature of the problem is
% that it is singular. If the dynamics were discretized and the cost
% was optimized, then the solution oscillates wildly in the singular
% region, so the optimal control in that region would not be determined.
% The oscillations are removed using a penalty term based on the total
% variation of the optimal control. The dynamics are discretized using
% Euler's method with a constant control on each mesh interval. If u_i
% is the control on the i-th interval, then we add the constraint
%
% u_{i+1} - u_i = iota_i - zeta_i
%
% where zeta_i and iota_i >= 0. The penalty term that we add to the objective
% function is p * sum_{i = 1}^n iota_i + zeta_i, which is p times the total
% variation in the control. Besides the control u, which can be singular,
% there is another control v which is bang/bang. The optimization in this
% problem is over x = [u, zeta, iota, v] where u and v are both constrained
% to the interval [0, 1], while the only constraint on zeta and iota is the
% nonnegativity constraint. If there are n intervals in the mesh, then the
% dimension of x is 4*n - 2 since u and v both have n components while
% zeta and iota have n-1 components. For pasa, the constraints should be
% written in the form bl <= A*x <= bu and lo <= x <= hi. Thus a row of the A
% matrix corresponds to the constraint:
%
%          0 <= u_i - u_{i+1} + iota_i - zeta_i <= 0,  1 <= i <= n-1
% 
% The state variable in the control problem is the triple (S, I, R)
% corresponding to the susceptible, infected, and recovered individuals.
% The control u is vaccination rate while v is referred to as treatment rate.
% The controls are linear in both the dynamics and the cost, which leads
% to the singularity of the solution.
%
%   * Optimal Control for a SIR Epidemiological Model with Time-varying
%   Populations by Urszula Ledzewicz, Mahya Aghaee, and Heinz Schaettler,
%   2016 IEEE Conference on Control Applications (CCA), pp. 1268-1273,
%   DOI: 10.1109/CCA.2016.7587981

function demoOC
    % ------ Initialize constant parameters ------ %
    global gamma beta nu rho kappa mu alpha eta T n a b c p
    clf ;
    gamma = 0.00683 ;  %birth rate of the population
    nu = 0.00188 ;     %natural death rate of the population
    beta = 0.2426 ;    %rate of infectiousness of the disease
    rho = 0.007 ;      %resensitization rate
    kappa = 0.3 ;      %effectiveness of vaccination
    mu = 0.005 ;       %disease induced death rate
    alpha = 0.00002 ;  %rate at which disease is overcome
    eta = 0.1 ;        %effectiveness of treatment
    T = 50 ;           %time horizon (weeks)
    n = 500 ;          % Dimension of u and v, the number of mesh intervals
    a = 5 ;            % Constant in cost function
    b = 50 ;           % Constant in cost function
    c = 300 ;          % Constant in cost function
    p = 1e-1 ;         % penalty parameter in cost function
    umax = 1 ;         % maximum vaccination rate
    vmax = 1;          % maximum treatment rate
    h = T/n ;          % Step size
    S = zeros (n, 1) ;
    I = zeros (n, 1) ;
    R = zeros (n, 1) ;

%% --- Store problem description in a structure which we call pasadata --- %
    % ------------ Setup sparse matrix A ----------- %
    A1 = spdiags([ones(n-1,1) -ones(n-1,1)], [0,1], n-1, n) ;
    A2 = speye(n-1) ;
    A = sparse (n-1, 4*n-2) ;
    % A1 corresponds to the u_i - u_{i+1} term in the contraint while
    % A2 is used for the zeta and iota terms in the constraint
    A(:, 1:3*n-2) = [A1 -A2 A2] ;

    % put A in the structure
    pasadata.A = A ;

    % If the constraint bl <= A*x <= bu is present, then pasa uses the
    % dimensions of A to determine the number of linear constraints
    % and the number of components in x. Thus if the constraint matrix lies
    % in the upper left nrow by ncol submatrix of a larger matrix Afull,
    % set pasadata.A = Afull(1:nrow, 1:ncol). In the example above, the
    % statement A = sparse (n-1, 4*n-2) specified the dimensions of A

    % ------- store the bounds for A*x ------- %
    pasadata.bl = zeros (n-1, 1) ;
    pasadata.bu = zeros (n-1, 1) ;

    % -------- store the bounds for x -------- %
    pasadata.lo = zeros(4*n-2,1) ;
    pasadata.hi = ...
            [umax*ones(n,1); inf*ones(n-1,1); inf*ones(n-1,1); vmax*ones(n,1)] ;

    % The codes to evaluate the cost function and its gradient appear below.
    % Store the name of the codes in the pasadata structure.
    pasadata.grad  = @grad ;  % objective gradient
    pasadata.value = @cost ;  % objective value

%% ---------------- User defined parameter values for pasa ------------- %%
    % Type "pasa readme" for discussion of the parameters.
    % By default, there is no printing of statistics.
    pasadata.pasa.PrintStat = 1 ;    % print statistics for used routines
    pasadata.pasa.grad_tol = 1.e-8 ; % PASA stopping tolerance (1.e-6 default)

    % --------------- Call pasa to determine optimal x -------------- %
    [x, stats] = pasa (pasadata) ;

    % Since pasadata.pasa.PrintStat = 1, the statistics are displayed at
    % the end of the run. Since the stats structure was included as an
    % output, the corresponding numerical entries can be found in the
    % structures stats.pasa and stats.cg

    % ---------------------- Plot the states ---------------------- %
    u = x (1:n) ;          % extract the control u from the returned solution x
    v = x (3*n-1:4*n-2) ;  % extract the control v from the returned solution x
    [S, I, R] = state (u, v) ; % the state associated with the optimal controls

    t = linspace (0, T-h, n) ;
    subplot(3,2,1)
    plot (t, S,'linewidth',2) ;
    xlabel('Time')
    ylabel('S')
    subplot(3,2,2)
    plot (t, I, 'linewidth',2) ;
    xlabel('Time')
    ylabel('I')
    subplot(3,2,3)
    plot (t, R, 'linewidth',2) ;
    xlabel('Time')
    ylabel('R')

    % ----------------- Plot the controls u and v ----------------- %
    subplot(3,2,5);
    plot (t, u,'linewidth',2) ;
    xlabel('Time')
    ylabel({'Control u','(Vaccination)'})
    subplot(3,2,6);
    plot (t, v, 'linewidth',2) ;
    xlabel('Time')
    ylabel({'Control v','(Treatment)'})

%% ------------------ User defined functions for pasa ------------------ %%
    % ---- Objective function ---- %
    function J = cost(x)
        h = T/n ;
        u = x(1:n) ;
        zeta = x(n+1:2*n-1) ;
        iota = x(2*n:3*n-2) ;
        v = x(3*n-1: 4*n-2) ;
        [S, I, R] = state (u, v) ;
        J = h*(a*sum(I) + b*sum(u) + c*sum(v)) ;
        J = J + p*sum(zeta + iota) ;
    end

    % ---- Gradient of objective function ---- %
    function g = grad(x)
        h = T/n ;
        u = x(1:n) ;
        v = x(3*n-1: 4*n-2) ;

        % Compute state and costate
        [S, I, R] = state (u, v);
        [lS, lI, lR] = costate (S, I, R, u, v) ;

        % Update gradient values
        for i=1:n
            Fu(i) = h*(b - lS(i)*kappa*S(i) + lR(i)* kappa*S(i)) ;
            Fv(i) = h*(c - lI(i)*eta*I(i) + lR(i)*eta*I(i)) ;
        end

        % Store gradient values in array g to return
        g = [Fu, p*ones(1,n-1), p*ones(1,n-1), Fv] ;
    end

    % ---- State ---- %
    function [S, I, R] = state (u, v)
        h = T/n;
        S(1) = 1000 ;
        I(1) = 10 ;
        R(1) = 0 ;

        for i = 1:n-1
            N = S(i) + I(i) + R(i) ;
            S(i + 1) = S(i) + h*(gamma*N - nu*S(i) - (beta*S(i)*I(i))/N ...
                +  rho*R(i) - kappa*S(i)*u(i)) ;
            I(i + 1) = I(i) + h*(beta*S(i)*I(i)/N - (nu + mu + alpha)*I(i)...
                - eta*I(i)*v(i)) ;
            R(i + 1) = R(i) + h*( - nu*R(i) - rho*R(i) ...
                +  kappa*S(i)*u(i) + alpha*I(i) + eta*I(i)*v(i)) ;
        end
    end

    % ---- Costate ---- %
    function [lS, lI, lR] = costate (S, I, R, u, v)
        h = T/n ;
        lS(n) = 0 ;
        lI(n) = 0 ;
        lR(n) = 0 ;

        for i=n:-1:2
            N = S(i) + I(i) + R(i) ;
            lS(i-1) = lS(i)+ h*lS(i)*(gamma - nu - beta*(I(i)/N) ...
                        + beta*(S(i)*I(i)/(N^2)) - kappa*u(i))...
                        + h*lI(i)*(beta*(I(i)/N) - beta*(S(i)*I(i)/(N^2)))...
                        + h*lR(i)*(kappa*u(i)) ;
            lI(i-1) = lI(i)+ h*a +h*lS(i)*(gamma - (beta*S(i))/N ...
                        + (beta*S(i)*I(i))/N^2)...
                        + h*lI(i)*(( beta*S(i))/N - (beta*S(i)*I(i))/(N^2) ...
                        - (nu + mu + alpha) - eta*v(i))...
                        + h*lR(i)*(alpha + eta*v(i)) ;
            lR(i-1) = lR(i)+ h*lS(i)*(gamma + (beta*S(i)*I(i))/(N^2) + rho)...
                        + h* lI(i)*(- (beta*S(i)*I(i))/(N^2) ) ...
                        + h* lR(i)*(- nu - rho) ;
        end
    end
end
