function demoQP
%% -------------------------- Problem Details -------------------------- %%
%   This demo shows how to set up and solve the CUTEst QP problem BIGGSC4.
%   The problem is as follows:
%   
%              min             0.5 x'H*x
%              s.t.     0 <= x <= 5, bl <= A*x <= bu
%
%   where A = |1 1 0 0| bl = |2.5| bu = |7.5|
%             |1 0 1 0|      |2.5|      |7.5|
%             |1 0 0 1|      |2.5|      |7.5|
%             |0 1 1 0|      |2.0|      |7.0|
%             |0 1 0 1|      |2.0|      |7.0|
%             |0 0 1 1|      |1.5|      |6.5|
%             |1 1 1 1|      |5.0|      |inf|
%
%   H is completely zero except for 4 elements:
%
%                  h_{31} = h_{13} = h_{42} = h_{24} = -1
%
%% --------------------- Initialize inputs for pasa -------------------- %%

%   A is input as dense matrix.
    A = [1 1 0 0
         1 0 1 0
         1 0 0 1
         0 1 1 0
         0 1 0 1
         0 0 1 1
         1 1 1 1] ;
    pasadata.A = A ;
    pasadata.bl = [2.5 2.5 2.5 2.0 2.0 1.5 5.0]' ;
    pasadata.bu = [7.5 7.5 7.5 7.0 7.0 6.5 inf]' ;
    pasadata.lo = zeros (4, 1) ;
    pasadata.hi = 5*ones (4,1) ;

    % The Hessian is input as triples (row #, column #, matrix element)
    % and then converted to a sparse matrix.
    H = [ 3  1 -1
          1  3 -1
          4  2 -1
          2  4 -1 ] ;
    pasadata.H = sparse (H(:, 1), H(:, 2), H(:, 3)) ;

    % Since the linear term in the quadratic objective is zero, there is no
    % need to define pasadata.c.

%% ---------------- User defined parameter values for pasa ------------- %%
    % See pasa_default inside SuiteOPT/PASA/Source/pasa.c for a description
    % of the parameters. A printout of the parameters for pasa is generated
    % by the statement: pasadata.pasa.PrintParm = 1 ; by default,
    % PrintParm = 0 and no parameters are printed. Detailed statistics
    % for pasa is generated by the statement: pasadata.pasa.PrintStat = 1 ;
    % Again, the default is PrintStat = 0.

    % --------------------- Set up the pasa structure -------------------- %
    % since an initial guess pasadata.x for the solution and pasadata.lamba
    % for the multiplier associated with the constraints bl <= A*x <= bu
    % are not given, they are assumed to be zero
    % if the constraints bl <= A*x <= bu did not exist, then the assignments
    % to pasadata.A, pasadata.bl, and pasadata.bu would be omitted. If
    % pasadata.lo is omitted, it is assumed to be -infinity. Similarly, if
    % pasadata.hi is omitted, it is assumed to be +infinity.
    % The bounds bl and bu are treated in a similar fashion.

    % -------------------------- Call pasa -------------------------- % 
    [x, stats, lambda] = pasa (pasadata) ;
    % The stats and lambda outputs are optional.

%% ------------------------ Print pasa solution ------------------------- %%

    % print x 
    fprintf('\n  -------------------------- Solution ')
    fprintf('-------------------------\n')
    fprintf('    x = ') 
    disp(x)

    % We extract statistics from the stats structure and display them below.
    fprintf('\n\n ***************** Statistics for PASA Run **********') ;
    fprintf('**********\n\n') ;

    [nrow ncol] = size(A) ;
    fprintf('Number of variables                     = %-10ld\n', ncol) ;
    fprintf('Number of linear constraints            = %-10ld\n\n', nrow) ;
    fprintf('Iterations of gradient projection (GP)  = %-10ld\n', ...
              stats.pasa.gpit) ;
    fprintf('Iterations of active set GP             = %-10ld\n\n', ...
              stats.pasa.agpit) ;
    fprintf('Function evaluations in GP              = %-10ld\n', ...
              stats.pasa.gpnf) ;
    fprintf('Function evaluations in active set GP   = %-10ld\n\n', ...
              stats.pasa.agpnf) ;
    fprintf('Gradient evaluations in GP              = %-10ld\n', ...
              stats.pasa.gpng) ;
    fprintf('Gradient evaluations in active set GP   = %-10ld\n\n', ...
              stats.pasa.agpng) ;
    fprintf('Number of projections onto feasible set = %-10ld\n', ...
              stats.pasa.nproject) ;
    fprintf('Number of Cholesky factorizations       = %-10i\n', ...
              stats.pproj.nchols) ;
    fprintf('Sup-norm of projected gradient          = %-16.7e\n', ...
              stats.pasa.err) ;
    fprintf('Final objective value                   = %-16.7e\n', ...
              stats.pasa.f) ;
    fprintf('\n ******************************************************') ;
    fprintf('************\n\n') ;
