/* =========================================================================
   ================================= pasa ==================================
   =========================================================================
       ________________________________________________________________
      |Solve an optimization problem with linear and bound constraints |
      |                                                                |
      |      min f(x)  subject to  lo <= x <= hi,  Bl <= Ax <= Bu      |
      |                                                                |
      |  An active set method is employed. The algorithm uses the      |
      |  gradient projection method to identify active constraints     |
      |  and a projected conjugate gradient algorithm in an active     |
      |  face of the constraint polyhedron.                            |
      |                                                                |
      |                Version 1.0  (September 1, 2019)                |
      |                        William W. Hager                        |
      |                     University of Florida                      |
      |                                                                |
      |        Copyright by William W. Hager  January 1, 2018          |
      |________________________________________________________________|

       ________________________________________________________________
      |This program is free software; you can redistribute it and/or   |
      |modify it under the terms of the GNU General Public License as  |
      |published by the Free Software Foundation; either version 2 of  |
      |the License, or (at your option) any later version.             |
      |This program is distributed in the hope that it will be useful, |
      |but WITHOUT ANY WARRANTY; without even the implied warranty of  |
      |MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   |
      |GNU General Public License for more details.                    |
      |                                                                |
      |You should have received a copy of the GNU General Public       |
      |License along with this program; if not, write to the Free      |
      |Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, |
      |MA  02110-1301  USA                                             |
      |________________________________________________________________|

      Alternative licenses are also available.
      Please contact William Hager for details.
*/

#include "pasa.h"

int pasa
(
    PASAdata  *pasadata /* Structure containing pasa input data, it is
                           initialized using pasa_setup and then modified
                           to describe the user's problem */
)
{
    int Aexists, exponent, loExists, hiExists, Bounds, ProbType,
        repeat, status, toggle, use_napheap, use_pproj, use_penalty ;
    PASAINT Afixed, Asize, i, j, k, kold, l, mem, nfree, ni, nr, ns, nsing,
            p0, p, q, ucol, *Ai, *Ap, *colperm, *ifree, *iwork,
           *ineqList, *rowList, *si, *sj, *singcount, *singperm,
           *row_sing ;
    PASAFLOAT UBl, UBu, r, s, t, u, *Ax, *cgwork, *b, *Bl, *Bu, *c, *sc,
              *singc, *lo, *slo, *singlo, *hi, *RowScale,
              *shi, *singhi, *sx, *temp, *work;
    char const FT [] = "FT" ;
    PASAcom Com ;
    PASAhess Hess ; /* Hessian structure */
    PASAcopy Copy ; /* copy of problem data in pproj format */
    PPdata ppdataDEBUG ; /* input data for PPROJ */
    PPprob *Prob ;

    /* extract the problem data from the pasadata structure */
    PASAINT              nrow = pasadata->nrow ;
    PASAINT              ncol = pasadata->ncol ;
    PASAFLOAT         *userlo = pasadata->lo ;
    PASAFLOAT         *userhi = pasadata->hi ;
    PASAFLOAT         *userBl = pasadata->bl ;
    PASAFLOAT         *userBu = pasadata->bu ;
    PASAFLOAT          *userc = pasadata->c ;
    PASAFLOAT          *userd = pasadata->d ;
    PASAFLOAT          *usera = pasadata->a ;
    PASAFLOAT              *x = pasadata->x ;
    PASAFLOAT         *lambda = pasadata->lambda ;
    PASAparms          *Parms = pasadata->Parms ;
    PASAstats          *Stats = pasadata->Stats ;
    PASAparm        *pasaparm = Parms->pasa ;
    PASAstat        *pasastat = Stats->pasa ;
    CGparm            *cgparm = Parms->cg ;
    CGstat            *cgstat = Stats->cg ;
    PPparm         *pprojparm = Parms->pproj ;
    PPstat         *pprojstat = Stats->pproj ;
    NAPparm          *napparm = Parms->napheap ;
    NAPstat          *napstat = Stats->napheap ;
    PASAFLOAT        grad_tol = pasaparm->grad_tol ;
    PASAFLOAT          *xWork = pasadata->xWork ;
    PASAINT            *iWork = pasadata->iWork ;
    int const      PrintLevel = pasaparm->PrintLevel ;
    int const    GradProjOnly = pasaparm->GradProjOnly ;
    int const use_restoration = pasaparm->use_restoration ;
    int const     use_cholmod = pprojparm->cholmod ;
    int const              sI = sizeof (PASAINT) ;
    int const              sF = sizeof (PASAFLOAT) ;
    NAPdata          *napdata = pasadata->napdata ;
    PPdata            *ppdata = pasadata->ppdata ;
    CGdata            *cgdata = pasadata->cgdata ;

    Com.temp_userBl    = NULL ;
    Com.temp_userBu    = NULL ;
    Com.temp_ifree     = NULL ;

    Com.pasadata  = pasadata ;
    Com.Stats     = Stats ;
    Com.pasaparm  = pasaparm ;
    Com.pasastat  = pasastat ;
    Com.napparm   = napparm ;
    Com.napstat   = napstat ;
    Com.pprojparm = pprojparm ;
    Com.pprojstat = pprojstat ;
    Com.cgparm    = cgparm ;
    Com.cgstat    = cgstat ;
    Com.Copy      = &Copy ;
    Com.Hess      = &Hess ;
    Com.ucol      = ucol = ncol ; /* user specified columns */
    Com.Copy->col_start = NULL ;
    Com.flagAG    = FALSE ;       /* T => active gradproj had ascent */
    Com.lambda_scale = PASAONE ;

    Stats->use_pasa    = FALSE ;
    Stats->use_napheap = FALSE ;
    Stats->use_pproj   = FALSE ;
    Stats->use_cg      = FALSE ;
    /* the FALSE above is changed to TRUE when pasa uses one of these codes */

    /* print the pasa parameters if requested by user */
    int const PrintParm = pasaparm->PrintParm ;
    if ( PrintParm )
    {
        pasa_print_parm (pasadata) ;
    }

    if ( ncol <= 0 )
    {
        status = PASA_PROBLEM_DIMENSION_NOT_GIVEN ;
        pasa_wrapup (status, TRUE, &Com) ;
        return (status) ;
    }

    /* If necessary, convert an input dense matrix or a matrix in triples
       format  to a sparse matrix, beginning with the constraint matrix. */
    status = PASA_OK /* = -1 */ ;
    if ( pasadata->Ap == NULL )
    {
        /* check for triples format */
        if ( (pasadata->Ti != NULL) && (pasadata->Tj != NULL) &&
             (pasadata->Tx != NULL) && (pasadata->Tnz > 0) )
        {
            PASAINT Nrow, Ncol ;
            Nrow = nrow ;
            Ncol = ncol ;
            status = pasa_convert_triple_to_sparse (&Ap, &Ai, &Ax, &Nrow, &Ncol,
                     pasadata->Ti, pasadata->Tj, pasadata->Tx, pasadata->Tnz,
                     TRUE, pasadata->sym);
            if ( status == SOPT_MATRIX_ELEMENT_WAS_ZERO )
            {
                status = 0 ; /* warning already printed by sopt */
            }
            /* NOTE: if nrow is empty, then it is replaced by the
                     max number of rows associated with Ti */
            if ( Nrow > nrow )
            {
                pasadata->nrow = nrow = Nrow ;
            }
        }
        else if ( (pasadata->A_by_rows != NULL)||(pasadata->A_by_cols != NULL) )
        {
            PASAFLOAT *denseA ;
            int const by_rows = (pasadata->A_by_rows == NULL) ? FALSE : TRUE ;
            if ( by_rows ) denseA = pasadata->A_by_rows ;
            else           denseA = pasadata->A_by_cols ;
            status = pasa_convert_dense_to_sparse (&Ap, &Ai, &Ax, denseA,
                                                    nrow, ncol, by_rows) ;
        }

        /* status = PASA_OK => do nothing, no triples/dense input */
        if ( status != PASA_OK )
        {
            /* if status nonzero, then there was an error */
            if ( status )
            {
                /* map from sopt error messages to pasa error messages */
                if ( status == SOPT_OUT_OF_MEMORY )
                    status = PASA_OUT_OF_MEMORY ;
                else if ( status == SOPT_ERROR_IN_INPUT_MATRIX )
                    status = PASA_ERROR_IN_INPUT_MATRIX ;
                pasastat->status = status ;
                if ( pasaparm->PrintStatus ) pasa_print_status (pasadata) ;
                return (status) ;
            }
            else /* success */
            {
                pasadata->A_created = TRUE ;
                pasadata->Ap = Ap ;
                pasadata->Ai = Ai ;
                pasadata->Ax = Ax ;
            }
        }
    }

    if ( pasaparm->CheckMatrix )
    {
        status =
        pasa_check_matrix (pasadata->Ap, pasadata->Ai, pasadata->Ax, ncol) ;
        if ( status )
        {
            status = PASA_ERROR_IN_INPUT_MATRIX ;
            pasa_wrapup (status, TRUE, &Com) ;
            return (status) ;
        }
    }
    PASAINT   *userAp = pasadata->Ap ;
    PASAINT   *userAi = pasadata->Ai ;
    PASAFLOAT *userAx = pasadata->Ax ;

    /* if the Hessian for a quadratic is input as either a dense matrix or
       as a triple, then convert to sparse matrix format */
    status = PASA_OK ;
    if ( pasadata->Hp == NULL )
    {
        PASAINT *Hp, *Hi ;
        PASAFLOAT *Hx ;
        /* check for triples */
        if ( (pasadata->HTi != NULL) && (pasadata->HTj != NULL) &&
             (pasadata->HTx != NULL) && (pasadata->Hnz > 0) )
        {
            PASAINT Nrow, Ncol ;
            Nrow = Ncol = ncol ;
            status = pasa_convert_triple_to_sparse (&Hp, &Hi, &Hx, &Nrow, &Ncol,
                              pasadata->HTi, pasadata->HTj, pasadata->HTx,
                              pasadata->Hnz, TRUE, pasadata->Hsym) ;
            if ( (Nrow != Ncol) && (status != SOPT_OUT_OF_MEMORY)
                                && (status != SOPT_ERROR_IN_INPUT_MATRIX) )
            {
                printf ("In pasa, the user input the Hessian in triples\n"
                        "format, however, the matrix was not symmetric\n") ;
                status = PASA_ERROR_IN_INPUT_MATRIX ;
            }
            else if ( status == SOPT_MATRIX_ELEMENT_WAS_ZERO )
            {
                status = 0 ; /* warning already printed by sopt */
            }
            /* check for repeating elements in column if requested */
            if ( pasaparm->CheckMatrix && (status == 0) )
            {
                PASAINT newHi, oldHi, bad_col ;
                p = 0 ;
                for (j = 0; j < ncol; j++)
                {
                    q = Hp [j+1] ;
                    if ( q > p )
                    {
                        oldHi = Hi [p] ;
                        for (p++; p < q; p++)
                        {
                            if ( (newHi = Hi [p]) <= oldHi )
                            {
                                status = PASA_HESSIAN_ELEMENT_REPEATS ;
                                bad_col = j ;
                            }
                            oldHi = newHi ;
                        }
                    }
                }
                if ( status == PASA_HESSIAN_ELEMENT_REPEATS )
                {
                    printf ("In pasa, the user input the Hessian in\n"
                        "triples format; it was found that a row index\n"
                        "repeated in one or more columns. For example,\n"
                            "see column %ld\n", (LONG) bad_col) ;
                    status = PASA_ERROR_IN_INPUT_MATRIX ;
                }
            }
        }
        else if ( pasadata->Hdense != NULL )
        {
            status = pasa_convert_dense_to_sparse (&Hp, &Hi, &Hx,
                                          pasadata->Hdense, ncol, ncol, FALSE) ;
        }
        /* if status = PASA_OK, then do nothing since no triples/dense input */
        if ( status != PASA_OK )
        {
            /* if status nonzero, then there was an error */
            if ( status )
            {
                /* map from sopt error messages to pasa error messages */
                if ( status == SOPT_OUT_OF_MEMORY )
                     status = PASA_OUT_OF_MEMORY ;
                else if ( status == SOPT_ERROR_IN_INPUT_MATRIX )
                     status = PASA_ERROR_IN_INPUT_MATRIX ;
                pasastat->status = status ;
                if ( pasaparm->PrintStatus ) pasa_print_status (pasadata) ;
                return (status) ;
            }
            else /* success */
            {
                pasadata->H_created = TRUE ;
                pasadata->Hp = Hp ;
                pasadata->Hi = Hi ;
                pasadata->Hx = Hx ;
                status = PASA_OK ;
            }
        }
    }

    i = (pasaparm->UNC  == TRUE) + (pasaparm->BNC     == TRUE) +
        (pasaparm->LP   == TRUE) + (pasaparm->QP      == TRUE) +
        (pasaparm->NL   == TRUE) + (pasaparm->NAPSACK == TRUE) +
        (pasaparm->PROJ == TRUE) ;
    if ( i > 1 )
    {
        status = PASA_PROBLEM_OVERSPECIFIED ;
        pasa_wrapup (status, TRUE, &Com) ;
        return (status) ;
    }

    if ( i == 1 ) /* user uniquely specified the problem classification */
    {
        if ( PrintLevel )
        {
            printf ("User specified problem classification:\n") ;
        }
        if      ( pasaparm->UNC      == TRUE ) ProbType = PASA_UNC ;
        else if ( pasaparm->BNC      == TRUE ) ProbType = PASA_BNC ;
        else if ( pasaparm->LP       == TRUE ) ProbType = PASA_LP ;
        else if ( pasaparm->QP       == TRUE ) ProbType = PASA_QP ;
        else if ( pasaparm->NL       == TRUE ) ProbType = PASA_NL ;
        else if ( pasaparm->NAPSACK  == TRUE ) ProbType = PASA_NAPSACK ;
        else                                   ProbType = PASA_PROJ ;
    }
    else /* i = 0, problem not specified */
    {
        status = pasa_autodetect (pasadata) ;
        if ( status > 0 )
        {
            pasa_wrapup (status, TRUE, &Com) ;
            return (status) ;
        }
        /* if status < 0, it defines the problem classification */
        ProbType = status ;
        if ( PrintLevel )
        {
            printf ("Problem classification deduced by autodetect:\n") ;
        }
    }
    if ( PrintLevel)
    {
        char *mess [7] ;
        mess [0] = "unconstrained optimization" ;
        mess [1] = "bound constrained optimization" ;
        mess [2] = "linear program" ;
        mess [3] = "quadratic program" ;
        mess [4] = "nonlinear program with polyhedral constraints" ;
        mess [5] = "separable convex quadratic objective with knapsack "
                   "constraints" ;
        mess [6] = "projection onto a polyhedron" ;
        printf ("%s\n", mess [PASA_UNC-ProbType]) ;
    }

    int const     UNC = (ProbType == PASA_UNC)     ? TRUE : FALSE ;
    int const     BNC = (ProbType == PASA_BNC)     ? TRUE : FALSE ;
    int const      LP = (ProbType == PASA_LP)      ? TRUE : FALSE ;
    int const      QP = (ProbType == PASA_QP)      ? TRUE : FALSE ;
    int const      NL = (ProbType == PASA_NL)      ? TRUE : FALSE ;
    int const NAPSACK = (ProbType == PASA_NAPSACK) ? TRUE : FALSE ;
    int const    PROJ = (ProbType == PASA_PROJ)    ? TRUE : FALSE ;
    Com.QP = QP ;
    Com.LP = LP ; pasadata->LP = LP ;
    Com.PROJ = PROJ ;

    if ( QP && (pasadata->value != NULL) )
    {
        if ( PrintLevel )
        {
            printf ("\nWarning: There is an ambiguity in the objective\n"
                      "         since the user provides both a pointer\n"
                      "         to a routine that evaluates the objective,\n"
                      "         and another pointer to a Hessian of\n"
                      "         of a quadratic. PASA assumes that\n"
                      "         the function is a quadratic.\n\n") ;
        }
    }
    if ( !QP && (pasadata->c != NULL) && (pasadata->value != NULL) )
    {
        if ( PrintLevel )
        {
            printf ("\nWarning: There is an ambiguity in the objective\n"
                      "         since the user provides both a pointer\n"
                      "         to a routine to evaluate the objective,\n"
                      "         and another pointer to a linear cost\n"
                      "         vector. PASA assumes the objective is\n"
                      "         linear.\n\n") ;
        }
    }

    /* check that each problem type has the necessary data */
    if ( UNC || BNC || NL )
    {
        /* both the objective and its gradient need to be provided */
        if ( (pasadata->value == NULL) || (pasadata->grad == NULL) )
        {
            status = PASA_FUNCTION_VALUE_OR_GRADIENT_MISSING ;
            pasa_wrapup (status, TRUE, &Com) ;
            return (status) ;
        }
    }

    if ( NL || QP || LP || PROJ )
    {
        /* if a sparse matrix is present, then all 3 arrays are needed */
        if ( (userAp != NULL) || (userAi != NULL) || (userAx != NULL) )
        {
            if ( (userAp == NULL) || (userAi == NULL) || (userAx == NULL) )
            {
                status = PASA_MATRIX_INCOMPLETE ;
                pasa_wrapup (status, TRUE, &Com) ;
                return (status) ;
            }
        }

        /* check if the matrix is given, but not the right/left side */
        if ( (userBl == NULL) && (userBu == NULL) && (userAp != NULL) )
        {
            status = PASA_MATRIX_GIVEN_BUT_RHS_MISSING ;
            pasa_wrapup (status, TRUE, &Com) ;
            return (status) ;
        }

        /* check if the right/left side are given, but not the matrix */
        if ( ((userBl != NULL) || (userBu != NULL)) && (userAp == NULL) )
        {
            status = PASA_RHS_GIVEN_BUT_MATRIX_MISSING ;
            pasa_wrapup (status, TRUE, &Com) ;
            return (status) ;
        }

        /* if the problem is a QP and it has constraints, then hprod
           must be given (as opposed to cg_hprod which is specific to cg) */
        if ( QP )
        {
            /* check if the problem has constraints */
            if ( (userAp != NULL) || (usera != NULL) || (userlo != NULL) ||
                  (userhi != NULL) )
            {
                /* the problem has constraints so either hprod or the Hessian
                   should be given */
                if ( (pasadata->hprod == NULL) && (pasadata->Hp == NULL) )
                {
                    status = PASA_MISSING_HESSIAN_FOR_QP ;
                    pasa_wrapup (status, TRUE, &Com) ;
                    return (status) ;
                }
                if      ( pasadata->Hp    != NULL ) Com.hprod_status = 1 ;
                else if ( pasadata->hprod != NULL ) Com.hprod_status = 0 ;
                else                                Com.hprod_status = 2 ;
            }
            else /* the problem has no constraints (unconstrained) */
            {
                if ( (pasadata->cg_hprod == NULL) && (pasadata->Hp == NULL) )
                {
                    status = PASA_MISSING_HESSIAN_FOR_CG_QP ;
                    pasa_wrapup (status, TRUE, &Com) ;
                    return (status) ;
                }
                if      ( pasadata->Hp       != NULL ) Com.cg_hprod_status = 1 ;
                else if ( pasadata->cg_hprod != NULL ) Com.cg_hprod_status = 0 ;
                else                                   Com.cg_hprod_status = 2 ;
            }
            Com.QPshift = pasaparm->QPshift ;
        }
    }

    /* if nrow is EMPTY (not specified), then try to determine its value */
    if ( nrow == EMPTY )
    {
        if      ( pasadata->a != NULL )
        {
            nrow = 1 ;
        }
        else if ( userAp != NULL )
        {
            nrow = 1 + pasa_supi (userAi, userAp[ncol]);
        }
        /* else problem either unconstrained or bound constrained */
        else
        {
            nrow = 0 ;
        }
    }

    /* if there is a dual multiplier, but it is NULL, then allocate it,
       and return it to the user in pasadata->lambda. In pasadata, we
       also store a pointer in lambda_created to any allocated array.
       pasa_terminate will free an array that was allocated in pasa. */
    if ( (nrow > 0) && (lambda == NULL) )
    {
        if ( pasaparm->use_lambda == TRUE )
        {
            status = PASA_LAMBDA_IS_NULL_BUT_USE_LAMBDA_IS_TRUE ;
            pasa_wrapup (status, TRUE, &Com) ;
            return (status) ;
        }
        lambda = pasadata->lambda = pasadata->lambda_created =
                   (PASAFLOAT *) pasa_malloc (&status, nrow, sizeof (PPFLOAT)) ;
        if ( status )
        {
            status = PASA_OUT_OF_MEMORY ;
            pasa_wrapup (status, TRUE, &Com) ;
            return (status) ;
        }
        /* only first element set to zero in case napheap needs it */
        *lambda = PASAZERO ;
    }

    /* If the user did not allocate memory for the solution, then allocate
       memory. If the problem also requires a starting guess, then set
       the guess to zero. */
    if ( x == NULL )
    {
        x = pasadata->x = pasadata->x_created =
            (PASAFLOAT *) pasa_malloc (&status, ncol, sF) ;
        if ( status )
        {
            status = PASA_OUT_OF_MEMORY ;
            pasa_wrapup (status, TRUE, &Com) ;
            return (status) ;
        }
        /* if the problem requires a starting guess, then set it to zero */
        if ( UNC || BNC || QP || NL )
        {
            pasa_initx (x, PASAZERO, ncol) ;
            if ( PrintLevel )
            {
                printf ("Warning: This problem class requires a starting\n"
                        "         guess for the solution. Since no starting\n"
                        "         guess was given, we use zero.\n") ;
            }
        }
    }
    Com.x = x ; /* user's guess (later: returned solution) */

    /* directly call napheap if the problem has the right structure */
    if ( NAPSACK || ((nrow == 1) &&
         ((userd != NULL) || ((userc != NULL) && !QP))) )
    {
        napdata->n = ncol ;
        napdata->x = x ;
        napdata->d = userd ;
        if ( usera == NULL ) /* matrix input in sparse format */
        {
            napdata->a = (PASAFLOAT *) pasa_malloc (&status, ncol, sF) ;
            if ( status )
            {
                status = PASA_OUT_OF_MEMORY ;
                pasa_wrapup (status, TRUE, &Com) ;
                return (status) ;
            }
            pasa_initx (napdata->a, PASAZERO, ncol) ;
            p = 0 ;
            for (j = 0; j < ncol; j++)
            {
                q = userAp [j+1] ;
                if ( p < q )
                {
                    napdata->a [j] = userAx [p] ;
                    p = q ;
                }
            }
        }
        else napdata->a = usera ;

        napdata->lo = userlo ;
        napdata->hi = userhi ;

        if ( userc != NULL )
        {
            napdata->c = (PASAFLOAT *) pasa_malloc (&status, ncol, sF) ;
            if ( status )
            {
                status = PASA_OUT_OF_MEMORY ;
                pasa_wrapup (status, TRUE, &Com) ;
                return (status) ;
            }
            /* the linear cost vector in napheap has a negative sign */
            pasa_scale (napdata->c, pasadata->c, -PASAONE, ncol) ;
        }
        else napdata->c = NULL ;

        if ( userBl == NULL )
        {
             napdata->blo = -PASAINF ;
        }
        else
        {
             napdata->blo = *userBl ;
        }

        if ( userBu == NULL )
        {
             napdata->bhi = PASAINF ;
        }
        else
        {
             napdata->bhi = *userBu ;
        }

        /* if the user does not provide a guess for lambda, but
           use_lambda is TRUE, then use pasa's lambda
           (note that if lambda was not given in pasa and it was created,
           then *lambda = 0 which is ignored in napheap) */
        if ( (napdata->lambda == PASAINF) && pasaparm->use_lambda )
        {
            napdata->lambda = -*lambda ; /* napheap uses opposite sign mult */
        }

        if ( PrintParm )
        {
            napheap_print_parm (napdata) ;
        }

        /* call napheap */
        status = napheap (napdata) ;

        /* compute the objective value */
        s = t = PASAZERO ;
        int const cExists = (userc == NULL) ? FALSE : TRUE ;
        if ( userd != NULL ) /* quadratic objective */
        {
            for (j = 0; j < ncol; j++)
            {
                PASAFLOAT const xj = x [j] ;
                t += xj*xj*pasadata->d [j] ;
                if ( cExists ) s += xj*pasadata->c [j] ;
            }
            pasastat->f = .5*t + s ;
        }
        else /* linear objective */
        {
            for (j = 0; j < ncol; j++)
            {
                s += x [j]*pasadata->c [j] ;
            }
            pasastat->f = s ;
        }

        /* solution computed to machine precision accuracy */
        pasastat->err = PASAZERO ;

        /* print the error status if requested */
        pasa_wrapup (status, TRUE, &Com) ;

        /* if user wants statistics, print them */
        Stats->use_napheap = TRUE ;
        if ( pasaparm->PrintStat == TRUE ) pasa_print_stats (pasadata) ;

        /* return the multiplier to pasa, flip sign for consistency with pasa */
        pasadata->lambda [0] = -napdata->lambda ;

        /* free y if it was created */
        if ( userc != NULL ) pasa_free (napdata->c) ;

        /* free a if it was created */
        if ( usera == NULL ) pasa_free (napdata->a) ;

        return (status) ;
    }

    /* initialize pasa statistics */
    pasastat->mcnf  = 0 ;    /* function evaluations in main code */
    pasastat->mcng  = 0 ;    /* gradient evaluations in main code */
    pasastat->gpit  = 0 ;    /* number of iterations in grad_proj */
    pasastat->gpnf  = 0 ;    /* function evaluations in grad_proj */
    pasastat->gpng  = 0 ;    /* gradient evaluations in grad_proj */
    pasastat->agpit = 0 ;    /* number of iterations in active set grad_proj */
    pasastat->agpnf = 0 ;    /* function evaluations in active set grad_proj */
    pasastat->agpng = 0 ;    /* gradient evaluations in active set grad_proj */
    cgstat->iter    = 0 ;    /* number of iterations in cg_descent */
    cgstat->nfunc   = 0 ;    /* function evaluations in cg_descent */
    cgstat->ngrad   = 0 ;    /* gradient evaluations in cg_descent */
    cgstat->IterSub = 0 ;    /* number of iterations inside subspace */
    cgstat->NumSub  = 0 ;    /* number of subspaces */
    pasastat->nproject = 0 ; /* number of projections performed */
    pasastat->err = PASAZERO;/* error in pasa initialized to 0 */
    /* Due to rounding errors and errors in the projection, the
       gradient project search direction may not be a descent direction.
       The variable nbad_dir counts the number of bad search directions;
       that is, the number of times that the computed search direction fails
       to be a descent direction. */
    pasastat->nbad_dir = 0 ;

    /* directly call cg_descent if the problem is unconstrained */
    if ( UNC || (QP && (userAp == NULL) && (usera  == NULL) &&
                       (userlo == NULL) && (userhi == NULL)) )
    {
        Com.use_cg = Stats->use_cg = TRUE ;
        /* save original version of cg parameters */
        Com.userparms.c_grad_tol    = cgparm->grad_tol ;
        Com.userparms.c_QPshift     = cgparm->QPshift ;
        Com.userparms.c_PrintStatus = cgparm->PrintStatus ;
        Com.userparms.c_PrintStat   = cgparm->PrintStat ;
        Com.userparms.c_PrintParm   = cgparm->PrintParm ;

        cgparm->PrintStatus = FALSE ;
        cgparm->grad_tol = grad_tol ;
        cgparm->QPshift  = pasaparm->QPshift ;
        cgdata->x        = x ;
        cgdata->n        = ncol ;
        cgdata->value    = pasadata->value ;
        cgdata->grad     = pasadata->grad ;
        cgdata->valgrad  = pasadata->valgrad ;
        cgdata->hess     = pasadata->cghess ;
        cgdata->hprod    = pasadata->cg_hprod ;
        cgdata->c        = pasadata->c ;
        cgdata->Hp       = pasadata->Hp ;
        cgdata->Hi       = pasadata->Hi ;
        cgdata->Hx       = pasadata->Hx ;
        if ( PrintParm == TRUE )
        {
            cg_print_parm (cgdata) ;
        }

        status = cg_descent (cgdata) ;

        pasastat->f   = cgstat->f ;
        pasastat->err = cgstat->err ;
        pasa_wrapup (status, TRUE, &Com) ;

        /* if user wants statistics, print them */
        Stats->use_cg = TRUE ;
        if ( pasaparm->PrintStat == TRUE )
        {
            cg_print_stat (cgdata) ;
        }
        return (status) ;
    }
 
    status = PASA_OK ;
    /* check if userBl or userBu need to be malloc'd and initialized */
    if ( nrow > 0 )
    {
        if ( userBl == NULL )
        {
            userBl = (PASAFLOAT *) pasa_malloc (&status, nrow, sF) ;
            Com.temp_userBl = userBl ;
            pasa_initx (userBl, -PASAINF, nrow) ;
        }
        if ( userBu == NULL )
        {
            userBu = (PASAFLOAT *) pasa_malloc (&status, nrow, sF) ;
            Com.temp_userBu = userBu ;
            pasa_initx (userBu, PASAINF, nrow) ;
        }
        if ( status == SOPT_OUT_OF_MEMORY )
        {
            status = PASA_OUT_OF_MEMORY ;
            pasa_wrapup (status, TRUE, &Com) ;
            return (status) ;
        }
    }
    else
    {
        userBl = userBu = NULL ;
    }

#ifndef NDEBUG
    /* if debugging is used, save the user arrays in Com structure */
    Com.BL = userBl ;
    Com.BU = userBu ;
    Com.AX = userAx ;
    Com.AI = userAi ;
    Com.AP = userAp ;
    Com.LO = userlo ;
    Com.HI = userhi ;
#endif

    ni = 0 ;     /* number of strict inequalities */
    nsing = 0 ;  /* number of column singletons */

    if ( PrintLevel ) printf ("\n") ;
    if ( (nrow == 0) || ((userAp == NULL) && (usera == NULL)) )
    {
        Aexists = FALSE ; /* projection is simply trunction */
        Asize = 0 ;
        nrow = 0 ;
        use_napheap = FALSE ;
        use_pproj = FALSE ;
    }
    else
    {
        Aexists = TRUE ; /* either sparse matrix or dense "a" vector */
        if ( (userAp != NULL) && (usera != NULL) )
        {
            status = PASA_BOTH_A_AND_A_EXIST ;
            pasa_wrapup (status, TRUE, &Com) ;
            return (status) ;
        }
        i = (nrow == 1) && (pasaparm->use_napheap == TRUE) ;
        use_napheap = (i) ? TRUE  : FALSE ;
        use_pproj   = (i) ? FALSE : TRUE ;
        if ( userAp != NULL ) Asize = userAp [ncol] ; /* nnz in A */
        else                  Asize = ncol ;          /* dense a vector */
        if ( PrintLevel )
        {
            printf ("input matrix has %ld rows/ %ld cols/ %ld nz\n",
                    (LONG) nrow, (LONG) ncol, (LONG) Asize) ;
        }
    }
    Com.Aexists = Aexists ;
    Com.use_restoration = use_restoration ;
    Stats->use_napheap = Com.use_napheap = use_napheap ;
    Stats->use_pproj   = Com.use_pproj   = use_pproj ;
    Stats->use_cg      = Com.use_cg      = FALSE ;
    loExists = (userlo == NULL) ? FALSE : pasaparm->loExists;
    hiExists = (userhi == NULL) ? FALSE : pasaparm->hiExists ;
    int const user_loExists = loExists ;
    int const user_hiExists = hiExists ;
    int const ScaleRows =
                     ((pasaparm->ScaleRows && use_pproj) || LP) ? TRUE : FALSE ;
    Com.ScaleRows = ScaleRows ;

    /* The user's parameters may be changed during the operation of pasa.
       At the conclusion of the run, the user's original parameters are
       restored. Here we save a copy of parameters that may be changed
       by pasa. */
    if ( use_pproj )
    {
        Com.userparms.p_PrintStatus     = pprojparm->PrintStatus ;
        Com.userparms.p_grad_tol        = pprojparm->grad_tol ;
        Com.userparms.p_return_data     = pprojparm->return_data ;
        Com.userparms.p_use_prior_data  = pprojparm->use_prior_data ;
        Com.userparms.p_loExists        = pprojparm->loExists ;
        Com.userparms.p_hiExists        = pprojparm->hiExists ;
        Com.userparms.p_getfactor       = pprojparm->getfactor ;
        Com.userparms.p_permute         = pprojparm->permute ;
        Com.userparms.p_start_guess     = pprojparm->start_guess ;
        Com.userparms.p_use_startup     = pprojparm->use_startup ;
        Com.userparms.p_LP              = pprojparm->LP ;
        Com.userparms.p_check_if_y_feas = pprojparm->check_if_y_feas ;
    }
    else if ( use_napheap )
    {
        Com.userparms.n_PrintStatus    = napparm->PrintStatus ;
        Com.userparms.n_return_data    = napparm->return_data ;
        Com.userparms.n_use_prior_data = napparm->use_prior_data ;
        Com.userparms.n_loExists       = napparm->loExists ;
        Com.userparms.n_hiExists       = napparm->hiExists ;
        Com.userparms.n_d_is_one       = napparm->d_is_one ;
    }

    nfree = ncol ;
    Afixed = 0 ; /* space in A associated with fixed variables */
    /* If both lo and hi exist, then the problem could have fixed variables.
       Check for fixed variables, and if it is found that lo/hi are
       -/+ infinity, then change the corresponding parameters
       loExists/hiExists to FALSE. */
    int boundchk = FALSE ;
    if ( (loExists == TRUE) && (hiExists == TRUE) )
    {
        boundchk = TRUE ;
        ifree = (PASAINT *) pasa_malloc (&status, ncol, sizeof(PASAINT)) ;
        Com.temp_ifree = ifree ;
        /* terminate if there is not enough memory */
        if ( status )
        {
            status = PASA_OUT_OF_MEMORY ;
            pasa_wrapup (status, TRUE, &Com) ;
            return (status) ;
        }
        loExists = FALSE ;
        hiExists = FALSE ;
        nfree = 0 ; /* counts number of free variables */
        for (j = 0; j < ucol; j++)
        {
            t = userhi [j] - userlo [j] ;
            if ( t < PASAZERO )
            {
                pasastat->lobad = userlo [j] ;
                pasastat->hibad = userhi [j] ;
                pasastat->ibad  = j ;
                status = PASA_INVALID_VARIABLE_BOUNDS ;
                pasa_wrapup (status, TRUE, &Com) ;
                return (status) ;
            }
            else if ( t > PASAZERO )
            {
                if ( LP )
                {
                    k = userAp [j+1] - userAp [j] ;
                    if ( k <= 1 ) /* column singleton or zero column */
                    {
                        /* ncol is the number of free columns */
                        ncol-- ;
                        /* save the fixed variables at the end of ifree */
                        ifree [ncol] = j ;
                        if ( k == 1 )
                        {
                            /* removing this variable will remove some space
                               space from the matrix, update both the space
                               removed and the number of column singletons */
                            Afixed++ ;
                            nsing++ ; /* this variable is a column singleton */
                        }
                        /* else when k = 0, the variable can be removed from
                           the problem, but no space is removed */
                    }
                    else /* free variable and more than 1 entry in column */
                    {
                        if ( userhi [j] < PASAINF )
                        {
                            hiExists = TRUE ;
                        }
                        if ( userlo [j] > -PASAINF )
                        {
                            loExists = TRUE ;
                        }
                        ifree [nfree] = j ;
                        nfree++ ;
                    }
                }
                else /* free variable, problem not an LP */
                {
                    if ( userhi [j] < PASAINF )
                    {
                        hiExists = TRUE ;
                    }
                    if ( userlo [j] > -PASAINF )
                    {
                        loExists = TRUE ;
                    }
                    ifree [nfree] = j ;
                    nfree++ ;
                }
            }
            else /* t = 0 or nan, a fixed variable should be removed */
            {
                if ( t != t ) /* infinite bounds */
                {
                    pasastat->lobad = userlo [j] ;
                    pasastat->hibad = userhi [j] ;
                    pasastat->ibad  = j ;
                    status = PASA_INVALID_VARIABLE_BOUNDS ;
                    pasa_wrapup (status, TRUE, &Com) ;
                    return (status) ;
                }
                /* ncol is the number of free columns */
                ncol-- ;
                /* save the fixed variables at the end of ifree */
                if ( LP ) ifree [ncol] = -(j+1) ;
                else      ifree [ncol] = j ;
                /* evaluate the space in A associated with fixed variables */
                if ( Aexists )
                {
                    Afixed += userAp [j+1] - userAp [j] ;
                }
            }
        }
    }

    /* even if bounds were not checked, an LP could have deleted columns
       due to column singletons */
    if ( LP && !boundchk )
    {
        nfree = 0 ; /* counts number of free variables */
        ifree = (PASAINT *) pasa_malloc (&status, ncol, sizeof(PASAINT)) ;
        Com.temp_ifree = ifree ;
        for (j = 0; j < ucol; j++)
        {
            if ( (k = (userAp [j+1] - userAp [j])) <= 1 )
            {
                /* ncol is the number of free columns */
                ncol-- ;
                /* save the fixed variables at the end of ifree */
                ifree [ncol] = j ;
                if ( k == 1 )
                {
                    /* count the number of column singletons */
                    nsing++ ;
                }
            }
            else
            {
                ifree [nfree] = j ;
                nfree++ ;
            }
        }
        Afixed = nsing ;
    }
    Asize -= Afixed ;

    int const fixed = (ncol < ucol) ;
    if ( !fixed && Com.temp_ifree != NULL )
    {
        pasa_free (ifree) ;
        Com.temp_ifree = NULL ;
    }

    int const singExists = (nsing > 0) ; /* TRUE if column singletons exist */

    if ( Aexists )
    {
        /* for an LP we may need to allocate some temporary variables */
        if ( singExists )
        {
            /* list of rows where the inequality is strict */
            ineqList = (PASAINT *) pasa_malloc (&status, 2*nrow+1, sI) ;

            if ( status )
            {
                status = PASA_OUT_OF_MEMORY ;
                pasa_wrapup (status, TRUE, &Com) ;
                return (status) ;
            }
            /* total number of column singletons in each row */
            singcount = ineqList+nrow ;

            pasa_initi (singcount, (PASAINT) 0, nrow+1) ;
            singcount = singcount+1 ;
            /* singcount later converted to an array that points to start of
               each row with singletons. After using this pointer array to
               reorder the cost elements associated with each singleton,
               we set singcount = singcount-1 so that the pointer array again
               points to the start of each row. */
        }

        /* count the number of strict inequalities */
        for (i = 0; i < nrow; i++)
        {
            if ( (UBl = userBl [i]) < (UBu = userBu [i]) )
            {
                if ( singExists )
                {
                    ineqList [ni] = i ;
                    /* since the inequality will be converted to an equality
                       if there are column singletons, add the strict
                       inequalities into the singcount array */
                    singcount [i]++ ;
                }
                ni++ ;
            }
            else if ( (UBl > UBu) || (UBl >= PASAINF) )
            {
                pasastat->lobad = UBl ;
                pasastat->hibad = UBu ;
                pasastat->ibad  = i ;
                status = PASA_INVALID_LINEAR_CONSTRAINT_BOUNDS ;
                pasa_wrapup (status, TRUE, &Com);
                return (status) ;
            }
        }
    }

    /* for an LP, strict inequalities are converted to equalities
       when column singletons are present */
    const PASAINT nsingni = nsing + ni ;
    Com.nsing = nsing ;
    Com.ni = ni ;

    if ( PrintLevel >= 1 )
    {
        printf ("%ld free variables/ %ld fixed variables",
                (LONG) nfree, (LONG) ucol - ncol - nsing) ;
        if ( LP )
        {
            printf ("/ %ld column singletons", (LONG) nsing) ;
        }
        printf ("/ %ld total variables\n", (LONG) ucol) ;
        printf ("%ld strict inequalities/ %ld equalities\n",
                (LONG) ni, (LONG) nrow - ni) ;
        if ( (Aexists == TRUE) && Afixed )
        {
            printf ("removing fixed variables") ;
            if ( LP ) printf (" & column singletons") ;
            printf (" deletes %ld of %ld matrix elements\n",
                   (LONG) Afixed, (LONG) userAp [ucol]) ;
        }
    }

    Com.fixed = fixed ;
    Com.loExists = loExists ;
    Com.hiExists = hiExists ;
    if ( (loExists == TRUE) || (hiExists == TRUE) )
    {
        Bounds = TRUE ;
    }
    else
    {
        Bounds = FALSE ;
    }
    Com.Bounds = Bounds ;
    if ( PrintLevel >= 1 )
    {
        printf ("Aexists: %c/ loExists: %c/ hiExists: %c/ Bounds: %c/ "
                "use_napheap: %c/ use_pproj: %c\n",
                 FT[Aexists], FT[loExists], FT[hiExists], FT[Bounds],
                 FT[use_napheap], FT[use_pproj]) ;
        if ( !singExists ) printf ("\n") ;
    }

    /* if GradProjOnly is FALSE and the objective is not linear,
       then cg_descent will be used so set up the parm and stat structures */
    if ( (GradProjOnly == FALSE) && (LP == FALSE) )
    {
        /* setup the cgdata structure */
        cgdata->value   = pasadata->value ;
        cgdata->grad    = pasadata->grad ;
        cgdata->valgrad = pasadata->valgrad ;

        /* note that cgdata->c, cgdata->hprod, and cgdata->hess are not
           used when the problem has constraints since the evaluation
           routines will use the information from pasadata */
        cgdata->c     = NULL ;
        cgdata->hprod = NULL ;
        cgdata->hess  = NULL ;

        /* save cg parameters that may be changed */
        Com.userparms.c_grad_tol    = cgparm->grad_tol ;
        Com.userparms.c_QPshift     = cgparm->QPshift ;
        Com.userparms.c_PrintStatus = cgparm->PrintStatus ;
        Com.userparms.c_PrintStat   = cgparm->PrintStat ;
        Com.userparms.c_PrintParm   = cgparm->PrintParm ;

        /* set the cg parameters */
        cgparm->PrintStatus = FALSE ;
        cgparm->PrintStat = FALSE ;
        cgparm->PrintParm = FALSE ;

        use_penalty = (Aexists) ? pasaparm->use_penalty : FALSE ;
        Com.use_penalty = use_penalty ;

        if ( PrintParm == TRUE )
        {
            cg_print_parm (cgdata) ;
        }
    }

    /* If there are linear equalities/inequalities, then either napheap
       or pproj will be used. */
    if ( Aexists == TRUE )
    {
        /* if the matrix has only one row, use napheap if possible */
        if ( use_napheap == TRUE )
        {
            /* some parameters for napheap were set above, now set
               the remaining parameters */
            napparm->PrintStatus = FALSE ;
            napparm->d_is_one = TRUE ;
            napparm->return_data = TRUE ;
            napparm->loExists = loExists ;
            napparm->hiExists = hiExists ;

            if ( PrintParm )
            {
                napheap_print_parm (napdata) ;
            }
        }
        else /* use pproj */
        {
#ifndef NOPPROJ
            /* set pprojparm parameter's to match the values in pasa */
            pprojparm->PrintStatus = FALSE ;
            pprojparm->loExists = loExists ;
            pprojparm->hiExists = hiExists ;
            pprojparm->check_if_y_feas = TRUE ;

            if ( !use_cholmod && !LP && !GradProjOnly )
            {
                status = PASA_MUST_USE_CHOLMOD ;
                pasa_wrapup (status, TRUE, &Com) ;
                return (status) ;
            }

            if ( PrintParm )
            {
                pproj_print_parm (ppdata) ;
            }
#endif
        }
    }

    /* In PASA, variables associated with active constraints are removed
       from the problem and in PPROJ, we reorder unknowns to reduce fill-in.
       Consequently, when we evaluate the user's objective function, we
       need to reorder variables. Two different orderings are involved.
       If there are fixed variables or if Aexists (and PPROJ is employed),
       but there are no bound constraints, then we reorder using a
       permutation called colperm. If there are bounds on the variables
       (either lo and hi) and the active set methods are used (either
       activeGP or cg_descent), then we reorder with a permutation
       called ifree. The array ifree is dynamic in the sense that as
       variables reach a bound, they are removed from ifree and it shrinks
       in size, while colperm is static since there are no bounds. When
       deciding which permutation to use, we first check if order_with_ifree
       is TRUE, in which case we use it; if it is FALSE, and
       order_with_colperm is TRUE, then we use colperm. If they are both
       FALSE, then no reordering is needed. */
    Com.order_with_colperm = FALSE ;
    Com.order_with_ifree = FALSE ;
    if ( Bounds && !GradProjOnly )
    {
        Com.order_with_ifree = TRUE ; /* if location = AGP or CG */
    }
    if ( fixed || (use_pproj && use_cholmod) )
    {
        Com.order_with_colperm = TRUE ;
        /* used if location = GP or
                  (location = AGP or CG and order_with_ifree is FALSE) */
    }
    if ( Com.order_with_ifree || Com.order_with_colperm )
    {
        Com.need_userx = TRUE ;
    }
    else
    {
        Com.need_userx = FALSE ;
    }

    if ( PrintLevel )
    {
        printf ("order with ifree: ") ;
        pasa_print_TF (Com.order_with_ifree) ;
        printf ("order with colperm: ") ;
        pasa_print_TF (Com.order_with_colperm) ;
    }

    /* Allocate PASAFLOAT work arrays.
       Note that the value of ncol provided by the user has been reduced
       to take into account any fixed variables. ucol denotes the
       original version of ncol specified by the user's input argument.
       pasaparm->M space is needed for storing the M values of the objective
       function at the M most recent iterates.  6*ncol space needed for
       the following arrays. The userx and userg arrays are due the
       the following: When polyhedral constraints are present, the
       rows and columns of A are reordered to help improve the sparsity
       in the linear algebra. Throughout the code, we work in this
       permuted coordinate system. Hence, when the user's function or
       gradient is evaluated, we need to permute to the user's coordinate
       system. Also, in the active set method, variables can be fixed
       at bounds. Currently, these fixed variables are removed from the
       problem. When the user's function or gradient is evaluated, we only
       copy the unfixed variable values into the user's coordinates.

       xnew   = x + alpha*d
       gnew   = gradient evaluated at xnew
       g      = gradient evaluated at x
       d      = search direction
       step_g = point whose projection yields d (= -stepsize*gk)
       userx  = x in the user's coordinate system
       userg  = g in the user's coordinate system

       For a quadratic objective, we also need space to store:

       Qd   = Hessian times d

       The space allocation for CG is the same as the space allocation
       in the cg_descent code except that 4*ncol space is shared with
       the main pasa code.  The routine pproj for projecting
       a point onto a polyhedron handles its own memory internally.  */

    /* If use_pproj is true, then the problem has a general linear
       constraint.  If GradProjOnly is false, then an active set
       method will be used. If both PROJ and LP are false, then the
       objective is not linear or a square, it must be a general nonlinear
       objective that will be optimized by an active set method. In this
       case, copies of Ap, Ai, and Ax are needed since they are compressed
       as bound constraints become active. */
    int ActiveSetGeneralA = (Aexists && !PROJ && !LP && !GradProjOnly &&Bounds);

    /* figure out space for float arrays and allocate a block of memory */
    if ( xWork == NULL )
    {
        i = 0 ;

        /* Com->d */
        i += ncol ;

        /* if bounds, fixed, or perm, also need userx and userg */
        if ( Com.need_userx )
        {
            i += 2*ucol ; /* userx and userg */
        }

        if ( use_pproj ) /* Com->step_g */
        {
            i += ncol ;
        }

        /* for linear cost problems, need extra copy of the linear term */
        if ( LP )
        {
            /* store a copy of c in Com.c since it will be scaled */
            i += ncol ;

            /* allocate space for ColScale and offset */
            i += 2*ncol ;

            if ( singExists )
            {
                /* space for sx (matrix element for singleton)
                             sc (cost element)
                            slo (lower bound)
                            shi (upper bound) */
                i += 4*nsingni ;

                /* singlo and singhi are sorted versions of slo and shi */
                i += 2*nsingni ;

                /* singc is the sorted version of sc */
                i += nsingni ;
            }
        }
        else /* LP is FALSE */
        {
            /* Com.xnew, Com.gnew, Com.g, nonmonotone line search memory */
            i += 3*ncol + pasaparm->M ;
        }
        if ( use_napheap )
        {
            /* 3*ncol for xwork array and ncol for nap_a */
            i += 4*ncol ;

            
            /* save a copy of a if bounds are present and an
               active set method will be used */
            if ( Bounds && !GradProjOnly )
            {
                i += ncol ; /* nap_acopy */
                /* if use_restoration, need space to store nap_xold and
                   nap_dx = xnew - xold */
                if ( use_restoration ) i += 2*ncol ;
            }

            /* when using napheap with fixed variables, need to save
               a copy of a before removing the fixed variables */
            if ( fixed )
            {
                i += ucol ; /* nap_auser */
            }
        }

        /* if lambda == NULL, then allocate space for lambda array */
        if ( lambda == NULL )
        {
            i += nrow ;
        }

        /* Need an extra copy of lo, hi, bl, bu, A in the following cases:
           1. fixed variables are present, so some lo and hi's removed
           2. the problem is not a projection problem (PROJ), but it is
              an active set method -- as bounds become active, variable are
              removed from the problem
           3. for an LP, the problem is rescaled
           4. bounds are present and an active set method is used */
        if ( fixed || LP || ActiveSetGeneralA
                         || (Bounds && !GradProjOnly && !PROJ) )
        {
            if ( loExists )
            {
                i += ncol ;
            }
            if ( hiExists )
            {
                i += ncol ;
            }

            /* Copies of bl and bu (of size nrow) are needed to account
               for modifications of the right side due to the fixed
               variables, compressions in a active set method, and scalings
               in an LP. Also, need a copy of pproj's Prob->b and the A matrix.
               Note that Prob->bl and Prob->bu have size ni+1 <= nrow +1. */
            if ( use_pproj ) /* the inequalities bl <= A*x <= bu exist */
            {
                i += 2*(nrow+1) + nrow + Asize ;
            }
        }
        else if ( use_pproj && !PROJ )
        {
            /* Allocate space for Com.bl, Com.bu, and Com.b since these
               are used in the computation of Prob->bl, Prob->bu, and Prob->b
               in the projection routines of pasa, and in the initial call
               to pproj.  Also, need space for a copy of Ax if the rows are
               scaled. Note that if there is no scaling, then userBl and
               userBu can be used for the initial call to pproj. */
            if ( ScaleRows ) i += 3*(nrow+1) + Asize ;
            else             i += 2*(ni+1) ;
        }

        /* Allocate the space needed for CG. If GradProjOnly is TRUE or
           the cost is linear, then CG is not performed and this space is
           not needed. */
        if ( !GradProjOnly && !LP )
        {
            /* in CG, memory is the minimum of ncol and LBFGSmemory */
            mem = PASAMIN (cgparm->LBFGSmemory, ncol) ;

            /* if penalty term is used, then allocate space for it */
            if ( use_penalty == TRUE )
            {
                /* cgcom-> lambda_pen, gpen, AAd, in CG */
                i += 2*ncol + nrow ;
            }

            if ( Aexists == TRUE )
            {
                /* need 3*ncol for projected gradients at x and x + alpha*d,
                   and for the search direction before the final projection */
                i += 3*ncol ;
            }

            /* if L-BFGS is potentially used, then allocate space for it */
            if ( cgparm->LBFGS >= 1 )
            {
                i += 2*mem*(ncol+1) ;
            }

            /* if limited memory CG used, need additional space */
            if ( cgparm->LBFGS == 3 )
            {
                i += (mem+2)*ncol + (3*mem+9)*mem + 5 ;
            }

            if ( pasaparm->use_QR )
            {
                i += ncol*(nrow + PASAMIN (nrow, ncol)) + nrow ;
            }

            /* Need extra space for the expansion steps if the problem is
               purely bound constrained. */
            if ( !Aexists )
            {
                if ( QP ) i += 4*ncol ;
                else      i += ncol ;
            }
        }

        if ( Aexists ) /* A exists */
        {
            /* active set method */
            if ( (GradProjOnly == FALSE) && (LP == FALSE) )
            {
                /* Com->copylo, Com->copyhi used in gradproj initialization */
                if ( loExists == TRUE )
                {
                    i += ncol ;
                }
                if ( hiExists == TRUE )
                {
                    i += ncol ;
                }
                if ( ni > 0 )
                {
                    /* Com->Axk, Adk are used in CG */
                    i += 2*(ni+1) ;
                }
                if ( use_pproj )
                {
                    /* variables used in gradproj initialization */
                    i += nrow + 2*(ni+1) ; /* Copy->b, bl, bu */
                    i += 2*Asize ;         /* Copy->ATx, Ax */
                }
            }
            if ( ScaleRows ) /* space for RowScale */
            {
                i += nrow ;
            }
        }
        /* bound constraints only, if active set method, allocate copy arrays */
        else if ( GradProjOnly == FALSE )
        {
            /* Com->copylo, Com->copyhi */
            if ( loExists )
            {
                i += ncol ;
            }
            if ( hiExists )
            {
                i += ncol ;
            }
        }

        if ( QP == TRUE )
        {
            /* we need space to store the total gradient in user coordinate,
               as well as space for the Hessian times search direction */
            i += 2*ncol ; /* Com->Qd, gtot */
        }
        work = (PASAFLOAT *) pasa_malloc (&status, i, sF) ;
        Com.work = work ; /* work in Com structure for later free */
    }
    else
    {
        work = xWork ;
        Com.work = NULL ;
    }

    /* terminate if there is not enough memory */
    if ( status == SOPT_OUT_OF_MEMORY )
    {
        status = PASA_OUT_OF_MEMORY ;
        pasa_wrapup (status, TRUE, &Com) ;
        return (status) ;
    }

    /* Allocate space for integer arrays. */
    if ( iWork == NULL )
    {
        i = 0 ;

        if ( LP )
        {
            /* row si and column sj numbers for column singletons */
            i += 2*nsingni ;

            /* singperm (permutation from sorted singc to sc) */
            i += nsingni ;

            /* row_sing [[row_sing[i], row_sing[i+1]) =singc indices for row i*/
            i += nrow + 1 ;
        }

        if ( use_napheap )
        {
            /* iwork array of napheap */
            i += 4*ncol + 2 ;
        }
        /* If fixed variables are present, then save space for Ai and Ap
           if they exist. If the problem is not a projection and an
           active set method is used, then also need space for Ai and Ap
           due to potential compressions as bounds become active or
           inequalities drop.  Otherwise, utilize userAi and userAp */
        if ( (use_pproj && fixed) || ActiveSetGeneralA )
        {
            /* space for Ap and Ai */
            i += ncol + 1 + Asize ;
        }

        /* in the following cases, we need the permutation colperm
           between the internal ordering of variables and the user ordering */
        if ( (fixed || use_pproj) && !LP )
        {
            i += ncol ;
        }

        if ( !GradProjOnly && !LP ) /* active set method */
        {
            /* Com->ifree, bound_cols */
            if ( Bounds )
            {
                i += 2*ncol ;
            }
            /* bound_rows */
            i += ni ;

            if ( use_pproj == TRUE )
            {
                /* Copy->ineq_row, Copy->sol_to_blk, Copy->ATp,
                   Com->row_to_ineq, Com->invperm, Copy->Ap,
                   Copy->Ai, Copy->ATi */
                i += 2*(ni + 2) + ucol + ncol + 2*(nrow + 1 + Asize) ;
            }
        }

        iwork = (PASAINT *) pasa_malloc (&status, i, sI) ;
        Com.iwork = iwork ; /* iwork in Com structure for later free */
    }
    else
    {
        iwork = iWork ;
        Com.iwork = NULL ;
    }

    /* terminate if there is not enough memory */
    if ( status == SOPT_OUT_OF_MEMORY )
    {
        status = PASA_OUT_OF_MEMORY ;
        pasa_wrapup (status, TRUE, &Com) ;
        return (status) ;
    }

    /* -------- REST OF COMMON BLOCK ----------*/
    Com.grad_tol = (LP) ? pasaparm->LPgrad_tol : pasaparm->grad_tol ;
    Com.value   = pasadata->value ;   /* objective function */
    Com.grad    = pasadata->grad ;    /* gradient of objective function */
    Com.valgrad = pasadata->valgrad ; /* combined function and gradient*/
    Com.Hp      = pasadata->Hp ;      /* column pointers for QP Hessian */
    Com.Hi      = pasadata->Hi ;      /* row indices for QP Hessian */
    Com.Hx      = pasadata->Hx ;      /* numerical values in QP Hessian */
    Com.hprod   = pasadata->hprod ;   /* routine to evaluate Hessian-vector
                                         product for a QP */
    Com.hess    = pasadata->hess ;    /* evaluate Hessian at a given point */
    Com.GPtol = pasaparm->GPtol ;     /* error tolerance for projected grad. */
    Com.nrow = nrow ;    /* number of rows in A */
    Com.ncol = ncol ;    /* number of component in x (cols in A if A exists) */
    Com.userc = userc ;  /* user's linear term in the objective */
    /* initialize factor for switching between GPA and active set methods */
    Com.switchfactor = pasaparm->switchfactor ;
    /* line search could change from Armijo to approximate Armijo, so store
       the parameter value in Com where it can be changed */
    Com.approxstep = pasaparm->approxstep ;
    Com.nf = ncol ;
    Com.nr = 0 ;  /* no bound rows by default */
    Com.nc = 0 ;  /* no bound cols by default */
    Com.location = PASA_BASE_CODE ; /* other locations: PASA_GRAD_PROJ
                                                        PASA_ACTIVE_BP
                                                        PASA_CG_DESCENT */
    Com.unitstep = FALSE ;  /* used in activeGP */
    Com.factor_ok = FALSE ; /* used in activeGP */
    Com.first = TRUE ;      /* used in project, denotes first iteration */

    /* initialize error and function value to infinity */
    Com.Egp  = PASAINF ;
    Com.E1   = PASAINF ;
    Com.E    = PASAINF ;
    Com.fnew = PASAINF ;

    /* ======== FLOAT WORKSPACE ===========================================*/
    Com.d = work ;  work += ncol ;

    if ( Com.need_userx )
    {
        Com.userx = work ;  work += ucol ; /* note: ucol not ncol */
        Com.userg = work ;  work += ucol ; /* note: ucol not ncol */
    }

    if ( use_pproj )
    {
        Com.step_g = work ;  work += ncol ;
    }

    /* set up the structure to store the Hessian */
    if ( Com.need_userx ) Hess.x = Com.userx ;
    else                  Hess.x = x ;
    Hess.Hp = NULL ;
    Hess.Hi = NULL ;
    Hess.Hx = NULL ;
    Hess.ncol = ucol ;
    /* for linear cost problems, need extra copy of the linear term */

    if ( LP )
    {
        /* store a copy of c in Com.c since it will be scaled */
        Com.c = work ;  work += ncol ;

        Com.ColScale = work ;  work += ncol ;
        Com.offset   = work ;  work += ncol ;

        /* if column singleton exist, allocate for slo, shi, sx, sc */
        if ( singExists )
        {
            sx     = Com.sx     = work ;  work += nsingni ;
            sc     = Com.sc     = work ;  work += nsingni ;
            slo    = Com.slo    = work ;  work += nsingni ;
            shi    = Com.shi    = work ;  work += nsingni ;
            singlo = Com.singlo = work ;  work += nsingni ;
            singhi = Com.singhi = work ;  work += nsingni ;
            singc  = Com.singc  = work ;  work += nsingni ;
        }
    }
    else /* the problem is not an LP */
    {
        /* xnew, Com->g, gnew, nonmonotone line search memory */
        Com.xnew  = work ;  work += ncol ;
        Com.gnew  = work ;  work += ncol ;
        Com.g     = work ;  work += ncol ;
        Com.fmem  = work ;  work += pasaparm->M ;
    }

    if ( use_napheap )
    {
        Com.napxwork = work ;  work += 3*ncol ;
        Com.nap_a    = work ;  work +=   ncol ;
        /* save a copy of a if bounds are present and an
           active set method will be used */
        if ( Bounds && !GradProjOnly )
        {
            Com.nap_acopy  = work ;  work += ncol ;
            if ( use_restoration )
            {
                Com.nap_xold = work ;  work += ncol ;
                Com.nap_dx   = work ;  work += ncol ;
            }
        }
        else
        {
            Com.nap_acopy  = NULL ;
        }
        if ( fixed )
        {
            Com.nap_auser  = work ;  work += ucol ;
        }
        else
        {
            Com.nap_auser  = Com.nap_acopy ;
        }
    }
    else
    {
        Com.napxwork = NULL ;
        Com.nap_a    = NULL ;
        Com.nap_acopy = NULL ;
    }

    if ( lambda == NULL )
    {
        lambda = work ;  work += nrow ;
    }
    Com.lambda = lambda ;

    /* initialize lo, hi, Bu, Bl, Ap, Ai, Ax to the user values */

    lo = (loExists == TRUE) ? userlo : NULL ;
    hi = (hiExists == TRUE) ? userhi : NULL ;
    Bu = userBu ;
    Bl = userBl ;
    Ax = userAx ;
    Ap = userAp ;
    Ai = userAi ;

    /* Need an extra copy of lo and hi in the following cases:
       1. fixed variables are present, so some lo and hi's removed
       2. the problem is not a projection problem (PROJ), but it is
          an active set method -- as bounds become active, variables are
          removed from the problem
       3. for an LP, the problem is rescaled */
    if ( fixed || LP || ActiveSetGeneralA || (Bounds && !GradProjOnly && !PROJ))
    {
        if ( loExists == TRUE )
        {
            lo = work ;  work += ncol ;
        }
        else
        {
            lo = NULL ;
        }
        if ( hiExists == TRUE )
        {
            hi = work ;  work += ncol ;
        }
        else
        {
            hi = NULL ;
        }

        /* Copies of bl and bu (of size nrow) are needed to account
           for modifications of the right side due to the fixed
           variables, compressions in a active set method, and scalings
           in an LP. Also, need a copy of pproj's Prob->b and the A matrix.
           Note that Prob->bl and Prob->bu have size ni+1 <= nrow +1. */
        if ( use_pproj )
        {
            /* wasteful since only need ni+1 later for bl/bu, which is
               typically less than nrow */
            b  = work ;  work += nrow ;
            Bl = work ;  work += (nrow+1) ;
            Bu = work ;  work += (nrow+1) ;
            Ax = work ;  work += Asize ;
        }
    }
    else if ( use_pproj && !PROJ )
    {
        /* allocate space for Com.bl, Com.bu, and Com.b since these
           are used in the computation of Prob->bl, Prob->bu, and Prob->b
           when computing a projection */
        b  = work ;  work += nrow ;
        /* when rows are scaled, linear equality bounds used 2 ways:
           input to pproj during initial projection and later during
           each projection */ 
        if ( ScaleRows )
        {
            Bl = work ;  work += nrow + 1 ;
            Bu = work ;  work += nrow + 1 ;
            Ax = work ;  work += Asize ;
        }
        else /* no scaling, input to pproj during initial projection is the
                user arrays */
        {
            Bl = work ;  work += ni + 1 ;
            Bu = work ;  work += ni + 1 ;
        }
    }

    /* if an active set algorithm used and the objective is not linear,
       then set up cg memory */
    if ( !GradProjOnly && !LP )
    {
        /* This is where cg memory starts. */
        Com.cgwork = cgwork = work ;

        /* if use_penalty is TRUE, allocate space for the multiplier used
           in penalty function */
        if ( use_penalty == TRUE )
        {
            Com.lambda_pen = work ;  work += nrow ;
            Com.gpen       = work ;  work += ncol ;
            Com.cgwork = cgwork = work ;
            /* jump over space for AAd */
            work += ncol ;
        }

        if ( Aexists == TRUE )
        {
            /* jump over space for projected gradients at x and x + alpha*d,
               and for the search direction before the final projection */
            work += 3*ncol ;
        }

        /* if L-BFGS is potentially used, then jump over its space */
        if ( cgparm->LBFGS >= 1 )
        {
            work += 2*mem*(ncol+1) ;
        }

        /* if limited memory CG used, then jump over its space */
        if ( cgparm->LBFGS == 3 )
        {
            work += (mem+2)*ncol + (3*mem+9)*mem + 5 ;
        }

        /* if a QR factorization is done, jump over the space for it */
        if ( pasaparm->use_QR )
        {
            work += ncol*(nrow + PASAMIN (nrow, ncol)) + nrow ;
        }

        /* Need extra space for the expansion steps if the problem is
           purely bound constrained. */
        if ( !Aexists )
        {
            Com.cgexpand = work ;
            if ( QP ) work += 4*ncol ;
            else      work +=   ncol ;
        }
    }

    /* Com.lo and hi are set to the value computed above; however, if lo
       and hi did not get new value above, then Com.lo = userlo and
       Com.hi = userhi */
    Com.lo = lo ;
    Com.hi = hi ;

    /* copylo and and copy hi are initialized to the value given above */
    Com.copylo = lo ;
    Com.copyhi = hi ;

    Com.Ax = Ax ; /* nonzero numerical entries in A */
    Com.b = b ;
    Com.bl = Bl ; /* left side of linear inequalities */
    Com.bu = Bu ; /* right side of linear inequalities */

    /* if A exists, then allocate space for copy arrays and arrays used in CG */
    if ( Aexists == TRUE )
    {
        if ( (GradProjOnly == FALSE) && (LP == FALSE) )
        {
            if ( loExists == TRUE )
            {
                Com.copylo = work ;  work += ncol ;
            }
            if ( hiExists == TRUE )
            {
                Com.copyhi = work ;  work += ncol ;
            }
            if ( ni > 0 )
            {
                Com.Axk = work ;  work += ni + 1 ;
                Com.Adk = work ;  work += ni + 1 ;
            }
            if ( use_pproj == TRUE )
            {
                Copy.b   = work ;  work += nrow ;
                Copy.bl  = work ;  work += ni + 1 ;
                Copy.bu  = work ;  work += ni + 1 ;
                Copy.ATx = work ;  work += Asize ;
                Copy.Ax  = work ;  work += Asize ;
            }
        }
        if ( ScaleRows )
        {
            Com.RowScale = work ;  work += nrow ;
        }
    }
    /* else bound constraints only, if active set method, allocate copy arrays*/
    else if ( GradProjOnly == FALSE )
    {
        if ( loExists )
        {
            Com.copylo = work ;  work += ncol ;
        }
        if ( hiExists )
        {
            Com.copyhi = work ;  work += ncol ;
        }
    }

    if ( QP == TRUE )
    {
        /* Com->Qd (Hessian times direction) */
        Com.Qd   = work ;  work += ncol ;
        /* Com->gtot total gradient in user's coordinates with fixed variables
                removed.  gtot differs from userg in that userg is gradient
                at the current x, which may not be accepted, while
                gtot is the gradient at an accepted iterate x. We only
                update gtot when the iterate is accepted.  */
        Com.gtot = work ;  work += ncol ;
    }

    /* ======== INTEGER WORKSPACE =========================================*/

    if ( singExists )
    {
        si       = Com.si       = iwork ;  iwork += nsingni ;
        sj       = Com.sj       = iwork ;  iwork += nsingni ;
        singperm = Com.singperm = iwork ;  iwork += nsingni ;
        row_sing = Com.row_sing = iwork ;  iwork += nrow + 1 ;
    }

    if ( use_napheap )
    {
        Com.napiwork = iwork ;  iwork += 4*ncol + 2 ;
    }
    else
    {
        Com.napiwork = NULL ;
    }

    /* If fixed variables are present, then save space for Ai and Ap
       if they exist. If the problem is not a projection and an
       active set method is used, then also need space for Ai and Ap
       due to potential compressions as bounds become active or
       inequalities drop.  Otherwise, employ space in userAi and userAp */
    if ( (use_pproj && fixed) || ActiveSetGeneralA )
    {
        Ap = iwork ;  iwork += ncol + 1 ;
        Ai = iwork ;  iwork += Asize ;
    }
    if ( (fixed || (use_pproj && use_cholmod)) && !LP )
    {
        Com.colperm = iwork ;  iwork += ncol ;
    }
    else
    {
        Com.colperm = NULL ;
    }

    Com.ifree = NULL ;
    if ( !GradProjOnly && !LP ) /* active set method */
    {
        /* ifree, bound_cols, bound_rows */
        if ( Bounds )
        {
            Com.ifree      = iwork ;  iwork += ncol ;
            Com.bound_cols = iwork ;  iwork += ncol ;
        }
        Com.bound_rows = iwork ;  iwork += ni ; /* ni could vanish */
        if ( use_pproj )
        {
            /* Copy->ineq_row, Copy->ATp, Com->row_to_ineq, Com->invperm,
                   Copy->Ap, Copy->Ai, Copy->ATi */
            Copy.ineq_row   = iwork ;  iwork += ni + 2 ;

            /* use int for consistency with pproj (waste when iwork is long) */
            Copy.sol_to_blk = (int *) iwork ;  iwork += ni + 2 ;

            Copy.ATp        = iwork ;  iwork += nrow + 1 ;
            Com.row_to_ineq = iwork ;  iwork += nrow ;
            Com.invperm     = iwork ;  iwork += ucol ;
            Copy.Ap         = iwork ;  iwork += ncol + 1 ;
            Copy.ATi        = iwork ;  iwork += Asize ;
            Copy.Ai         = iwork ;  iwork += Asize ;
        }
    }

    /* Ap and Ai are either userAp and userAi if no fixed variables,
       or they were allocated space above if there are fixed variables
       and the matrix is compressed */
    Com.Ap = Ap ; /* column pointers for A */
    Com.Ai = Ai ; /* row indices for A */

    /* =========== ALLOCATION OF MEMORY IS COMPLETE =========== */

    if ( use_pproj )
    {
        pprojparm->grad_tol = pasaparm->GPtol ;
        ppdata->lambda      = lambda ;
        ppdata->Parm        = pprojparm ;
        ppdata->Stat        = pprojstat ;
        ppdata->nrow        = nrow ;
        ppdata->ncol        = ncol ;
        ppdata->ni          = ni ;
        ppdata->nsing       = (singExists) ? nsingni : 0 ;
        ppdata->Ap          = Ap ;
        ppdata->Ai          = Ai ;
        ppdata->Ax          = Ax ;
        ppdata->lo          = lo ;
        ppdata->hi          = hi ;
        if ( fixed || LP || ScaleRows )
        {
            ppdata->bl      = Bl ;
            ppdata->bu      = Bu ;
        }
        else
        {
            ppdata->bl      = userBl ;
            ppdata->bu      = userBu ;
        }
        if ( singExists )
        {
            ppdata->row_sing = row_sing ;
            ppdata->singlo   = singlo ;
            ppdata->singhi   = singhi ;
            ppdata->singc    = singc ;
        }
        else
        {
            ppdata->row_sing = NULL ;
            ppdata->singlo   = NULL ;
            ppdata->singhi   = NULL ;
            ppdata->singc    = NULL ;
         }
        /* still to assign: x, y (these input are chosen in different ways
                                  throughout the code) */
    }

    ns = 0 ; /* singleton number */
    /* Remove any fixed variables from the problem. Moreover, if Aexists
       and the rows are to be scaled, then compute the absolute largest
       element in the row. */
    if ( ScaleRows )
    {
        RowScale = Com.RowScale ;
        pasa_initx (RowScale, PASAZERO, nrow) ;
    }
    if ( fixed )
    {
        /* store the unfixed parts of userlo, userhi, and x in lo, hi, and x */
        for (j = 0; j < ncol; j++)
        {
            i = ifree [j] ;
            /* userx is returned to user with problem solution, so use it */
            x [j] = x [i] ; /* bound variables stored in userx */
            if ( loExists ) lo [j] = userlo [i] ;
            if ( hiExists ) hi [j] = userhi [i] ;
            if ( LP )  Com.c [j] = userc [i] ;
        }

        /* if A exists, place the fixed variables in userx, and adjust Bl, Bu,
           and A to account for the fixed variables */
        if ( use_pproj )
        {
            PASAINT ColNew, ColOld, LocNew, LocOld ;
            pasa_copyx (Bl, userBl, nrow) ;
            pasa_copyx (Bu, userBu, nrow) ;
            Ap [0] = 0 ;
            ColOld  = 0 ; /* column in old A */
            ColNew  = 0 ; /* column in new A */
            LocNew  = 0 ; /* location in new Ai or Ax */
            LocOld  = 0 ; /* location in old Ai or Ax */

            for (i = ucol-1; i >= ncol; i--)
            {
                j = ifree [i] ; /* index in user's coordinates */
                if ( LP )
                {
                    if ( j >= 0 ) /* either column singleton or zero column */
                    {
                        /* Although column singletons were put in the fixed
                           variable list, they are not fixed.  Similar to
                           fixed variables, the columns with singletons will
                           be removed from the problem (not discarded but
                           placed in a separate structure) */
                        PASAINT Apj = userAp [j] ;
                        k = userAp [j+1] - Apj ;
                        if ( k == 1 )
                        {
                            if ( user_loExists ) slo [ns] = userlo [j] ;
                            else                 slo [ns] = -PASAINF ;
                            if ( user_hiExists ) shi [ns] = userhi [j] ;
                            else                 shi [ns] = PASAINF ;
                            Com.sc [ns] = userc [j] ;    /* cost */
                            const PASAINT row = userAi [Apj] ;
                            si [ns] = row ;          /* row of singleton */
                            singcount [row]++ ;      /* count # in row */
                            sj [ns] = j ;            /* col of singleton */
                            sx [ns] = userAx [Apj] ; /* numerical entry */
                            ns++ ;
                            t = PASAZERO ; /* nothing to subtract from b */
                        }
                        else /* k = 0, the column is zero, determine value */
                        {
                            if ( userc [j] > PASAZERO )
                            {
                                if ( loExists ) t = userlo [j] ;
                                else            t = -PASAINF ;
                            }
                            else if ( userc [j] < PASAZERO )
                            {
                                if ( hiExists ) t = userhi [j] ;
                                else            t = PASAINF ;
                            }
                            else /* c [j] = 0, take closest number to 0 */
                            {
                                if ( userlo [j] > PASAZERO )
                                {
                                    t = userlo [j] ;
                                }
                                else
                                {
                                    if ( userhi [j] < PASAZERO )
                                    {
                                        t = userhi [j] ;
                                    }
                                    else /* 0 \in [userlo, userh] */
                                    {
                                        t = PASAZERO ;
                                    }
                                }
                            }
                        }
                    }
                    else /* variable is bound but not singleton or zero column*/
                    {
                        j = -(j+1) ; /* j converted to number >= 0 */
                        t = userlo [j] ; /* = userhi [j] */
                    }
                }
                else /* variable is bound */
                {
                    t = userlo [j] ; /* = userhi [j] */
                }
                Com.userx [j] = t ;
                /* copy column m up to column j-1 from userA into A */
                p = LocOld ;
                LocOld = userAp [j] ;
                if ( ScaleRows )
                {
                    pasa_scalecopy (RowScale, LocOld-p, Ax+LocNew, userAx+p,
                                                        Ai+LocNew, userAi+p) ;
                }
                else
                {
                    pasa_copyx (Ax+LocNew, userAx+p, LocOld-p) ;
                    pasa_copyi (Ai+LocNew, userAi+p, LocOld-p) ;
                }
                for (l = ColOld; l < j; l++)
                {
                    LocOld = userAp [l+1] ;
                    LocNew += LocOld - p ;
                    ColNew++ ;
                    Ap [ColNew] = LocNew ;
                    p = LocOld ;
                }
                ColOld = j + 1 ;

                /* subtract column j times fixed value t from right and
                   left side of inequality if t nonzero */
                LocOld = userAp [ColOld] ;
                if ( t )
                {
                    for (; p < LocOld; p++)
                    {
                        l = userAi [p] ;
                        s = t*userAx [p] ;
                        Bl [l] -= s ;
                        Bu [l] -= s ;
                    }
                }
            }

            /* copy from end of last fixed column to the end of A */
            p = LocOld ;
            LocOld = userAp [ucol] ;
            if ( ScaleRows )
            {
                pasa_scalecopy (RowScale, LocOld-p, Ax+LocNew, userAx+p,
                                                    Ai+LocNew, userAi+p) ;
            }
            else
            {
                pasa_copyx (Ax+LocNew, userAx+p, LocOld-p) ;
                pasa_copyi (Ai+LocNew, userAi+p, LocOld-p) ;
            }
            for (l = ColOld; l < ucol; l++)
            {
                LocOld = userAp [l+1] ;
                LocNew += LocOld - p ;
                ColNew++ ;
                Ap [ColNew] = LocNew ;
                p = LocOld ;
            }
        }
        /* Compression handled in separate block below for napheap
           Note that ScaleRows is FALSE for napheap. */
        else if ( !use_napheap ) /* no Ap and a are present */
        {
            for (i = ucol-1; i >= ncol; i--)
            {
                j = ifree [i] ;
                Com.userx [j] = userlo [j] ;
            }
        }
    }
    else if ( LP ) /* no fixed variables in LP */
    {
        /* save copy of c since it will be scaled (ucol = ncol) */
        pasa_copyx (Com.c, userc, ucol) ;

        /* save copy of the equation bounds since they will be scaled */
        pasa_copyx (Bl, userBl, nrow) ;
        pasa_copyx (Bu, userBu, nrow) ;
        if ( ScaleRows )
        {
            pasa_scalecopy (RowScale, Asize, Ax, userAx, userAi, userAi) ;
        }
        else
        {
            pasa_copyx (Ax, userAx, Asize) ; /* A is initially scaled below */
        }
        if ( loExists ) pasa_copyx (lo, userlo, ncol) ;
        if ( hiExists ) pasa_copyx (hi, userhi, ncol) ;
    }
    else if ( !PROJ && !GradProjOnly ) /* active set method */
    {
        if ( loExists ) pasa_copyx (lo, userlo, ncol) ;
        if ( hiExists ) pasa_copyx (hi, userhi, ncol) ;
        if ( use_pproj ) /* linear inequalities exist */
        {
            pasa_copyx (Bl, userBl, nrow) ;
            pasa_copyx (Bu, userBu, nrow) ;
            pasa_copyi (Ap, userAp, ncol+1) ;
            if ( ScaleRows )
            {
                pasa_scalecopy (RowScale, Asize, Ax, userAx, Ai, userAi) ;
            }
            else
            {
                pasa_copyx (Ax, userAx, Asize) ;
                pasa_copyi (Ai, userAi, Asize) ;
            }
        }
    }
    else if ( ScaleRows )
    {
        pasa_copyx (Bl, userBl, nrow) ;
        pasa_copyx (Bu, userBu, nrow) ;
        pasa_scalecopy (RowScale, Asize, Ax, userAx, userAi, userAi) ;
    }

    /* Determine the row scale factors so that they are powers of 2 and
       apply the scale factor to the equation bounds. LPs are treated
       differently from general optimization problem due to the handling
       of column singletons and the conversion of strict inequalities to
       equalities */
    if ( LP )
    {
        if ( singExists )
        {
            /* augment the singleton data with all strict inequalities */
            for (k = 0; k < ni; k++)
            {
                i = ineqList [k] ;
                slo [ns] = Bl [i] ;
                shi [ns] = Bu [i] ;
                Bl [i]  = PASAZERO ;
                sc [ns] = PASAZERO ; /* cost */
                si [ns] = i ;        /* row of singleton */
                sj [ns] = -1 ;       /* no column */
                sx [ns] = -PASAONE ; /* numerical entry */
                ns++ ;
            }
            ppdata->bu = Bl ; /* = ppdata->bl */
            /* strict inequalities converted to equalities when column
               singletons exist */
            ppdata->ni = 0 ;
            for (i = 0; i < nrow; i++)
            {
                if ( (t = RowScale [i]) != PASAZERO )
                {
                    /* square root decreases change in row */
                    t = frexp (sqrt (t), &exponent) ;
                    t = ldexp (1, -exponent) ;
                    Bl [i] *= t ;
                }
                else
                {
                    t = PASAONE ;
                }
                RowScale [i] = t ;
            }
        }
        else
        {
            for (i = 0; i < nrow; i++)
            {
                if ( (t = RowScale [i]) != PASAZERO )
                {
                    /* square root decreases change in row */
                    t = frexp (sqrt (t), &exponent) ;
                    t = ldexp (1, -exponent) ;
                    Bl [i] *= t ;
                    Bu [i] *= t ;
                }
                else
                {
                    t = PASAONE ;
                }
                RowScale [i] = t ;
            }
        }
    }
    else if ( ScaleRows ) /* not an LP */
    {
        for (i = 0; i < nrow; i++)
        {
            if ( (t = RowScale [i]) != PASAZERO )
            {
                /* square root decreases change in row */
                t = frexp (sqrt (t), &exponent) ;
                t = ldexp (1, -exponent) ;
                Bl [i] *= t ;
                Bu [i] *= t ;
            }
            else
            {
                t = PASAONE ;
            }
            RowScale [i] = t ;
        }
    }

    /* save constraint vector a if napheap is used, remove any fixed variables*/
    if ( use_napheap )
    {
        PASAINT jfixed, jnext ;
        /* if fixed variable are present, then save the first fixed index */
        if ( fixed )
        {
            jfixed = ucol - 1 ;
            jnext = ifree [jfixed] ;
        }
        else
        {
            jnext = ucol ;
        }
        k = 0 ;
        /* if usera not provided, then extract it from input sparse matrix */
        t = PASAZERO ;
        if ( usera == NULL )
        {
            p = 0 ;
            for (j = 0; j < ucol; j++)
            {
                q = userAp [j+1] ;
                if ( p < q )
                {
                    PASAFLOAT const aj = userAx [p] ;
                    if ( j < jnext ) /* j is not fixed */
                    {
                        Com.nap_a [k] = aj ;
                        if ( fixed ) Com.nap_auser [j] = aj ;
                        k++ ;
                        t += aj*aj ;
                        p = q ;
                    }
                    else /* the variable is fixed, removed it */
                    {
                        /* store fixed variable in userx */
                        Com.userx [j] = userlo [j];
                        /* subtract contribution from right side of constraint*/
                        if ( fixed ) Com.nap_auser [j] = aj ;
                        u = userlo [j]*aj ;
                        Bl [0] -= u ;
                        Bu [0] -= u ;
                        jfixed-- ;
                        if ( jfixed < ncol ) jnext = ucol ;
                        else                 jnext = ifree [jfixed] ;
                    }
                }
                else
                {
                    if ( fixed ) Com.nap_auser [j] = PASAZERO ;
                    Com.nap_a [k] = PASAZERO ;
                    k++ ;
                }
            }
        }
        else /* usera is given, remove any fixed variables if they exist*/
        {
            /* if there are fixed variables, also save a copy of a before the
               removal of the fixed variables */
            if ( fixed ) pasa_copyx (Com.nap_auser, usera, ucol) ;
            for (j = 0; j < ucol; j++)
            {
                PASAFLOAT const aj = usera [j] ;
                if ( j < jnext )
                {
                    Com.nap_a [k] = aj ;
                    k++ ;
                    t += aj*aj ;
                }
                else              /* remove this variable from problem */
                {
                    u = userlo [j]*aj ;
                    /* store fixed variable in userx */
                    Com.userx [j] = userlo [j];
                    /* subtract contribution from right side of constraint */
                    Bl [0] -= u ;
                    Bu [0] -= u ;
                    jfixed-- ;
                    if ( jfixed < ncol ) jnext = ucol ;
                    else                 jnext = ifree [jfixed] ;
                }
            }
        }
        /* the following pointers in napdata do not change in the algorithm */
        napdata->a     = Com.nap_a ;
        napdata->lo    = lo ;
        napdata->hi    = hi ;
        napdata->xWork = Com.napxwork ;
        napdata->iWork = Com.napiwork ;
        /* By default, the multiplier is in Com.nap_lambda. If the multiplier
           is not supposed to be used for the starting guess, then set it
           to -infinity. Before napheap is called, we set napdata->lambda =
           -Com.nap_lambda, and napheap interprets an infinite value as
           something to be ignored. */
        if ( pasaparm->use_lambda == FALSE )
        {
            Com.nap_lambda = -PASAINF ;
        }
 
        Com.nap_a2 = Com.nap_a2save = Com.nap_a2full = t ;
        if ( Com.nap_acopy != NULL )
        {
            pasa_copyx (Com.nap_acopy, Com.nap_a, ncol) ;
        }
        Com.nap_bl = Com.nap_blcopy = *Bl ;
        Com.nap_bu = Com.nap_bucopy = *Bu ;
    }

    /* If the problem is an LP, then do the following:

       1. Multiply each row by a factor to make the largest element
          in the row around one (exclude all column singletons). This was
          done above for the right and left side of the linear inequalities.
       2. Multiply each column by a factor to make the Euclidean norm one.
       3. If the bound [lo [j], hi [j]] does not contain 0, then make
          make a translation in x [j] so that 0 lies in the transformed
          bound interval.
       4. Multiply the right side bl and bu by a scalar so that the
          right side vector has absolute largest element 1.
       5. Multiply the cost vector c by a scalar so that the absolute
          largest element is 1. */
    if ( LP )
    {

        /* scale matrix rows and adjust the bound such that 0 \in [lo, hi] */
        PASAFLOAT *offset = Com.offset ;
        c = Com.c ;
        p = 0 ;
        PASAFLOAT normc = PASAZERO ;
        for (j = 0; j < ncol; j++)
        {
            if      ( loExists && (lo [j] > PASAZERO) ) t = lo [j] ;
            else if ( hiExists && (hi [j] < PASAZERO) ) t = hi [j] ;
            else                                        t = PASAZERO ;
            offset [j] = t ;
            q = Ap [j+1] ;
            p0 = p ;
            s = PASAZERO ;
            if ( singExists ) /* constraint is equality, only Bl used */
            {
                for (; p < q; p++)
                {
                    i = Ai [p] ;
                    r = RowScale [i]*Ax [p] ;
                    Ax [p] = r ;
                    Bl [i] -= r*t ;
                    s += r*r ; /* sum of squares of elements in the column */
                }
            }
            else
            {
                for (; p < q; p++)
                {
                    i = Ai [p] ;
                    r = RowScale [i]*Ax [p] ;
                    Ax [p] = r ;
                    Bl [i] -= r*t ;
                    Bu [i] -= r*t ;
                    s += r*r ; /* sum of squares of elements in the column */
                }
            }
            s = frexp (sqrt (s), &exponent) ;
            s = ldexp (1, -exponent) ;
            for (p = p0; p < q; p++)
            {
                Ax [p] *= s ;
            }
            u = PASAONE/s ;
            if ( hiExists ) hi [j] = u*(hi [j] - t) ;
            if ( loExists ) lo [j] = u*(lo [j] - t) ;
            Com.ColScale [j] = s ;
            c [j] *= s ;
            if ( fabs (c [j]) > normc ) normc = fabs (c [j]) ;
            x [j] *= u ;
        }

        /* if there are column singletons, then create the associated
           data structures */
        PASAFLOAT normb ;
        normb = PASAZERO ;
        if ( singExists )
        {
            /* make singcount point to start of storage for singletons in row */
            k = 0 ;
            nr = 0 ;
            rowList = ineqList ; /* do not need ineqList, use it for rowList */
            for (i = 0; i < nrow; i++)
            {
                l = k ;
                if ( singcount [i] )
                {
                    k += singcount [i] ;
                    rowList [nr] = i ;   /* store the rows with singletons */
                    nr++ ;
                }
                singcount [i] = l ; /* cummulative number of singletons */
            }

            /* allocate space for local variables */
            PASAINT *itemp = (PASAINT *) pasa_malloc (&status, 3*nsingni, sI) ;
            if ( status )
            {
                status = PASA_OUT_OF_MEMORY ;
                pasa_wrapup (status, TRUE, &Com) ;
                return (status) ;
            }
                     iwork = itemp ;  itemp += nsingni ;
            PASAINT *tempk = itemp ;  itemp += nsingni ;
            PASAINT *order = itemp ;  itemp += nsingni ;

            PASAFLOAT *tempc = (PASAFLOAT *) pasa_malloc (&status, nsingni,sF);
            if ( status )
            {
                status = PASA_OUT_OF_MEMORY ;
                pasa_free (iwork) ; /* also frees tempk and order */
                pasa_wrapup (status, TRUE, &Com) ;
                return (status) ;
            }
            for (k = 0; k < ns; k++)
            {
                const PASAINT row = Com.si [k] ;
                l = singcount [row]++ ;
                const PASAFLOAT ax = RowScale [row]*sx [k] ;
                sx [k] = ax ;
                tempc [l] = sc [k]/ax ; /* tempc contains c/ax vector by rows */
                tempk [l] = k ;         /* index of tempc [l] in sc array */
            }
            /* Since singcount [i] is now the end of each row, we set
               singcount = singcount-1 to switch back to start of each row
               Arrange the c vector so that the entries are increasing for
               each row of the matrix. Output the corresponding arrays of
               singleton lower and upper bound, along with the permutation
               that maps back to the original data stored in Com. */
            l = -1 ; /* start of the elements in c is l + 1 */
            PASAINT lastrow = -1 ;
            singcount = singcount-1 ;
            PASAINT rownum ;
            for (rownum = 0; rownum < nr; rownum++)
            {
                const PASAINT row = rowList [rownum] ;
                for (i = lastrow+1; i <= row; i++)
                {
                    row_sing [i] = l + 1 ;
                }
                lastrow = row ;
                PASAINT     start = singcount [row] ;
                PASAINT   *kstart = tempk+start ; /* point from tempc to sc */
                PASAFLOAT *cstart = tempc+start ; /* start of c values for row*/
                /* num_in_row = number of singletons in row */
                const PASAINT num_in_row = singcount [row+1] - start ;

                pasa_minsortx (order, cstart, iwork, num_in_row) ;
                PASAFLOAT last_c_value = PASAINF ;
                for (i = 0; i < num_in_row; i++)
                {
                    /* index associated with the i-th largest singleton */
                    PASAINT const orderi = order [i] ;
                    k = kstart [orderi] ;
                    PASAFLOAT const newcost = cstart [orderi] ;

                    /* perform shift to achieve 0 \in [lok, hik] */
                    const PASAFLOAT lok = slo [k] ;
                    const PASAFLOAT hik = shi [k] ;

                    /* t = shift */
                    if      ( lok > PASAZERO ) t = lok ;
                    else if ( hik < PASAZERO ) t = hik ;
                    else                       t = PASAZERO ;

                    /* ax = matrix element * RowScale for singleton */
                    const PASAFLOAT ax = sx [k] ;

                    /* adjust Bl due to shift in x */
                    Bl [row] -= t*ax ;

                    if ( newcost == last_c_value ) /* c value repeats */
                    {
                        /* flag the current column if the c value repeats
                           since all the repeated values of c are compressed
                           into a single entry, as well as lo and hi */
                        if ( !repeat )
                        {
                            singperm [l] = -(singperm [l]+1) ;
                        }
                        repeat = TRUE ;
                        /* si used to store the row numbers, now use it
                           to form a linked list that points from last si
                           to new k value, empty marks end of list */
                        Com.si [kold] = k ;
                        kold = k ;
                        Com.si [k] = EMPTY ;
                        /* The bounds are added to the running totals.
                           Note that in pproj, xj is replaced by -xj.  */
                        if ( ax > PASAZERO ) /* sign change flips bounds */
                        {
                            singlo [l] += (t - hik)*ax ;
                            singhi [l] += (t - lok)*ax ;
                        }
                        else
                        {
                            singlo [l] += (t - lok)*ax ;
                            singhi [l] += (t - hik)*ax ;
                        }
                    }
                    else
                    {
                        repeat = FALSE ; /* new distinct c value */
                        last_c_value = newcost ;
                        l++ ; /* store a new c value and bounds */
                        singperm [l] = k ;
                        kold = k ;
                        /* Note that in pproj, xj is replaced by -xj.  */
                        if ( ax > PASAZERO ) /* scaling flips bounds */
                        {
                            singlo [l] = (t - hik)*ax ;
                            singhi [l] = (t - lok)*ax ;
                        }
                        else
                        {
                            singlo [l] = (t - lok)*ax ;
                            singhi [l] = (t - hik)*ax ;
                        }
                        singc [l] = newcost ;
                        if ( fabs (newcost) > normc ) normc = fabs (newcost) ;
                    }
                }
            }
            /* the last one column singleton is in location l so the
               number of column singletons is l + 1 */
            l++ ;
            for (i = lastrow+1; i <= nrow; i++) row_sing [i] = l ;
            ppdata->nsing = l ;
            if ( PrintLevel )
            {
                printf ("%ld column singletons and strict inequalities "
                        "after compression\n\n", (LONG) l) ;
            }
            pasa_free (iwork) ; /* also frees tempk and order */
            pasa_free (tempc) ;
            pasa_free (ineqList) ; /* also free singcount */

            for (i = 0; i < nrow; i++)
            {
                const PASAFLOAT Bli = Bl [i] ;
                if ( fabs (Bli) > normb )
                {
                    normb = fabs (Bli) ;
                    if ( Bli >= PASAINF )
                    {
                        pasastat->lobad = PASAZERO ;
                        pasastat->hibad = PASAZERO ;
                        pasastat->ibad  = i ;
                        status = PASA_INVALID_MATRIX_ELEMENT ;
                        pasa_wrapup (status, TRUE, &Com) ;
                        return (status) ;
                    }
                }
            }
        }
        else /* no column singletons, compute normb = max min right side */
        {
            for (i = 0; i < nrow; i++)
            {
                PASAFLOAT const fabsBl = fabs (Bl [i]) ;
                PASAFLOAT const fabsBu = fabs (Bu [i]) ;
                if ( fabsBl <= fabsBu )
                {
                    if ( fabsBl > normb ) normb = fabsBl ;
                }
                else
                {
                    if ( fabsBu > normb ) normb = fabsBu ;
                }
            }
        }
#if 0
sopt_printxMATLAB (singlo, ppdata->nsing, "singlo1") ;
sopt_printxMATLAB (singhi, ppdata->nsing, "singhi1") ;
sopt_printxMATLAB (singc, ppdata->nsing, "singc1") ;
sopt_printxMATLAB (Bl, nrow, "b_pasa1") ;
sopt_printxMATLAB (c, ncol, "c_pasa1") ;
sopt_printxMATLAB (lo, ncol, "lo_pasa1") ;
sopt_printxMATLAB (hi, ncol, "hi_pasa1") ;
exit(1) ;
#endif
        if ( normb == PASAZERO ) normb = PASAONE ;
        t = frexp (normb, &exponent) ;
        Com.scale_x = ldexp (1, -exponent) ;

        if ( normc == PASAZERO ) normc = PASAONE ;
        t = frexp (normc, &exponent) ;
        Com.scale_l = ldexp (1, -exponent) ;
        if ( PrintLevel >= 1 )
        {
            printf ("initial scale_x: %e scale_l: %e\n",
                     Com.scale_x, Com.scale_l) ;
        }
        /* problem solved by gradprojLP so both pasa and pproj statistics
           will be reported */
        Stats->use_pasa = TRUE ;

        status = pasa_gradprojLP (&Com) ;

        if ( status == PPROJ_OUT_OF_MEMORY ) pasa_wrapup (status, TRUE, &Com) ;
        else                                 pasa_wrapup (status, FALSE, &Com) ;

        return (status) ;
    }

    /* apply RowScale to the rows of A and to the dual multiplier if a
       starting guess is given */
    if ( ScaleRows )
    {
        for (j = 0; j < Asize; j++) Ax [j] *= RowScale [Ai [j]] ;
        if ( pasaparm->use_lambda == TRUE )
        {
            for (i = 0; i < nrow; i++) lambda [i] /= RowScale [i] ;
        }
    }

    /* For quadratic objectives, we need to convert between the user
       ordering of variables, and the ordering used in pproj. When bound
       constraints exist, then the ordering is stored in ifree. If there
       are no bound constraints, but either pproj is used or
       there are fixed variables, then the ordering is based on colperm.
       Otherwise, no reordering is needed. */

    Com.order = NULL ;
    if ( Com.order_with_ifree == TRUE )
    {
        Com.order = Com.ifree ;
    }
    else if ( Com.order_with_colperm == TRUE )
    {
        Com.order = Com.colperm ;
    }
#if 0
sopt_printxMATLAB (Bl, nrow, "bl1") ;
sopt_printxMATLAB (Bu, nrow, "bu1") ;
sopt_printxMATLAB (userc, ncol, "c1") ;
sopt_printxMATLAB (lo, ncol, "lo1") ;
sopt_printxMATLAB (hi, ncol, "hi1") ;
sopt_printAMATLAB (ncol, Ap, Ai, NULL, Ax, "A1") ;
exit(1) ;
#endif

#ifndef NDEBUG
    if ( use_pproj ) Com.ppdataDEBUG = ppdata ;
    else             Com.ppdataDEBUG = NULL ;
#endif

    /* If there are linear constraints, do not insist on computing the
       factorization when generating a feasible point since the initial
       point may be feasible. If only a projection is being performed,
       then do not touch the user's parameters */
    Com.initial_projection = TRUE ;
    if ( use_pproj && !PROJ )
    {
        pprojparm->getfactor = FALSE ;
        pprojparm->return_data = TRUE ;
        pprojparm->use_prior_data = FALSE ;
    }

    /* project d (starting guess) onto polyhedron */

    /* Project the starting guess on the polyhedron. If the starting guess
       is feasible, then this terminates quickly. */

    if ( PrintLevel > 0 )
    {
        printf ("Project starting point on polyhedron\n") ;
    }

    if ( PROJ ) /* projection problem, solve it and return to user */
    {
        status = pasa_project (x, pasadata->y, &Com) ;

        /* when fixed variables present, copy free variables into userx
           and then copy userx to the solution vector x */
        if ( fixed )
        {
            for (j = 0; j < ncol; j++)
            {
                Com.userx [ifree [j]] = x [j] ;
            }
            pasa_copyx (x, Com.userx, ucol) ;
        }

        /* pproj objective value not returned */
        pasastat->f = PASAINF ;
        pasastat->err = pprojstat->errdual ;
        pasa_wrapup (status, TRUE, &Com) ;

        /* print the statistics if requested */
        Stats->use_pproj = TRUE ;
        if ( pasaparm->PrintStat == TRUE )
        {
            pasa_print_stats (pasadata) ;
        }

        /* free memory in pasa */
        if ( Com.work != NULL )
        {
            pasa_free (Com.work) ;
        }
        if ( Com.iwork != NULL )
        {
            pasa_free (Com.iwork) ;
        }
        return (status) ;
    }
    else /* project starting guess on polyhedron */
    {
        /* if x is feasible, then return x in Com.d after running
           pproj_init */
        pprojparm->check_if_y_feas = 2 ;
        status = pasa_project (Com.d, x, &Com) ; /* projection in Com.d */
        /* in the active set methods, do not check for feasibility of
           projection point */
        pprojparm->check_if_y_feas = 0 ;
    }

    /* if an error occurred in pasa_project, then return */
    if ( status > 1 )
    {
        pasa_wrapup (status, TRUE, &Com) ;
        return (status) ;
    }

    /* not unconstrained or a knapsack problem or a projection problem */
    Stats->use_pasa = TRUE ;

    /* set the linear constraint status for napheap if it is used */
    if ( use_napheap == TRUE )
    {
        if ( *Bl == *Bu )
        {
            Com.nap_constraint = 2 ;
        }
        else if ( *(Com.lambda) > PASAZERO )   /* active inequality */
        {
            Com.nap_constraint = -1 ;          /* lower bound */
        }
        else if ( *(Com.lambda) == PASAZERO )  /* inactive inequality */
        {
            Com.nap_constraint = 0 ;
        }
        else /* lambda < 0 */                  /* active inequality */
        {
            Com.nap_constraint = 1 ;           /* upper bound */
        }
    }

    /* copy unfixed components (free) components of projection into userx */
    if ( fixed )
    {
        for (j = 0; j < ncol; j++)
        {
            Com.userx [ifree [j]] = Com.d [j] ;
        }
    }
    else
    {
        if ( Com.need_userx ) pasa_copyx (Com.userx, Com.d, ncol) ;
        else                  pasa_copyx (x, Com.d, ncol) ;
    }

    /* evaluate function and gradient at userx */
    t = PASAZERO ;
    pasa_evaluate (t, &t, &Com, "fg") ;

    if ( PrintLevel >= 1 )
    {
        printf ("starting function value: %25.15e\n", Com.f) ;
    }

    /* if starting function value is nan or infinite, then return */
    if ( (Com.f >= PASAINF) || (Com.f <= -PASAINF) || (Com.f != Com.f) )
    {
        status = PASA_STARTING_FUNCTION_VALUE_INFINITE_OR_NAN ;
        pasa_wrapup (status, TRUE, &Com) ;
        return (status) ;
    }

    if ( fixed )
    {
        for (j = 0; j < ncol; j++)
        {
            Com.g [j] = Com.userg [ifree [j]] ;
        }
    }
    else if ( Com.need_userx )
    {
        pasa_copyx (Com.g, Com.userg, ncol) ;
    }

    /* Store starting gradient in user's coordinates when the objective is
       quadratic. This will be updated internally in the code. */
    if ( QP == TRUE )
    {
        pasa_copyx (Com.gtot, Com.g, ncol) ;
    }

    /* When A exists, pproj is used to project onto the feasible set.
       The ordering of x in pproj is generally different from the user's
       ordering. Hence, the code employs pproj's ordering; when
       the user's function or gradient is evaluated, we need to convert
       to the user's ordering. Below, we convert x and g (the gradient)
       to pproj's ordering, and store data from pproj that is used to
       setup for the projection. Also, for an active set method, we need
       to store copies of pproj's data since it is needed during the
       initialization of the active set algorithm. As constraints become
       active or as variables become bound, this data is compressed. */
    if ( use_pproj ) /* A exists and pproj is used */
    {
        /* in pasa_project, the prior data structure of ppdata was stored in
           Com.ppcom */
        Prob = Com.ppcom->Prob ;

        /* Throughout the code, Com.x is ordered according to pproj's
           ordering of variables; thus, do not permute inside pproj */
        pprojparm->permute = FALSE ;

        /* Before projecting the user's starting guess on the polyhedron,
           we first checked if the guess was feasible, in which case no
           projection is needed.  Henceforth, feasibility is no longer
           checked since it is unlikely that the point generated by a
           step along the negative gradient is feasible. */
        pprojparm->check_if_y_feas = FALSE ;

        /* set parameters in pproj depending on whether cholmod is used */
        if ( use_cholmod )
        {
            /* When pproj is called by pasa, the default starting guess is
               based on the prior bound structure for a solution.
               If start_guess is changed to 0, then need to perform
               the updates to lstart, ustart, and sol_start found
               at the bottom of pproj_hotchol. */

            pprojparm->start_guess = 2 ;

            /* Since we employ the factorization in the error estimation,
               we do not allow pproj to terminate when the error tolerance is
               satisfied unless the matrix has been factored */
            pprojparm->getfactor = TRUE ;

            colperm = Prob->colperm ;
            /* copy the projection in Com.d to x */
            pasa_convert_to_pproj (x, Com.d, colperm, ncol) ;

            /* grab some temporary work space from pproj */
            temp = Com.ppcom->Work->arrayd ;
            /* copy the gradient to temp */
            pasa_copyx (temp, Com.g, ncol) ;
            /* overwrite the gradient by the gradient using pproj's ordering */
            pasa_convert_to_pproj (Com.g, temp, colperm, ncol) ;
        }
        else /* iterative method used to compute the projection */
        {
            pprojparm->start_guess = 3 ;      /*start guess is input lambda*/
            pprojparm->getfactor   = FALSE ;
            /* copy the projection in Com.d to x */
            pasa_copyx (x, Com.d, ncol) ;
        }

        /* use pproj permuted version of A */
        Com.Ap = Prob->Ap ;
        Com.Ai = Prob->Ai ;
        Com.Ax = Prob->Ax ;
        /* we perform a change of variable before computing the projection;
           this allows us to directly compute the search direction.
           Hence, the problem description in Prob is overwritten by
           the transformed expression. */
        pasa_copyx (Com.b, Prob->b, nrow) ;
        /* We delay the allocation of Com.bl and Com.bu until here since
           we set Com.bl = Bl and Com.bu = Bu before calling project */
        Prob->bl [0] = PASAZERO ;
        Prob->bu [0] = PASAZERO ;
        if ( ni > 0 )
        {
            pasa_copyx (Com.bl, Prob->bl, ni+1) ;
            pasa_copyx (Com.bu, Prob->bu, ni+1) ;
        }
        if ( loExists )
        {
            pasa_copyx (Com.lo, Prob->lo, ncol) ;
        }
        if ( hiExists )
        {
            pasa_copyx (Com.hi, Prob->hi, ncol) ;
        }

        /* each time grad_proj is restarted, we need copies of the
           original arrays */
        if ( GradProjOnly == FALSE )
        {
            /* For an active method, also store a copy of the problem data in
               pproj format since it is needed during the initialization of the
               active set algorithm. */
            PPwork *Work ;
            Work = Com.ppcom->Work ;
            /* integer */
            Copy.ncol = ncol ;
            pasa_copyi (Copy.Ai,         Prob->Ai,         Asize    ) ;
            pasa_copyx (Copy.Ax,         Prob->Ax,         Asize    ) ;
            pasa_copyi (Copy.Ap,         Prob->Ap,         ncol+1   ) ;
            pasa_copyi (Copy.ATp,        Work->ATp,        nrow+1   ) ;
            pasa_copyi (Copy.ATi,        Work->ATi,        Asize    ) ;
            pasa_copyi (Copy.ineq_row,   Prob->ineq_row,   ni+2     ) ;
            pasa_copy_int (Copy.sol_to_blk, Work->sol_to_blk, ni+2 ) ;
            /* we cannot allocate col_start until now since we do not know
               how many blocks are in the multilevel decomposition until
               after the initial projection */
            Copy.col_start =
                (PASAINT *) pasa_malloc (&status, (Work->blks+1), sI);
            /* terminate if there is not enough memory */
            if ( status )
            {
                status = PASA_OUT_OF_MEMORY ;
                pasa_wrapup (status, FALSE, &Com) ;
                return (status) ;
            }

            pasa_copyi (Copy.col_start, Work->col_start, Work->blks+1) ;

            /* float */
            pasa_copyx (Copy.ATx,      Work->ATx,      Asize) ;
            pasa_copyx (Copy.b,        Prob->b,        nrow ) ;
            pasa_copyx (Copy.bl,       Prob->bl,       ni+1 ) ;
            pasa_copyx (Copy.bu,       Prob->bu,       ni+1 ) ;
            if ( loExists )
            {
                pasa_copyx (Com.copylo,Prob->lo,       ncol ) ;
            }
            if ( hiExists )
            {
                pasa_copyx (Com.copyhi,Prob->hi,       ncol ) ;
            }
            pasa_initi (Com.row_to_ineq, (PASAINT) 0, nrow) ;
            for (i = 1; i <= ni; i++)
            {
                Com.row_to_ineq [Prob->ineq_row [i]] = i ;
            }
        }
        if ( fixed ) /* fixed variables exist */
        {
            if ( use_cholmod )
            {
                for (j = 0; j < ncol; j++)
                {
                    /* colperm [j] = compressed index associated
                                     with pproj index j
                       ifree   [j] = user index associated
                                     with compressed index j */
                    Com.colperm [j] = ifree [colperm [j]] ;
                    /* colperm [j] = user index associated with pproj index j */
                }
            }
            else
            {
                pasa_copyi (Com.colperm, ifree, ncol) ;
            }
        }
        else if ( use_cholmod ) /* no fixed variables and cholmod used */
        {
            
            pasa_copyi (Com.colperm, colperm, ncol) ;
        }

        /* for an active set method, we need the inverse permutation */
        if ( (GradProjOnly == FALSE) && use_cholmod )
        {
            for (j = 0; j < ncol; j++)
            {
                Com.invperm [Com.colperm [j]] = j ;
            }
        }
    }
    else /* pproj not used */
    {
        /* copy the projection, stored in Com.d, into x */
        pasa_copyx (x, Com.d, ncol) ;

        /* fixed variables exist, store users unfixed variables in colperm */
        if ( fixed )
        {
            pasa_copyi (Com.colperm, ifree, ncol) ;
        }
        if ( GradProjOnly == FALSE ) /* only bound constraints + active set */
        {
            pasa_copyx (Com.copylo, lo, ncol) ;
            pasa_copyx (Com.copyhi, hi, ncol) ;
        }
    }
#ifndef NDEBUG
    /* When pasa is used in debugging mode, it computes a projection twice
       to confirm the correctness of the first computation. Since the
       input arguments to pproj on the second call to it are different,
       we have another ppdata structure in the debugging mode. */
    Com.ppdataDEBUG      = &ppdataDEBUG ;
    ppdataDEBUG.lambda   = lambda ;
    ppdataDEBUG.Parm     = pprojparm ;
    ppdataDEBUG.nrow     = nrow ;
    ppdataDEBUG.nsing    = nsing ;
    /* ncol, ni, nsing, lo, hi, Bl, Bu change as variables become fixed */
#endif

    /* done using ifree */
    if ( fixed )
    {
        pasa_free (ifree) ;
        Com.temp_ifree  = NULL ;
    }

    if ( use_pproj )
    {
        /* Henceforth, there is no need to malloc arrays or compute a
           filling-reducing ordering in pproj. */
        pprojparm->use_prior_data = TRUE ;

        /* since the initial projection may not provide much bound info
           (especially if the initial guess was feasible), we start off
           by running phase1 in pproj and do not exploit the bound
           structure of the initial point */
        Com.initial_projection = TRUE ;
    }
    Com.testtol = grad_tol ; /* reset in grad_proj */

    /* The switch between the gradient projection algorithm and
       the active set algorithm is controlled by the variable toggle.
       toggle = PASA_GRAD_PROJ => use gradproj,
       toggle = PASA_ACTIVE_GP => use active set algorithm gradient project,
       toggle = PASA_CG_DESCENT => use CG
       e  >= switchfactor*E => use active set method (ACTIVE_GP/CG_DESCENT),
       e  <  switchfactor*E => use gradient projection method (GRAD_PROJ) */

    toggle = PASA_GRAD_PROJ ; /* default */
    /* If the problem is unconstrained, then use cg_descent unless
       GradProjOnly is TRUE */
    if ( !Bounds && !Aexists )
    {
        if ( GradProjOnly == FALSE )
        {
            toggle = PASA_CG_DESCENT ;
            /* Since grad_proj and active grad_proj are not used,
               we have no estimate for the bb parameter so we set bbk = -1. */
            Com.bbk = -1 ;
            Com.testtol = grad_tol ;
        }
    }

    while ( 1 )
    {
        /* ==================================================================
           If xk is the current iteration, then we compute pk, the
           projection of xk - bbk*gk onto the polyhedron, where bbk is
           the BB cyclic BB parameter at step k.  A line search is
           performed on the line segment [xk, pk] to generate xk+1.
           ================================================================== */
        if ( toggle == PASA_GRAD_PROJ )
        {
            status = pasa_grad_proj (&Com) ;
        }
        /* ==================================================================
           The active set algorithm is implemented using two different
           algorithms. If xk is the current iterate, then the algorithms are

           A. The gradient projection algorithm, but the bound variables
              at x_k, are kept fixed at their bounds, while the inequalities
              Bl <= Ax <= Bu that are active at xk are kept active when
              projecting the gradient into the polyhedron.

           B. The conjugate gradient algorithm but with all gradients
              replaced by gradients projected into the current active bounds
              and inequalities.

              We stop the active set gradient projection algorithm when the
              number of iterations since there is an addition to the active
              constraints exceeds pasaparm->terminate_agp. We stop the projected
              conjugate gradient algorithm whenever a new variable reaches
              a bound or a new inequality constraint becomes active.
           ================================================================== */
        else if ( toggle == PASA_ACTIVE_GP )
        {
            /* active set gradient projection algorithm */
            status = pasa_activeGP (&Com) ;
        }
        else /* ( toggle == PASA_CG_DESCENT ) conjugate gradient algorithm */
        {
            cgparm->grad_tol = Com.testtol ;
            cgdata->x    = Com.x ;
            cgdata->n    = Com.nf ;
            cgdata->Work = cgwork ;
            Com.use_cg = Stats->use_cg = TRUE ;
            status = pasa_cg_descent (cgdata, &Com);
        }
        if ( status < 0 )
        {
            toggle = status ;
        }
        else /* the problem was solved or an error occurred */
        {
            break ;
        }
    }
    pasa_wrapup (status, FALSE, &Com) ;
    return (status) ;
}

/* ==========================================================================
   === pasa_setup ===========================================================
   ==========================================================================
    Generate a pointer to a PASAdata structure with default values for all
    parameters, and NULL values for all pasa inputs. Return NULL pointer if
    there is not enough memory.
   ========================================================================== */
PASAdata * pasa_setup (void)
{
    int status ;
    /* Initialize PASAdata structure */
    PASAdata *Data ;

    status = PASA_OK ;
    /* Allocate memory for parameters and statistics, initialize parameters */
    Data = (PASAdata  *) pasa_malloc (&status, 1, sizeof (PASAdata)) ;
    if ( status )
    {
        status = PASA_OUT_OF_MEMORY ;
        Data = NULL ;
        return (Data) ;
    }
    Data->Parms = (PASAparms *) pasa_malloc (&status, 1, sizeof (PASAparms)) ;
    Data->Stats = (PASAstats *) pasa_malloc (&status, 1, sizeof (PASAstats)) ;
    Data->cgdata = cg_setup () ;
    Data->napdata = napheap_setup () ;
    if ( status || (Data->cgdata == NULL) || (Data->napdata == NULL) )
    {
        Data = NULL ;
        return (Data) ;
    }
    Data->Stats->use_pasa    = FALSE ;
    Data->Stats->use_cg      = FALSE ;
    Data->Stats->use_pproj   = FALSE ;
    Data->Stats->use_napheap = FALSE ;

    /* pasa Parm and Stat structures */
    Data->Parms->pasa = (PASAparm *) pasa_malloc (&status, 1, sizeof(PASAparm));
    Data->Stats->pasa = (PASAstat *) pasa_malloc (&status, 1, sizeof(PASAstat));
    if ( status )
    {
        Data = NULL ;
        return (Data) ;
    }
    /* set pasa Parm to default values */
    pasa_default (Data->Parms->pasa) ;

    /* store Parm and Stat structures of cg and napheap in Parms and Stats */
    Data->Parms->cg      = Data->cgdata->Parm ;
    Data->Stats->cg      = Data->cgdata->Stat ;
    Data->Parms->napheap = Data->napdata->Parm ;
    Data->Stats->napheap = Data->napdata->Stat ;

#ifndef NOPPROJ
    /* PPROJ could be used */
    Data->ppdata = pproj_setup () ;
    if ( Data->ppdata == NULL )
    {
        Data = NULL ;
        return (Data) ;
    }
    Data->Parms->pproj = Data->ppdata->Parm ;
    Data->Stats->pproj = Data->ppdata->Stat ;

#else
    /* PPROJ not used */
    Data->Parms->pproj = NULL ;
    Data->Stats->pproj = NULL ;
    Data->ppdata       = NULL ;
#endif

    /* set the remaining values in the PASAdata structure to EMPTY (= -1)
       or NULL */
    Data->x = NULL ;

    Data->lambda = NULL ;

    /* number of rows in A not specified by default (pasa will set nrow to be
       the maximum entry in Ai if a sparse matrix is present, while nrow is
       0 for unconstrained and bound constrained optimization, and it is
       1 for a napsack constraint). */
    Data->nrow = EMPTY ;

    /* number of components of x (number of cols in A); a value for
       Data->ncol must be provided by the user */
    Data->ncol = EMPTY ;

    /* There are three different ways to input the constraint matrix A:
 
       1. A dense packed matrix containing the matrix entries, either by rows
          or by columns. If the matrix is stored by rows, then A_by_rows
          should point to the matrix start; if the matrix is stored by
          columns, then A_by cols should point to the matrix start. Both the
          nrow and ncol elements of the data structure must also be given.

       2. The nonzero elements in the matrix can be stored as a triple:
              Ti  (row    indices of nonzero elements)
              Tj  (column indices of nonzero elements)
              Tx  (nonzero matrix entries)
              Tnz (number of nonzeros)

       3. The matrix can be stored standard sparse matrix format:
              Ap  (column pointers, the number of entries in Ap is 1 plus to the
                   number of columns in A. Ap [j] is the location of the first
                   nonzero in Ax associated with column j
              Ai  (the associated row indices of the nonzero matrix elements,
                   in each column the row indices should be in increasing order)
              Ax  (the numerical nonzero matrix elements)
        When A does not exist, the elements of the data structure are
        assigned the value NULL */

    /* size nrow*ncol, dense packed array containing matrix entries by rows */
    Data->A_by_rows = NULL ;
    /* size nrow*ncol, dense packed array containing matrix entries by cols */
    Data->A_by_cols = NULL ;

    /* size Tnnz */
    Data->Tj = NULL ;   /* column indices of the nonzero matrix elements */
    Data->Ti = NULL ;   /* row indices of the nonzero matrix elements */
    Data->Tx = NULL ;   /* numerical values of the nonzero matrix elements */
    Data->Tnz = EMPTY ; /* number of nonzeros in matrix */
    Data->sym = FALSE ; /* TRUE  => matrix is symmetric and only nonzero
                                    matrix elements on the main diagonal
                                    and on one side are given
                           FALSE => all the nonzero matrix elements are
                                    given (the matrix could still be
                                    symmetric) */
    Data->Ap = NULL ; /* size ncol+1, column pointers */
    Data->Ai = NULL ; /* size Ap [ncol], row indices of nonzero matrix elements,
                         increasing order in each column */
    Data->Ax = NULL ; /* size Ap [ncol], numerical values of nonzero matrix
                         elements */

    /* although bl and bu are multi-element arrays in general, they have
       only one element for a knapsack constraint *bl <= a'*x <= *bu,
       and these single elements must be allocated by the user */
    Data->bl = NULL ; /* size nrow, lower bounds for b above*/
    Data->bu = NULL ; /* size nrow, upper bounds for b above*/
    Data->lo = NULL ; /* size ncol, lower bounds for x */
    Data->hi = NULL ; /* size ncol, upper bounds for x */
    Data->y = NULL ; /* size ncol, project y onto the polyhedron */

    /* size ncol, the linear term for a quadratic or linear objective is c'x,
       except in NAPHEAP where the linear term is -c'x. */
    Data->c = NULL ;

    /* size ncol, linear constraint vector in NAPHEAP constraint
      *bl <= a'*x <= *bu. Note that Data->bl and Data->bu are arrays in
      general, but for a napsack problem, the arrays have only one element. */
    Data->a = NULL ;

    /* size ncol, diagonal of quadratic term in NAPHEAP */
    Data->d = NULL ;

    /* evaluate the function f at x, value (&f, x, n) */
    Data->value = NULL ;

    /* evaluate the gradient at x, grad  (g, x, n) */
    Data->grad = NULL ;

    /* evaluate the function f and gradient g at x, valgrad (&f, g, x, n) */
    /* NULL => use value & grad */
    Data->valgrad = NULL ;

    /* evaluate Hessian of objective (currently not used, should be NULL) */
    Data->hess = NULL ;

    /* If the problem is a QP, then instead of providing routines
       to evaluate the objective function and its gradient, the user
       could simply provide the Hessian of the objective
       and any linear term in the objective (see argument c above).
       If the linear term is not given, then it is taken to be zero.
       Whenever the objective value or its gradient is needed by pasa,
       it will be computed internally using the provided Hessian.
       The same three ways used to input the constraint matrix can
       be used for the Hessian of the quadratic, however, due to the
       symmetry of the Hessian, the dense format by rows or columns are the
       same. Also, when using the triples format, be sure to indicate if
       all the matrix elements are given, or only the elements on the main
       diagonal and on one side are given.  */

    /* dense matrix */
    Data->Hdense = NULL ;

    /* triples format */
    Data->HTj = NULL ;  /* size Hnz (column indices) */
    Data->HTi = NULL ;  /* size Hnz (row indices) */
    Data->HTx = NULL ;  /* size Hnz (nonzero matrix entries) */
    Data->Hnz = 0 ;     /* number of nonzeros in the matrix */
    Data->Hsym = FALSE; /* TRUE  => matrix is symmetric and only matrix elements
                                   on main diagonal and on one side are given
                           FALSE => all matrix elements are given
                                    (the matrix could still be symmetric) */
    /* sparse matrix format */
    Data->Hp = NULL ; /* column pointers for Hessian, size ncol + 1 */
    Data->Hi = NULL ; /* row indices for nonzero in Hessian */
    Data->Hx = NULL ; /* nonzero numerical values for Hessian matrix */

    /* Alternatively, for a quadratic objective, the user could provide
       a routine for computing the product between the Hessian and a
       vector. hprod evaluates this product when the optimization problem
       is constrained and cg_hprod evaluates the product for an unconstrained
       problem. This product needs to be computed in a special way when
       the problem is constrained: hprod (Hd, d, ifree, m, n) should compute
       Hd = H(:, ifree)*d where ifree and d have length n and H is m by m.
       Alternatively, cg_hprod (Hd, d, n) simply computes Hd = H*d where d
       has length n and H is n by n. */
    Data->hprod    = NULL ;
    Data->cg_hprod = NULL ;

    /* NULL => allocate real work space. */
    Data->xWork = NULL ;

    /* NULL  => allocate integer work space. */
    Data->iWork = NULL ;

    /* None of the routines require a starting guess for the multiplier lambda.
       If the user does not provide a pointer to the lambda array, then PASA
       allocates memory for lambda and returns the multiplier in lambda.
       pasa_terminate will free any allocated memory. The lambda_created pointer
       is only used internally by PASA when it creates or frees memory for
       lambda.

       Routines such as NAPHEAP, PPROJ, and the LP do not require
       a starting guess. If the user does not provide a pointer to x,
       then PASA allocates memory for the solution x. pasa_terminate
       will free any allocated memory. The x_created pointer is only
       used internally by PASA when it creates or frees  memory for x */
    Data->x_created = NULL ;
    Data->lambda_created = NULL ;
    /* If the problem is constrained and the user provided the matrix
       in either dense or triples format, then the sparse version of this
       matrix is created by pasa. Later, when pasa_terminate is run,
       the created sparse matrix is freed. The Hessian of a QP is treated
       in the same way. */
    Data->A_created = FALSE ;
    Data->H_created = FALSE ;
    Data->LP        = FALSE ; /* TRUE when the problem is an LP */

    /* Return pointer to PASAdata structure */
    return (Data) ;
}

/* ==========================================================================
   === pasa_terminate =======================================================
   ==========================================================================
    Free allocated memory associated with PASAdata structure
   ========================================================================== */
void pasa_terminate
(
    PASAdata **DataHandle
)
{
    PASAdata *Data ;
    Data = *DataHandle ;
    /* Free memory allocated by pasa */
    pasa_free (Data->Parms->pasa) ;
    pasa_free (Data->Stats->pasa) ;
    if ( Data->lambda_created != NULL )
    {
        if ( Data->lambda_created == Data->lambda )
        {
            pasa_free (Data->lambda) ;
            Data->lambda = Data->lambda_created = NULL ;
        }
        else /* user created lambda */
        {
            pasa_free (Data->lambda_created) ;
            Data->lambda_created = NULL ;
        }
    }
    if ( Data->x_created != NULL )
    {
        if ( Data->x_created == Data->x )
        {
            pasa_free (Data->x) ;
            Data->x = Data->x_created = NULL ;
        }
        else /* user created x */
        {
            pasa_free (Data->x_created) ;
            Data->x_created = NULL ;
        }
    }

    /* if Ap, Ai, and Ax were created by pasa_read_matrix, then free them */
    if ( Data->A_created )
    {
        pasa_free (Data->Ap) ;
        pasa_free (Data->Ai) ;
        pasa_free (Data->Ax) ;
        Data->A_created = FALSE ;
    }

    /* if Hp, Hi, and Hx were created by pasa_read_matrix, then free them */
    if ( Data->H_created )
    {
        pasa_free (Data->Hp) ;
        pasa_free (Data->Hi) ;
        pasa_free (Data->Hx) ;
        Data->H_created = FALSE ;
    }

    cg_terminate (&(Data->cgdata)) ;
    napheap_terminate (&(Data->napdata)) ;
#ifndef NOPPROJ
    pproj_terminate (&(Data->ppdata)) ;
#endif
    pasa_free (Data->Parms) ;
    pasa_free (Data->Stats) ;

    /* Free Data struct */
    pasa_free (Data) ;
}

/* ==========================================================================
   === pasa_autodetect ======================================================
   ==========================================================================
    Analyze the user's pasadata structure and attempt to classify the
    problem into one of the following 7 classes:
        UNC     - unconstrained optimization:
                      min f(x) s.t. x in R^n
        BNC     - purely bound constrained optimization:
                      min f(x) s.t. lo <= x <= hi
        LP      - linear program:
                      min c'*x s.t. lo <= x <= hi, bl <= A*x <= bu
        QP      - quadratic program:
                      min 0.5*x'*Q*x + c'*x s.t. lo <= x <= hi, bl <= A*x <= bu
        NL      - nonlinear program:
                      min f(x) s.t. lo <= x <= hi, bl <= A*x <= bu
        NAPSACK - separable quadratic with napsack constraints:
                      min 0.5*x*'D*x + c'*x s.t. lo <= x <= hi, bl <= a'*x <= bu
        PROJ    - polyhedral constrained projection:
                      min 0.5*||x - y||^2 s.t. lo <= x <= hi, bl <= A*x <= bu
    NOTE: If f is a quadratic and the user provides either the Hessian of
          f or a routine for multiplying the Hessian by a vector, then the
          problem should be specified as a QP, not as UNC or NL.
   ========================================================================== */
int pasa_autodetect
(
    PASAdata *pasadata
)
{
    /* If y was provided, then the problem must be a projection. */
    if ( pasadata->y != NULL )
    {
        return (PASA_PROJ) ;
    }
    /* if Hessian or hprod is given, then problem is assumed to be a QP */
    else if ( (pasadata->Hp != NULL) || (pasadata->hprod != NULL)
                                     || (pasadata->cg_hprod  != NULL) )
    {
        return (PASA_QP) ;
    }
    /* If (a and d) or (a and c) or (d and c) were provided,
       then the problem structure is napsack */
    else if ( ((pasadata->a != NULL) && (pasadata->d != NULL)) ||
              ((pasadata->a != NULL) && (pasadata->c != NULL)) ||
              ((pasadata->d != NULL) && (pasadata->c != NULL)) )
    {
        return (PASA_NAPSACK) ;
    }
    /* check problem constraints */
    else if ( (pasadata->bl == NULL) && (pasadata->bu == NULL) )
    {
        /* problem does not have polyhedral constraints */
        if ( (pasadata->lo != NULL) || (pasadata->hi != NULL) ) 
        {   /* problem has bound constraints */
            return (PASA_BNC) ;
        }
        else
        {   /* problem is unconstrained */
            return (PASA_UNC) ;
        }
    }
    else if ( (pasadata->Ap != NULL) || (pasadata->a != NULL) )
    {   /* problem has polyhedral constraints */
        if (pasadata->c != NULL)
        {
            /* problem is LP */
            return (PASA_LP) ;
        }
        else if ( pasadata->value != NULL )
        {
            /* problem has a nonlinear objective */
            return (PASA_NL) ;
        }
        else /* missing objective */
        {
            return (PASA_MISSING_OBJECTIVE) ;
        }
    }
    else
    {
        return (PASA_RHS_GIVEN_BUT_MATRIX_MISSING) ;
    }
}

/* ==========================================================================
   === pasa_wrapup ==========================================================
   ==========================================================================
    Final messages and free memory before returning to user
   ========================================================================== */
void pasa_wrapup
(
    int     status, /* termination status */
    int fastreturn, /* T => return after printing status and freeing temp mem */
    PASAcom   *Com  /* common data for PASA */
)
{
    int location ;
    PASAparm *pasaparm ;
    PASAstat *pasastat ;
    PASAdata *pasadata = Com->pasadata ;

    pasaparm = Com->pasaparm ;
    pasastat = Com->pasastat ;
    pasastat->status = status ;

    /* free any temporary variables that were created */
    if ( Com->temp_userBl != NULL ) pasa_free (Com->temp_userBl) ;
    if ( Com->temp_userBu != NULL ) pasa_free (Com->temp_userBu) ;
    if ( Com->temp_ifree  != NULL ) pasa_free (Com->temp_ifree) ;

    /* print the run status */
    if ( pasaparm->PrintStatus && fastreturn )
    {
        if ( status != PASA_ERROR_TOLERANCE_SATISFIED )
        {
            pasastat->err = PASAINF ;
            pasastat->f = PASAINF ;
        }
        pasa_print_status (pasadata) ;
        printf ("\n") ;
    }

    if ( fastreturn ) return ;

    /* store final f and error in pasa statistics structure */
    pasastat->err = Com->E ;
    location = Com->location ;
    pasastat->f = Com->f ;

    /* print the run status */
    if ( pasaparm->PrintStatus )
    {
        pasa_print_status (pasadata) ;
        printf ("\n") ;
    }

    /* print the statistics */
    if ( (pasaparm->PrintStat == TRUE) )
    {
        pasa_print_stats (pasadata) ;
    }

    /* if A exists and the matrix was scaled by RowScale, then scale the
       multiplier by RowScale */
    if ( Com->use_pproj )
    {
        PASAINT   const      nrow = Com->nrow ;
        PASAFLOAT const         t = PASAONE/Com->lambda_scale ;
        int       const ScaleRows = Com->ScaleRows ;
        /* no need to permute lambda for a projection problem since PPROJ
           already does the permutation */
        if ( Com->PROJ == TRUE )
        {
            /* multiply components of lambda by corresponding scale factors */
            if ( ScaleRows )
            {
                pasa_hadamard (Com->lambda, Com->RowScale, t, nrow);
            }
            else 
            {
                pasa_scale (Com->lambda, Com->lambda, t, nrow) ;
            }
        }
        /* for LPs, the multiplier already transformed back */
        else if ( !Com->LP )
        {
            PASAINT const *rowperm = Com->ppcom->Prob->rowperm ;
            PASAINT i, j ;
            PASAFLOAT *lambdatemp = Com->ppcom->Work->arrayd ;
            int const permute = (rowperm != NULL) ;
            int const one     = (t == PASAONE) ;
            if ( ScaleRows )
            {

                for (i = 0; i < nrow; i++)
                {
                    j = (permute) ? rowperm [i] : i ;
                    lambdatemp [j] = (one) ?Com->lambda [i]*Com->RowScale [j]:
                                            Com->lambda [i]*Com->RowScale [j]*t;
                }
                pasa_copyx (Com->lambda, lambdatemp, nrow) ;
            }
            else
            {
                if ( permute )
                {
                    for (i = 0; i < nrow; i++)
                    {
                        j = rowperm [i] ;
                        lambdatemp [j] = (one) ? Com->lambda [i] :
                                                 Com->lambda [i]*t ;
                    }
                    pasa_copyx (Com->lambda, lambdatemp, nrow) ;
                }
                else
                {
                    pasa_scale (Com->lambda, Com->lambda, t, nrow) ;
                }
            }
        }
    }
    else if ( Com->use_napheap )
    {
        Com->lambda [0] = Com->nap_lambda/Com->lambda_scale ;
    }

    /* The optimal x is stored in userx with the following exceptions:
       1. If QP is TRUE, then we need to map the current iterate
          into userx and then copy the composite back to x
       2. If GradProjOnly is TRUE, there is no A, and there are no fixed
          variables, then x contains the optimal solution.
       3. If Com->need_userx is FALSE, then x contains the optimal solution.  */
    if ( Com->QP )
    {
        if ( (location == PASA_ACTIVE_GP) || (location == PASA_CG_DESCENT) )
        {
            if ( Com->order != NULL )
            {
                pasa_convert_to_user (Com->userx, Com->x, Com->order, Com->nf) ;
                pasa_copyx (Com->x, Com->userx, Com->ucol) ;
            }
        }
        else /* location = base code or grad_proj */
        {
            if ( Com->colperm != NULL )
            {
                pasa_convert_to_user (Com->userx, Com->x, Com->colperm,Com->nf);
                pasa_copyx (Com->x, Com->userx, Com->ucol) ;
            }
        }
    }
    else /* not a quadratic objective */
    {
        /* if ( (Com->Aexists == TRUE) || (Com->fixed == TRUE) ||
             (Com->nf < Com->ucol) ) */
        if ( Com->need_userx )
        {
             /* userx contains x in user's coordinates */
             pasa_copyx (Com->x, Com->userx, Com->ucol) ;
        }
        /* otherwise x contains the solution in user coordinates */
    }

    /* free memory */
    if ( (Com->use_pproj == TRUE) && (pasaparm->GradProjOnly == FALSE) )
    {
        if ( Com->Copy->col_start != NULL )
        {
            pasa_free (Com->Copy->col_start) ;
        }
    }
    if ( Com->work != NULL )
    {
        pasa_free (Com->work) ;
    }
    if ( Com->iwork != NULL )
    {
        pasa_free (Com->iwork) ;
    }

    /* restore pproj parameter values possibly modified in the main pasa code */
    if ( Com->use_pproj )
    {
        PPparm *pprojparm = Com->pprojparm ;
        pprojparm->grad_tol        = Com->userparms.p_grad_tol ;
        pprojparm->PrintStatus     = Com->userparms.p_PrintStatus ;
        pprojparm->return_data     = Com->userparms.p_return_data ;
        pprojparm->use_prior_data  = Com->userparms.p_use_prior_data ;
        pprojparm->loExists        = Com->userparms.p_loExists ;
        pprojparm->hiExists        = Com->userparms.p_hiExists ;
        pprojparm->getfactor       = Com->userparms.p_getfactor ;
        pprojparm->permute         = Com->userparms.p_permute ;
        pprojparm->start_guess     = Com->userparms.p_start_guess ;
        pprojparm->use_startup     = Com->userparms.p_use_startup ;
        pprojparm->LP              = Com->userparms.p_LP ;
        pprojparm->check_if_y_feas = Com->userparms.p_check_if_y_feas ;
    }
    else if ( Com->use_napheap )
    {
        NAPparm *napparm = Com->napparm ;
        napparm->PrintStatus    = Com->userparms.n_PrintStatus ;
        napparm->return_data    = Com->userparms.n_return_data ;
        napparm->use_prior_data = Com->userparms.n_use_prior_data ;
        napparm->loExists       = Com->userparms.n_loExists ;
        napparm->hiExists       = Com->userparms.n_hiExists ;
        napparm->d_is_one       = Com->userparms.n_d_is_one ;
    }
    else if ( Com->use_cg )
    {
        CGparm *cgparm = Com->cgparm ;
        cgparm->grad_tol        = Com->userparms.c_grad_tol ;
        cgparm->QPshift         = Com->userparms.c_QPshift ;
        cgparm->PrintStatus     = Com->userparms.c_PrintStatus ;
        cgparm->PrintStat       = Com->userparms.c_PrintStat ;
        cgparm->PrintParm       = Com->userparms.c_PrintParm ;
    }

#ifndef NOPPROJ
    /* if polyhedral constraints present, free associated pproj memory */
    if ( Com->use_pproj == TRUE )
    {
        /* free memory unless pproj terminated before malloc's performed */
        if ( (status != PPROJ_OUT_OF_MEMORY) &&
             (status != PPROJ_MISSING_PRIOR_DATA) &&
             (status != PPROJ_START_GUESS_NEEDS_PRIOR_DATA) &&
             (status != PPROJ_INVALID_LINEAR_CONSTRAINT_BOUNDS) &&
             (status != PPROJ_BOTH_NI_AND_NSING_POSITIVE) &&
             (status != PPROJ_NSING_START_GUESS_PROB) )
        {
            pproj_freeAll (&(Com->ppcom)) ; /* free the pproj memory */
            Com->pasadata->ppdata->priordata = NULL ;
        }
    }
#endif
}

/* ==========================================================================
   === pasa_cgexpandQP ======================================================
   ==========================================================================
    Perform an expansion step in cg_descent for a bound constrained problem
    whose objective is quadratic.
   ========================================================================== */
void pasa_cgexpandQP
(
    PASAFLOAT      *x, /* current iterate */
    PASAFLOAT      *d, /* the search direction */
    PASAINT         n, /* dimension of d */
    PASAFLOAT maxstep, /* the initial stepsize associated with Df */
    PASAFLOAT   Dfold, /* the cost improvement associated with current step */
    PASAcom      *Com  /* common data for pasa */
)
{
    CGparm *cgparm = Com->cgparm ; /* PASAparm structure */
    PASAINT   ncol = Com->ncol ;   /* number of cols before compress */
    /* expansion allowed if a bound constraint becomes active before an
       inequality becomes active */
    PASAINT j ;
    PASAFLOAT *Qdxold = Com->Qd ;
    PASAFLOAT   *work = Com->cgexpand ;     /* workspace for expansion */
    PASAFLOAT   *xnew = work ;  work += n ; /* new x to test improvement */
    PASAFLOAT   *xold = work ;  work += n ; /* prior tested x */
    PASAFLOAT  *dxnew = work ;  work += n ; /* compressed change in x */
    PASAFLOAT *Qdxnew = work ;  work += ncol ; /* full Q times dxnew */
 
    PASAFLOAT const      *lo = Com->lo ;        
    PASAFLOAT const      *hi = Com->hi ;
    PASAINT   const   *order = Com->order ;
    int       const loExists = Com->loExists ;
    int       const hiExists = Com->hiExists ;
    int const nexpandlimit = cgparm->nexpand ;
    PASAFLOAT step = maxstep ;  /* step associated with current iterate */
    PASAFLOAT stepold = step ;
    int nexpand = 0 ;           /* number of expansion steps */
    while ( nexpand < nexpandlimit )
    {
        step *= cgparm->step_growth ;
        pasa_step (xnew, x, d, step, n) ;
        /* truncate xnew so that it satisfies the bound constraints */
        for (j = 0; j < n; j++)
        {
            PASAFLOAT t = xnew [j] ;
            if      ( loExists && (t < lo [j]) ) t = lo [j] ;
            else if ( hiExists && (t > hi [j]) ) t = hi [j] ;
            xnew [j] = t ;
            dxnew [j] = t - x [j] ;
        }
        if ( Com->hprod_status )
        {
            pasa_builtin_hprod (Qdxnew, dxnew, order, Com->ncol, n,
                                Com->Hp, Com->Hi, Com->Hx) ;
        }
        else
        {
            Com->hprod (Qdxnew, dxnew, order, ncol, n) ;
        }
        PASAFLOAT dxQdx = PASAZERO ;
        for (j = 0; j < n; j++)
        {
            dxQdx += dxnew [j]*Qdxnew [order [j]] ;
        }
        PASAFLOAT Dfnew = 0.5*dxQdx + pasa_dot (Com->g, dxnew, n) ;
        if ( Dfnew >= Dfold ) break ; /* move did not improve cost */
        else                          /* the move improves cost */
        {
            Dfold = Dfnew ;
            stepold = step ;

            PASAFLOAT *y = xold ;
            xold = xnew ;
            xnew = y ;  /* will make a new step and store iterate in xnew */

            y = Qdxold ;
            Qdxold = Qdxnew ;
            Qdxnew = y ;

            nexpand++ ;
        }
    }
    Com->maxstep = stepold ; /* maxstep now stores the step taken */
    if ( nexpand == 0 )
    {
        pasa_daxpy (x, d, maxstep, n) ;
        pasa_daxpy (Com->gtot, Com->Qd, maxstep, ncol) ;
        /* the maxstep causes a variable to hit a bound, be sure that the
           variable is at its bound */
    }
    else
    {
        pasa_copyx (x, xold, n) ;
        pasa_daxpy (Com->gtot, Qdxold, PASAONE, ncol) ;
    }
    Com->f += Dfold ;
    if ( cgparm->PrintLevel >= 1 )
    {
        printf ("nexpand: %i fnew: %e\n", nexpand, Com->f) ;
    }
}

/* ==========================================================================
   === pasa_cgexpand ========================================================
   ==========================================================================
    Perform an expansion step in cg_descent for a bound constrained problem.
   ========================================================================== */
int pasa_cgexpand
(
    PASAFLOAT      *x, /* current iterate */
    PASAFLOAT      *d, /* the search direction */
    PASAINT         n, /* dimension of x and d */
    PASAFLOAT maxstep, /* the initial stepsize along d */
    PASAFLOAT    fold, /* the cost associated with current step */
    PASAcom  *pasacom, /* common data for pasa */
    CGcom      *cgcom  /* common data for cg */
)
{
    int status ;
    PASAINT j ;
    CGparm    *cgparm = pasacom->cgparm ; /* PASAparm structure */
    PASAFLOAT *pasaxnew = pasacom->xnew ; /* location of xnew destroyed below */
    PASAFLOAT *xold = pasaxnew ;          /* x of current best function value */
    PASAFLOAT  flag = -PASAONE ;          /* used in pasa_evaluate */
    PASAFLOAT  step = maxstep ;           /* step associated with current x */
    PASAFLOAT stepold = step ;

    /* workspace for expansion, also pasa_evaluate uses pasacom->xnew as the
       point where the function is evaluated */
    PASAFLOAT *xnew = pasacom->xnew = pasacom->cgexpand ;

    PASAFLOAT const      *lo = pasacom->lo ;        
    PASAFLOAT const      *hi = pasacom->hi ;
    int       const loExists = pasacom->loExists ;
    int       const hiExists = pasacom->hiExists ;
    int nexpand = 0 ;
    while ( nexpand < cgparm->nexpand )
    {
        step *= cgparm->step_growth ;
        pasa_step (xnew, x, d, step, n) ;
        /* truncate xnew so that it satisfies the bound constraints */
        for (j = 0; j < n; j++)
        {
            PASAFLOAT t = xnew [j] ;
            if      ( loExists && (t < lo [j]) ) t = lo [j] ;
            else if ( hiExists && (t > hi [j]) ) t = hi [j] ;
            xnew [j] = t ;
        }
        status =  pasa_evaluate (PASAZERO, &flag, pasacom, "f") ;
        if ( status != PASA_OK ) return (status) ;
        PASAFLOAT fnew = pasacom->f ;
        if ( fnew >= fold ) break ; /* move did not improve cost */
        else                        /* the move improves cost */
        {
            fold = fnew ;
            stepold = step ;
            PASAFLOAT *y = xold ;
            xold = xnew ;           /* xold stores best iterate so far */
            /* will make a new step and store iterate in xnew */
            nexpand++ ;
            xnew = pasacom->xnew = y ;
        }
    }
    pasacom->maxstep = stepold ; /* maxstep now stores the step taken */
    pasacom->f = cgcom->f = fold ;
    pasa_copyx (x, xold, n) ;
    if ( nexpand > 0 ) /* gradient needs to be calculated at the new iterate */
    {
        pasacom->xnew = xold ; /* flag = -1 => evaluate at xnew */
        status = pasa_evaluate (PASAZERO, &flag, pasacom, "g") ;
    }
    pasa_copyx (cgcom->g, cgcom->gnew, n) ;
    pasacom->xnew = pasaxnew ;
    if ( cgparm->PrintLevel >= 1 )
    {
        printf ("nexpand: %i fnew: %e\n", nexpand, fold) ;
    }
#ifndef NDEBUG
    pasa_check_feas (pasacom) ;
#endif
    /* both x and the gradient at x are stored in cgcom->x and cgcom->g */
    return (status) ;
}

/* ==========================================================================
   === pasa_scalecopy =======================================================
   ==========================================================================
    For a range of element in the A matrix, update the RowScale to take into
    account the new elements, and copy the elements into the new, possibly
    compressed matrix. For an LP, column singletons were removed so they
    do not contribute to the largest element in the row.
   ========================================================================== */
void pasa_scalecopy
(
    PASAFLOAT *RowScale, /* accumulate the absolute largest element in row */
    PASAINT           n, /* number of elements of process */
    PASAFLOAT    *newAx, /* the new location of the matrix elements */
    PASAFLOAT    *oldAx, /* the old location of the matrix elements */
    PASAINT      *newAi, /* the new location of the matrix row indices */
    PASAINT      *oldAi  /* the old location of the matrix row indices */
)
{
    PASAINT j ;
    if ( newAi != oldAi )
    {
        for (j = 0; j < n; j++)
        {
            PASAINT   const  i = newAi [j] = oldAi [j] ;
            PASAFLOAT const ax = newAx [j] = oldAx [j] ;
            if ( fabs (ax) > RowScale [i] ) RowScale [i] = fabs (ax) ;
        }
    }
    else /* no need to copy since newAi = oldAi */
    {
        for (j = 0; j < n; j++)
        {
            PASAINT   const  i = oldAi [j] ;
            PASAFLOAT const ax = newAx [j] = oldAx [j] ;
            if ( fabs (ax) > RowScale [i] ) RowScale [i] = fabs (ax) ;
        }
    }
}

/* ==========================================================================
   === pasa_grad_proj =======================================================
   ==========================================================================
    Perform the gradient projection algorithm until the termination condition
    local error e >= theta * global error E is satisfied. If xk is the
    current iteration, then we compute pk the projection of xk - bbk*gk
    onto the polyhedron.  A line search is performed on the
    line segment [xk, pk] to generate xk+1. The value of bbk is obtained
    from a cyclic version of the BB algorithm. This means that we reuse
    the BB approximation bbk*I to the Hessian for one or more iterations
    before updating its value. The cycle length is the number of times
    that the bb parameter is reused.
   ========================================================================== */
int pasa_grad_proj /* return:
                       PASA_ACTIVE_GP
                       PASA_ERROR_TOLERANCE_SATISFIED
                       PASA_POLYHEDRON_INFEASIBLE
                       PASA_INVALID_LINEAR_CONSTRAINT_BOUNDS
                       PASA_ITERATIONS_EXCEED_MAXITS_IN_GRAD_PROJ
                       PASA_LINE_SEARCH_STEPS_EXCEED_MAXSTEPS_IN_GRAD_PROJ
                       PASA_OUT_OF_MEMORY_IN_PPROJ
                       PASA_SSOR_ASCENT_FAILURE_IN_PPROJ
                       PASA_SEARCH_DIRECTION_NOT_DESCENT_DIRECTION_IN_GRAD_PROJ
                       PASA_SSOR_ITERATIONS_EXCEED_MAX_ITS_IN_PPROJ */
(
    PASAcom *Com  /* common data for PASA */
)
{
    int flagCG, its_since_test, it_start, na, PrintLevel, status,
        test_increment ;
    PASAINT i, j, k, *Ap, *Ai, *colperm, *ifree ;
    PASAFLOAT alpha, muAd, delta, deltaf, dQd, E, f0, fR, fRpert,
              gtd, gtdnew, pert, s, safe0, safe1,
              stepdecay, t, *Ax, *d, *x, *g, *gnew ;
    PASAparm *Parm ;      /* parameters for PASA */
    PASAstat *Stat ;      /* statistics for PASA */
    PASAcopy *Copy ;      /* copy of pproj's prob structure */
    PPprob   *Prob ;      /* problem in PPROJ */
    PPwork   *Work ;      /* work structure in PPROJ */

    /* if the code comes from CG and an estimate for the bb parameter
       was obtained, then we use it */
    int const prior_location = Com->location ;
    if ( prior_location == PASA_CG_DESCENT )
    {
        flagCG = TRUE ;
        if ( Com->cg_bb_est > PASAZERO )
        {
            Com->bbk = Com->cg_bb_est ;
        }
    }
    else flagCG = FALSE ;
    Parm = Com->pasaparm ; /* parameters */
    PrintLevel = Parm->PrintLevel ;
    if ( PrintLevel >= 1 )
    {
        printf ("START grad_proj, testtol: %e\n", Com->testtol) ;
    }
    int const            Aexists = Com->Aexists ;
    int const             Bounds = Com->Bounds ;
    int const                 QP = Com->QP ;
    int const        use_napheap = Com->use_napheap ;
    int const          use_pproj = Com->use_pproj ;
    int const order_with_colperm = Com->order_with_colperm ;
    int const       GradProjOnly = Parm->GradProjOnly ;
    int const           maxsteps = Parm->maxsteps ; /* max # Armijo backsteps */

    if ( use_pproj )
    {
        Prob = Com->ppcom->Prob ;
        Ai = Prob->Ai ;
        Ax = Prob->Ax ;
        Ap = Prob->Ap ;
    }
    x = Com->x ;
    g = Com->g ;
    Stat = Com->pasastat ;        /* pasa statistics */
    PASAINT const n = Com->ncol ; /* problem dimension */
    PASAINT const nf = Com->nf ;  /* # free variables in active set method */
    Com->nf = n ;                 /* all variables free in grad_proj */
    PASAINT const nc = Com->nc ;  /* number bound variables (columns) */
    Com->nc = 0 ;                 /* in grad_proj there are no bound columns */
    ASSERT (nc == n - nf) ;
    ifree = Com->ifree ;
    colperm = Com->colperm ;
    PASAINT *order = (order_with_colperm) ? colperm : NULL ;

    /* if returning from the active set algorithm, then restore problem data */
    if ( prior_location != PASA_BASE_CODE )
    {
        if ( PrintLevel >= 1 )
        {
            printf ("%ld free variables and %ld bound variables\n",
                   (LONG) nf, (LONG) (n-nf)) ;
        }

        /* Reset Com->switchfactor to its value in the Parm structure. Since
           Com->switchfactor enters into the value for testtol, the change
           in Com->switchfactor results in a change to Com->testtol. */
        Com->testtol *= Parm->switchfactor/Com->switchfactor ;
        Com->testtol = PASAMAX (Com->testtol, Com->grad_tol) ;
        Com->switchfactor = Parm->switchfactor ;

        Copy = Com->Copy ;

        /* The treatment of QPs and a general nonlinear function are different.
           For a QP each active bound on x is saved in userx, while none
           of the other elements of userx are utilized. For a general
           nonlinear function, the entire userx array needs to be
           constructed before the function can be evaluated. Hence,
           Hence, when returning from the active set part of the code
           (either the active gradient projection or the CG algorithm)
           for a general nonlinear function, we can construct the complete
           x array in the ordinary gradient projection code from userx.
           For a QP, on the other hand, we need to first insert the free
           components, stored in x, in userx before constructing the
           complete x array from userx. */
        if ( Bounds ) /* x is current iterate if no bounds present */
        {
            if ( QP )
            {
                for (j = 0; j < nf; j++)
                {
                    Com->userx [ifree [j]] = x [j] ;
                }
            }
            /* If there are bounds, userx is now correct, whether a QP
               or general f. Overwrite x with the complete vector excluding
               fixed variables. */
            if ( order_with_colperm ) /* bounds + fixed or pproj ordering */
            {
                pasa_convert_to_pproj (x, Com->userx, colperm, n) ;
            }
            else if ( Com->need_userx ) /* bounds + !fixed and !pproj ordering*/
            {
                pasa_copyx (x, Com->userx, n) ; /* straight copy */
            }
        }

        /* when pproj is employed, matrices and bounds are restored */
        if ( use_pproj == TRUE )
        {
            int *temp ;
            PASAINT nnz, nrow, p, *ATp, *ATi, *Anz, *bound_cols, *bound_rows,
                    *invperm, *ir, *row_to_ineq, *ineq_row, *col_beg, *col_end ;
            PASAFLOAT *ATx ;

            /* extract number of bound inequalities (number of rows) */
            PASAINT const nr = Com->nr ;
            /* all the strict inequalities in grad_proj are free, there are
               no bound inequalities */
            Com->nr = 0 ;
            /* prob_ni is the number of strict inequalities in the
               compressed problem */
            PASAINT const prob_ni = Prob->ni ;
            if ( PrintLevel >= 1 )
            {
                printf ("%ld free and %ld bound inequalities\n",
                       (LONG) prob_ni, (LONG) nr) ;
            }
            /* ni is the number of strict inequalities for the
               original problem */
            PASAINT const ni = Com->ni ;
            /* in grad_proj, the number of strict inequalities is ni */
            Prob->ni = ni ;
            ASSERT (prob_ni + nr == ni) ;

            /* Set ir greater than Com->ni for all dropped inequalities;
               for active inequalities, use row_to_ineq to set ir
               based on whether the lower or the upper bound is active */
            row_to_ineq = Com->row_to_ineq ;
            ineq_row = Prob->ineq_row ;
            Work = Com->ppcom->Work ;
            ir = Com->ppcom->Work->ir ;
            for (i = 1; i <= prob_ni; i++)
            {
                PASAINT const row = ineq_row [i] ;
                ASSERT ( ir [row] != 0 ) ;
                if ( ir [row] > prob_ni ) /* row is dropped */
                {
                    ir [row] = ni + 1 ;
                }
                else
                {
                    if ( ir [row] > 0 ) ir [row] =  row_to_ineq [row] ;
                    else                ir [row] = -row_to_ineq [row] ;
                }
            }

            bound_rows = Com->bound_rows ;
            for (k = 0; k < nr; k++)
            {
                i = bound_rows [k] ; /*-(row+1) if at low bound, else +(row+1)*/
                if ( i < 0 ) /* lower bind */
                {
                    i = -(i+1) ;
                    /* convert row # to inequality #, store negative number */
                    ir [i] = -row_to_ineq [i] ;
                }
                else         /* upper bind */
                {
                    i -= 1 ;
                    /* convert row # to inequality #, store positive number */
                    ir [i] = row_to_ineq [i] ;
                }
                Prob->b [i] = PASAZERO ;
            }
            nrow = Com->nrow ;
            pasa_copyi (Prob->Ap,             Copy->Ap,         n+1   ) ;
            pasa_copyi (Work->ATp,            Copy->ATp,        nrow+1) ;
            pasa_copyi (Work->AFTp,           Copy->ATp,        nrow+1) ;
            nnz = Prob->Ap [n] ;
            pasa_copyi (Work->ATi,            Copy->ATi,        nnz ) ;
            pasa_copyi (Prob->ineq_row,       Copy->ineq_row,   ni+2) ;
            pasa_copy_int (Work->sol_to_blk, Copy->sol_to_blk, ni+2) ;
            pasa_copyx (Work->ATx,            Copy->ATx,        nnz ) ;
            pasa_copyx (Com->b,               Copy->b,          nrow) ;
            pasa_copyx (Com->bl,              Copy->bl,         ni+1) ;
            pasa_copyx (Com->bu,              Copy->bu,         ni+1) ;
            pasa_copyx (Com->lo,              Com->copylo,      n   ) ;
            pasa_copyx (Com->hi,              Com->copyhi,      n   ) ;
            pasa_copyi (Work->col_start,      Copy->col_start,  Work->blks+1) ;
            Work->A->ncol = n ;
            Work->A->nzmax = nnz ;

            if ( Bounds )
            {
                /* set Work->ib according to current factorization */
                invperm = Com->invperm ;

                /* integer work space, will use n ints */
                temp = (int *) Work->arrayi ;
                bound_cols = Com->bound_cols ;

                /* If a column was fixed in the compressed matrix, then
                   we need to fix it in the original matrix
                       ifree [j]   = variable in the original matrix associated
                                     with column j of compressed matrix
                       invperm [j] = column in pproj associated with column j
                                     in original matrix
                    First set ib for the bound variables, store in temp array.*/
                for (k = 0; k < nc; k++)
                {
                    j = bound_cols [k] ;  /* index in original matrix */
                    if ( j < 0 )
                    {
                        j = -(j+1) ;
                        i = invperm [j] ; /* corresponding index in pproj */
                        temp [i] = -1 ;
                    }
                    else
                    {
                        i = invperm [j] ;
                        temp [i] = 1 ;
                    }
                }

                /* Then set ib for the current free variables. */
                for (j = 0; j < nf; j++) /* nf = # free columns from cg or agp*/
                {
                    temp [invperm [ifree [j]]] = Work->ib [j] ;
                }

                /* Copy temp into Work->ib */
                pasa_copy_int (Work->ib, temp, n) ;
#ifndef NDEBUG
                int *ActiveCols = Com->ppcom->Check->ActiveCols ;
                for (j = 0; j < n; j++)
                {
                    if ( temp [j] ) ActiveCols [j] = (int) 0 ;
                    else            ActiveCols [j] = (int) 1 ;
                }
#endif
            }
            Prob->ncol = n ;

            /* If ni > 0 construct A using AT and the current active rows.
               Otherwise, simply copy A. */
            Anz = Prob->Anz ;
            ATp = Work->ATp ;
            ATi = Work->ATi ;
            ATx = Work->ATx ;

            if ( ni == 0 )
            {
                pasa_copyi (Ai, Copy->Ai, nnz) ;
                pasa_copyx (Ax, Copy->Ax, nnz) ;
                for (j = 0; j < n; j++)
                {
                    Anz [j] = Ap [j+1] - Ap [j] ;
                }
            }
            else
            {

                /* done using temp, now use arrayi for col_end and col_beg,
                   2n PASAINTs needed */
                col_end = Work->arrayi ;
                pasa_copyi (col_end, Ap+1, n) ;
                col_beg = col_end + n ;
                pasa_copyi (col_beg, Ap, n) ;

                p = 0 ;
                for (i = 0; i < nrow; i++)
                {
                    PASAINT const q = ATp [i+1] ;
                    if ( ir [i] <= ni ) /* row is active */
                    {
                        for (; p < q; p++)
                        {
                            PASAINT const J = ATi [p] ;
                            Ai [col_beg [J]] = i ;
                            Ax [col_beg [J]] = ATx [p] ;
                            col_beg [J]++ ;
                        }
                    }
                    else               /* row is dropped */
                    {
                        for (; p < q; p++)
                        {
                            PASAINT const J = ATi [p] ;
                            col_end [J]-- ;
                            Ai [col_end [J]] = i ;
                            Ax [col_end [J]] = ATx [p] ;
                        }
                    }
                }
                for (j = 0; j < n; j++)
                {
                    Anz [j] = col_beg [j] - Ap [j] ;
                }
            }
        }
        else if ( use_napheap == TRUE )
        {
            pasa_copyx (Com->nap_a, Com->nap_acopy, n) ;
            Com->nap_bl = Com->nap_blcopy ;
            Com->nap_bu = Com->nap_bucopy ;
            Com->nap_a2 = Com->nap_a2save = Com->nap_a2full ;
        }

        /* Uncompress the bounds, all variables are free.
           These copies only occur when the variables exist. */
        pasa_copyx (Com->lo, Com->copylo, n) ;
        pasa_copyx (Com->hi, Com->copyhi, n) ;

        if ( QP )
        {
            /* copy total gradient from user's coordinates to pasa coor */
            if ( order_with_colperm )
            {
                pasa_convert_to_pproj (g, Com->gtot, colperm, n) ;
            }
            else
            {
                pasa_copyx (g, Com->gtot, n) ;
            }
        }
        else
        {
            /* Extract the full g-vector from the user's version.
               For a quadratic cost, we get g from gtot; otherwise, we get it
               from userg. */
            if ( order_with_colperm )
            {
                pasa_convert_to_pproj (g, Com->userg, colperm, n) ;
            }
            else if ( Com->need_userx )
            {
                pasa_copyx (g, Com->userg, n) ;
            }
        }
#ifndef NDEBUG
        if ( use_pproj )
        {
            Com->location = PASA_PRE_GRAD_PROJ ;
            pasa_checkA (Com) ;
        }
#endif
    }
    Com->location = PASA_GRAD_PROJ ;

#if 0
    /* In the following cases, we project the current iterate on the
       polyhedron to ensure feasibility. First, the matrix A exists and
       use_pproj is TRUE; moreover, the prior location was either cg_descent
       or the prior location was the active gradient projection algorithm
       and the stepsize in the gradient projection step was 1. In this
       last case, the current iterate was not re-projected onto the feasible
       set, so we do it here. */
    if ( use_pproj && ((prior_location == PASA_CG_DESCENT) ||
                       ((prior_location == PASA_ACTIVE_GP) &&
                        (Com->alpha == PASAONE))) )
    {
        /* project current iterate x on the feasible set and store
           the result in xnew */
        PASAFLOAT *xnew = Com->xnew ;
        status = pasa_project1 (xnew, x, Com) ;
        if ( status >= 0 ) /* an error occurred */
        {
            return (status) ;
        }

        /* Evaluate function and gradient at projection xnew
           computed above. Note that we have the function
           value at x stored in Com->f. */
        if ( PrintLevel )
        {
            printf ("function value before projection: %e\n", Com->f) ;
        }
        pasa_evaluate_x (xnew, x, g, Com->f, Com->ncol, order, Com) ;
        if ( PrintLevel )
        {
            printf ("function value after projection: %e\n", Com->f) ;
        }
        /* copy xnew to x; note that the evaluate_x routine
           evaluates the gradient and cost at xnew and stores
           them in Com->g and Com->f */
        pasa_copyx (x, xnew, n) ;
    }
#endif

    /* Throughout grad_proj, the new function value is stored in Com->fnew,
       however, here, at the start of the code, the most recent function
       value is in Com->f. Save this value in Com->fnew in case the code
       terminates early in this routine, before evaluating a new function
       value. */
    Com->fnew = Com->f ;

    /* Evaluate starting guess for bb parameter Com->bbk */
    Com->first = TRUE ; /* first iteration */

    status = pasa_update_bb (Com) ;
    /* subsequent calls to update_bb and update_reference are handled
       differently since update_bb sets Com->first = FALSE */
    if ( status == PASA_ERROR_TOLERANCE_SATISFIED )
    {
        return (PASA_ERROR_TOLERANCE_SATISFIED) ;
    }

    /* Initialize reference value to be the current function value */
    fR = pasa_update_reference (Com) ;
    Com->first = FALSE ; /* end of bb and reference updates */

    gnew = Com->gnew ;
    d = Com->d ;
    delta = Parm->Armijo_delta ;

    /* A nonmonotone Armijo line search is performed. An acceptable stepsize
       alpha satisfies the condition

               fnew <= fR + alpha*deltaf,  deltaf = delta*gk'*dk

       where dk = pk - xk and pk = projection (xk - bbk*gk) onto
       the polyhedron.  Here bbk is the BB parameter and alpha has
       the starting value 1.  If the acceptance condition is not satisfied,
       then we continue to decrease alpha until the acceptance condition
       is satisfied. Due to numerical errors that arise when xnew is near x,
       we may replace the Armijo acceptance condition by the following
       approximation, which can be evaluated more accurately than the
       original version of the acceptance condition when xnew is near x:

                   gnew'*dk <= 2(fR-f)/alpha + deltag,

        where deltag = (2*delta - 1)*gk'*dk.  This is an accurate
        approximation to the Armijo condition when xnew is close to
        x, but when xnew is far from x, it could yield a poor
        approximation. Hence, we also require fnew <= fRpert,
        which is slightly larger than fR. */
    delta = Parm->Armijo_delta ;

    /* The reduction in alpha is done by different methods in different
       situations. One method is to multiply alpha by the factor
       armijo_decay. Another method is to approximate a local minimum of
       the function in the search direction using either a quadratic
       interpolation step or a secant step, with safe guards to ensure
       that the new alpha lies in the interval [safe0, safe1]*alpha */
    stepdecay = Parm->stepdecay ;
    safe0 = Parm->safe0 ;
    safe1 = Parm->safe1 ;

    if ( PrintLevel >= 1 )
    {
        printf ("GP starting function value: %e\n", Com->f) ;
    }
    its_since_test = 0 ; /* counts number of iterations since testing error */
    test_increment = 0 ; /* number of iterations until next test */
    it_start = Stat->gpit + 1 ;
    na = 0 ;

    while ( Stat->gpit < Parm->gpmaxit )
    {
        Stat->gpit++ ;
        if ( PrintLevel >= 1 )
        {
            printf ("Ref value: %e  BB step for grad proj: %e\n",
                     Com->fr, Com->bbk) ;
        }

        /* Project xk - bbk*gk onto the polyhedron.  The search direction
           dk = pk - xk is stored in Com->d. For a bound constrained
           problem, we obtain both the local and global errors. Hence,
           for a bound constrained problem, we can determine whether the
           stopping condition was reached, or whether to switch to the
           active set routines. If there are also linear constraints,
           the gradproj_step routine will also return an estimate for
           the global error. Hence, we can determine whether the stopping
           condition was satisfied. */
        status = pasa_gradproj_step (Com->bbk, Com) ;

        if ( status > 1 ) /* error occurred */
        {
            return (status) ;
        }

        PASAFLOAT Eest  = (Parm->scale_error_by_gmax) ?
                                                 Com->Egp/Com->gmax : Com->Egp ;
        if ( PrintLevel )
        {
            printf ("error based on gradproj: %e\n", Com->Egp) ;
        }
        if ( Eest <= Com->grad_tol )
        {
            Com->E = Eest ;
            return (PASA_ERROR_TOLERANCE_SATISFIED) ;
        }
        /* test for err stagnation in first iteration */
        if ( it_start == Stat->gpit )
        {
            E = Com->Egp ; /* set the current error bound */
            if ( it_start == 1 ) /* save current gradient error and objective */
            {
                if ( PrintLevel )
                {
                    printf ("errBest: %e costBest: %e\n", E, Com->f) ;
                }
                Com->errBest  = E ;
                Com->costBest = Com->f ;
                Com->nstagnate = 0 ;
            }
            /* if err or cost improves, then save them */
            else if ( (E < Com->errBest) || (Com->f < Com->costBest) )
            {

                if ( Com->f < Com->costBest ) Com->costBest = Com->f ;
                if (      E < Com->errBest  ) Com->errBest  = E ;
                Com->nstagnate = 0 ;
                if ( PrintLevel )
                {
                    printf ("errBest: %e costBest: %e\n", Com->errBest, Com->f);
                }
            }
            else
            {
                Com->nstagnate++ ;
                if ( Com->nstagnate >= Parm->maxstagnate )
                {
                    status = PASA_ERROR_DECAY_STAGNATES ;
                    /* set the error estimate */
                    Com->E = E ;
                    return (status) ;
                }
            }
        }
        if ( !Aexists && !GradProjOnly && (Com->e > Com->testtol) )
        {
            return (PASA_ACTIVE_GP) ;
        }
        gtd = Com->gtd ;

#if 1
        /* If gtd > 0, then current search direction not descent direction.
           Due to accumulated rounding errors in cg_descent, we may have
           drifted away from the feasible region, which can lead to this
           error. Hence, we will project the current iterate onto the
           feasible set. */
        if ( (gtd > PASAZERO) && use_pproj && (flagCG || Com->flagAG) )
        {
            if ( PrintLevel >= 1 )
            {
                printf ("gtd = %e, no descent, project iterate on feasible "
                        "set\n", gtd) ;
            }
            Stat->nbad_dir++ ;
            if ( Stat->nbad_dir >= Parm->nbad_flip_start_guess )
            {
                Com->pprojparm->start_guess = 0 ;
            }
            if ( Stat->nbad_dir >= Parm->max_bad_dir )
            {
                Com->E = Eest ;
                return (PASA_TOO_MANY_ASCENT_DIRECTIONS) ;
            }

            /* See if a descent direct can be achieved by projecting the
               current iterate on the feasible set. The projection is
               stored in xnew. */
            PASAFLOAT *xnew = Com->xnew ;
            status = pasa_project2 (xnew, x, Com) ;
            if ( status >= 0 ) /* an error occurred */
            {
                return (status) ;
            }

            /* Evaluate function and gradient at projection xnew
               computed above. Note that we have the function
               value at x stored in Com->f. */
            if ( PrintLevel )
            {
                printf ("function value before projection: %e\n", Com->f) ;
            }
            pasa_evaluate_x (xnew, x, g, Com->f, Com->ncol, order, Com) ;
            if ( PrintLevel )
            {
                printf ("function value after projection: %e\n", Com->f) ;
            }
            /* copy xnew to x; note that the evaluate_x routine
               evaluates the gradient and cost at xnew and stores
               them in Com->g and Com->f */
            pasa_copyx (x, xnew, n) ;

            status = pasa_gradproj_step (Com->bbk, Com) ;

            if ( status > 1 ) /* error occurred */
            {
                return (status) ;
            }
            Eest  = (Parm->scale_error_by_gmax) ? Com->Egp/Com->gmax : Com->Egp;
            if ( PrintLevel )
            {
                printf ("error based on gradproj: %e\n", Com->Egp) ;
            }
            if ( Eest <= Com->grad_tol )
            {
                Com->E = Eest ;
                return (PASA_ERROR_TOLERANCE_SATISFIED) ;
            }
        }
#endif
        /* the feasibility step above is not done in further iterations */
        flagCG = FALSE ;

        gtd = Com->gtd ;

        /* to deactivate Lagrangian-based line search, change "#if 1" to
           "#if 0" below */
        muAd = PASAZERO ;
#if 1
        if ( use_pproj == TRUE )
        {
            PASAINT p, q ;
            PASAFLOAT u, v, *lambda ;
            if ( Aexists == TRUE )
            {
                lambda = Com->lambda ;
                v = PASAZERO ; /* = (lambda'*A)*d */
                for (j = 0; j < n; j++)
                {
                    if ( d [j] != PASAZERO )
                    {
                        u = PASAZERO ; /* = (lambda'*A)_j */
                        q = Ap [j+1] ;
                        for (p = Ap [j]; p < q; p++)
                        {
                            u += lambda [Ai [p]]*Ax [p] ;
                        }
                        v += u*d [j] ;
                    }
                }
                muAd = v/Com->actual_step ;
            }
        }
        else if ( use_napheap )
        {
            if ( Com->nap_lambda != PASAZERO ) /* otherwise muAd = 0 (default)*/
            {
                muAd = pasa_dot (d, Com->nap_a, n)*Com->nap_lambda
                                                             /Com->actual_step ;
            }
        }
#endif
        if ( muAd <= PASAZERO )
        {
            muAd = PASAZERO ;
        }
        else
        {
            gtd -= muAd ;
        }

        /* initial stepsize in line search */
        alpha = PASAONE ;

        /* For a quadratic objective, we first determine whether a unit
           step in the search direction is acceptable. If it is accepted,
           then we take it. Otherwise, the exact minimizer in the search
           direction is used, as long as the stepsize is between 0 and 1. */
        if ( QP )
        {
            /* note that for a quadratic, the evaluation routine only updates
               the gradient, not the function value */
            pasa_evaluate (PASAZERO, &alpha, Com, "fg") ;
            if ( order_with_colperm )
            {
                dQd = PASAZERO ;
                /* dQd = pasa_dot (d, Com->Qd, n) but, Qd is in user's coor */
                for (j = 0; j < n; j++)
                {
                    dQd += d [j]*Com->Qd [colperm [j]] ;
                }
            }
            else /* no bounds, no linear constraints, no fixed variables */
            {
                dQd = pasa_dot (d, Com->Qd, n) ;
            }
            /* When dQd <= 0, then in theory, both of the acceptance conditions
               that follow should be satisfied, and hence, alpha = 1, the
               default value given above. */
            if ( dQd <= PASAZERO ) ;    /* do nothing, alpha = 1 above */
            else if ( Com->approxstep ) /* approximate Armijo search */
            {
                gtdnew = pasa_dot (gnew, d, n) ;
                if ( gtdnew-muAd > 2*(fR-Com->f) + (2*delta-1)*gtd )
                {
                    if ( Com->gtd < PASAZERO ) alpha = -Com->gtd/dQd ;
                    else                       alpha = -gtd/dQd ;
                    alpha = PASAMIN (alpha, PASAONE) ;
                }
            }
            else                        /* ordinary Armijo line search */
            {
                if ( dQd > 2*(fR-Com->f + (delta-1)*gtd) )
                {
                    if ( Com->gtd < PASAZERO ) alpha = -Com->gtd/dQd ;
                    else                       alpha = -gtd/dQd ;
                    alpha = PASAMIN (alpha, PASAONE) ;
                }
            }
            if ( alpha == PASAONE ) /* unit step is acceptable */
            {
                    /* update cost and gtot (xnew and gnew were updated in
                       pasa_evaluate) */

                Com->fnew = Com->f + 0.5*dQd + Com->gtd ;
    
                /* update total gradient */
                pasa_step (Com->gtot, Com->gtot, Com->Qd, alpha, n) ;
            }
            else /* nonunit step */
            {
                /* update cost */
                Com->fnew = Com->f + alpha*(0.5*alpha*dQd + Com->gtd) ;
   
                /* update xnew */
                pasa_step (Com->xnew, Com->x, d, alpha, n) ;
  
                /* update gradient */
                pasa_step (Com->gtot, Com->gtot, Com->Qd, alpha, n) ;
 
                if ( order_with_colperm )
                {
                    /* copy gtot into gnew, adjust for user's coordinates */
                    pasa_convert_to_pproj (gnew, Com->gtot, colperm, n) ;
                }
                else
                {
                    pasa_copyx (gnew, Com->gtot, n) ;
                }
            }
            /* Check whether to switch between ordinary and approximate Armijo
               line search */
            if ( Com->approxstep )
            {
                if ( fabs(Com->f-Com->fnew)>=Parm->ArmijoSwitchFactor*fabs(fR) )
                {
                    if ( Parm->PrintLevel >= 1 )
                    {
                        printf ("-- change to ordinary Armijo line search\n") ;
                    }
                    Com->approxstep = FALSE ;
                }
            }
            else /* ordinary Armijo line search */
            {
                if ( fabs(fR-Com->fnew) < Parm->ArmijoSwitchFactor*fabs(fR) )
                {
                    if ( PrintLevel >= 1 )
                    {
                        printf("-- change to approximate Armijo line search\n");
                    }
                    Com->approxstep = TRUE ;
                }
            }
        }
        else if ( Com->approxstep ) /* Approximate Armijo line search */
        {
            if ( PrintLevel >= 1 )
            {
                printf ("approx armijo step in grad proj\n") ;
            }
            /* stopping condition in the approximate Armijo line search is

                   gnew'*d <= 2(fR-f)/alpha + deltaf,

               where deltaf = (2*delta - 1)*gk'*dk. This is an accurate
               approximation to the Armijo condition when xnew is close to
               x, but when xnew is far from x, it could yield a poor
               approximation. Hence, we also require fnew <= fRpert,
               which is slightly larger than fR.

               If PertRule = 1, then fRpert = fR + Parm->eps*|fR|,
               if PertRule = 0, then fRpert = fR + Parm->eps */

            if ( Parm->PertRule == 1 )
            {
                pert = Parm->pert_eps*fabs(fR) ;
            }
            else
            {
                pert = Parm->pert_eps ;
            }
            fRpert = fR + pert ;
            deltaf = (2.*delta - 1.)*gtd ;
            s = 2.*(fR - Com->f) + deltaf ; /* 2*(fR-f)/alpha + deltaf */
            for (na = 0; na < maxsteps; na++)
            {
                /* Evaluate func and grad at xnew = x + alpha*d. */
                status = pasa_evaluate (PASAZERO, &alpha, Com, "fg") ;
                if ( status == PASA_FUNCTION_NAN_OR_INF )
                {
                    return (status) ;
                }

                if ( PrintLevel >= 1 )
                {
                    printf ("    na: %i deltaf: %e fR: %e fnew: %e fold: %e\n",
                             na, deltaf, fR, Com->fnew, Com->f) ;
                }
                gtdnew = pasa_dot (gnew, d, n) ;
                {
                    if ( (gtdnew != gtdnew) || (gtdnew <= -PASAINF) )
                    {
                        gtdnew = PASAINF ;
                    }
                }
                /* compute derivative of Lagrangian function at xnew*/ 
                PASAFLOAT Lgtdnew = gtdnew - muAd ;
                if ( (Lgtdnew <= s) && (Com->fnew - alpha*muAd <= fRpert) )
                {
                    break ;
                }
                if ( Lgtdnew > PASAZERO )/* newalpha computed by secant method*/
                {
                    t = gtd/(gtd - Lgtdnew) ;
                    if ( (t >= safe0) && (t <= safe1) )
                    {
                        alpha *= t ;
                    }
                    else
                    {
                        alpha *= stepdecay ;
                    }
                }
                else
                {
                    alpha *= stepdecay ;
                }
                /* change fR to f + fpert after the first iteration */
                s = 2.*pert/alpha + deltaf ;
                fRpert = Com->f + pert ;
            }
            /* Check whether to switch back to ordinary Armijo line search */
            if ( fabs(Com->f - Com->fnew) >= Parm->ArmijoSwitchFactor*fabs(fR) )
            {
                if ( Parm->PrintLevel >= 1 )
                {
                    printf ("-- change to ordinary Armijo line search\n") ;
                }
                Com->approxstep = FALSE ;
            }
        }
        else /* Ordinary Armijo line search */
        {
            if ( PrintLevel >= 1 )
            {
                printf ("ordinary armijo step\n") ;
            }
            /* stopping condition in the line search is
                   fnew <= fR + alpha*deltaf, where deltaf = delta*gk'*dk */
            deltaf = delta*gtd ;
            /* save the original function value -- if the accepted step
               yields a worse objective value, then switch to the unit step */
            for (na = 0; na < maxsteps; na++)
            {
                /* evaluate objective at xnew = x + alpha*d */
                status = pasa_evaluate (PASAZERO, &alpha, Com, "f") ;
                if ( status == PASA_FUNCTION_NAN_OR_INF )
                {
                    return (status) ;
                }
                if ( na == 0 )
                {
                    f0 = Com->fnew ;
                }
                if ( PrintLevel >= 1 )
                {
                    printf ("    na: %i deltaf: %e fR: %e fnew: %e fold: %e"
                            " alpha: %e\n",
                    na, deltaf, fR, Com->fnew, Com->f, alpha) ;
                }
                if ( Com->fnew <= fR + alpha*(deltaf+muAd) )
                {
                    break ;
                }
                fR = Com->f ; /* change fR to f after the first iter */
                /* Perform safeguarded quadratic interpolation step */
                t = 2.*(Com->fnew - alpha*(muAd+gtd) - Com->f) ;
                if ( t > PASAZERO )
                {
                    t = -gtd*alpha/t ;
                    /* if the minimizer of the quadratic outside the
                       interval [safe0, safe1]*alpha, then multiply alpha
                       by the decay factor */
                    if ( (t >= safe0) && (t <= safe1) )
                    {
                        alpha *= t ;
                    }
                    else
                    {
                       alpha *= stepdecay ;
                    }
                }
                else /* quadratic not convex, multiply alpha by decay factor */
                {
                    alpha *= stepdecay ;
                }
            }
            /* Check whether to switch to approximate Armijo line search */
            if ( fabs(fR-Com->fnew) < Parm->ArmijoSwitchFactor*fabs(fR) )
            {
                if ( PrintLevel >= 1 )
                {
                    printf ("-- change to approximate Armijo line search\n") ;
                }
                Com->approxstep = TRUE ;
            }

            /* if the objective value associated with the unit step is
               better than the accepted objective value, then switch to a
               unit step */
            if ( Com->fnew > f0 )
            {
                Com->fnew = f0 ;
                alpha = PASAONE ;
                pasa_step (Com->xnew, Com->x, Com->d, alpha, n) ;
            }
            /* evaluate grad at xnew computed above */
            t = -PASAONE ;
            pasa_evaluate (PASAZERO, &t, Com, "g") ;
        }
        if ( PrintLevel >= 1 )
        {
            printf ("GP iter: %ld na: %i fnew: %25.15e fold: %25.15e "
                    "alpha: %e\n",
                    (LONG) Stat->gpit, na, Com->fnew, Com->f, alpha) ;
        }

        if ( Stat->gpit >= Parm->gpmaxit )
        {
            Com->f = Com->fnew ;
            pasa_copyx (Com->x, Com->xnew, n) ;
            pasa_copyx (Com->g, Com->gnew, n) ;
            break ;
        }

        if ( na == maxsteps )
        {
            return (PASA_LINE_SEARCH_STEPS_EXCEED_MAXSTEPS_IN_GRAD_PROJ) ;
        }

        /* bound constraints only */
        if ( !Aexists && !GradProjOnly && (Com->e > Com->testtol) )
        {
            return (PASA_ACTIVE_GP) ;
        }

        Com->alpha = alpha ;
        if ( alpha == PASAONE )
        {
            Com->unitstep = TRUE ;
        }
        else
        {
            Com->unitstep = FALSE ;
        }

        /* update the status of the linear constraint if napheap is used */
        if ( (use_napheap == TRUE) && (Com->nap_constraint < 2) )
        {
            if ( Com->unitstep == TRUE )
            {
                if ( Com->nap_lambda == PASAZERO )
                {
                    Com->nap_constraint = 0 ; /* strict inequality */
                }
                else if ( Com->nap_lambda > PASAZERO )
                {
                    Com->nap_constraint = -1 ;
                }
                else /* nap_lambda < 0 */
                {
                    Com->nap_constraint = 1 ;
                }
            }
            else /* unitstep is FALSE */
            {
                if ( Com->nap_lambda == PASAZERO )
                {
                    Com->nap_constraint = 0 ;
                }
                else if ( Com->nap_lambda < PASAZERO )
                {
                    /* if prior > 0, then no change, otherwise set to 0 */
                    if ( Com->nap_constraint <= 0 )
                    {
                        Com->nap_constraint = 0 ;
                    }
                }
                else /* nap_lambda > 0 */
                {
                    /* if prior < 0, then no change, otherwise set to 0 */
                    if ( Com->nap_constraint >= 0 )
                    {
                        Com->nap_constraint = 0 ;
                    }
                }
            }
        }

        /* update the BB parameter if necessary */
        status = pasa_update_bb (Com) ;

        pasa_copyx (Com->x, Com->xnew, n) ;
        pasa_copyx (Com->g, Com->gnew, n) ;
#ifndef NDEBUG
        if ( alpha < PASAONE ) /* test error decay */
        {
            if ( Com->f != PASAZERO ) /* test relative error */
            {
                if ( (Com->fnew - Com->f)/fabs (Com->f) > Parm->debugtol )
                {
                    printf ("old cost %25.15e < new cost %25.15e\n",
                              Com->f, Com->fnew) ;
                    pasa_error (-1, __FILE__, __LINE__, "stop") ;
                }
            }
        }
        Com->f = Com->fnew ;
        pasa_check_feas (Com) ;
#endif
        Com->f = Com->fnew ;

        /* test the error if the criteria are satisfied */
        its_since_test++ ;
        if ( PrintLevel > 1 )
        {
            printf ("its_since_test: %i test_increment: %i\n",
                    its_since_test, test_increment) ;
        }
        /* Determine whether the local and global error should be estimated. */
        if ( Aexists && (its_since_test >= test_increment) && !GradProjOnly )
        {
            /* The factorization at the projection point corresponds to
               alpha = 1. When alpha < 1, the current pproj factorization
               may not correspond to the current iterate x. By projecting
               the current iterate on the feasible set, we will update
               the factorization so as to correspond to the current iterate. */
            if ( use_pproj && !Com->unitstep )
            {
                /* project current iterate x on the feasible set and store
                   the result in xnew */
                PASAFLOAT *xnew = Com->xnew ;
                status = pasa_project2 (xnew, x, Com) ;
                if ( status >= 0 ) /* an error occurred */
                {
                    return (status) ;
                }

                /* Evaluate function and gradient at projection xnew
                   computed above. Note that we have the function
                   value at x stored in Com->f. */
                if ( PrintLevel )
                {
                    printf ("function value before projection: %e\n", Com->f) ;
                }
                pasa_evaluate_x (xnew, x, g, Com->f, Com->ncol, order, Com) ;
                if ( PrintLevel )
                {
                    printf ("function value after projection: %e\n", Com->f) ;
                }
                /* copy xnew to x; note that the evaluate_x routine
                   evaluates the gradient and cost at xnew and stores
                   them in Com->g and Com->f */
                pasa_copyx (x, xnew, n) ;
                /* The factorization now corresponds to the current x so
                   set unitstep = TRUE. */
                Com->unitstep = TRUE ;
            }

            /* estimate the local error Com->e by projecting gradient into
               the null space associated with current active constraints. */
            pasa_null_project (NULL, g, NULL, TRUE, Com) ;

            /* Estimate the global error and test for convergence */
            status = pasa_check_error (Com) ;

            if ( status == PASA_ERROR_TOLERANCE_SATISFIED ) return (status);

#if 0
            /* after one iteration in gradproj, branch to active gradproj */
            status = PASA_ACTIVE_GP ;
#endif
            its_since_test = 0 ;
            test_increment++ ;
            test_increment = PASAMIN (test_increment, Parm->testit) ;

            if ( status == PASA_ACTIVE_GP )
            {
                return (status) ;
            }
            /* else another iteration of grad_proj is performed, reduce
               switchfactor to make it easier to branch to phase two */
            t = Com->switchfactor*Parm->switchdecay ;
            if ( t < Parm->switchlower )
            {
                Com->testtol *= Parm->switchlower/Com->switchfactor ;
                Com->switchfactor = Parm->switchlower ;
            }
            else
            {
                Com->testtol *= Parm->switchdecay ;
                Com->switchfactor = t ;
            }
            Com->testtol = PASAMAX (Com->testtol, Com->grad_tol) ;
        }

        /* update the reference value */
        fR = pasa_update_reference (Com) ;
        if ( fR < Com->f ) fR = Com->f ;
    }

    return (PASA_ITERATIONS_EXCEED_MAXITS_IN_GRAD_PROJ) ;
}

/* ==========================================================================
   === pasa_activeGP ========================================================
   ==========================================================================
    Perform an active set version of the gradient projection algorithm
    until either "local error e < switchfactor*global error E",
    or the number of iterations since there is an addition to the number
    of active constraints exceeds Parm->terminate_agp. In the active
    set gradient projection algorithm, the variables at bounds at the
    current iterate xk, are kept fixed at their bounds in the next iteration,
    while the inequalities Bl <= Ax <= Bu that are active at xk are kept
    active when projecting onto the polyhedron.
   ========================================================================== */
int pasa_activeGP /* return:
                       PASA_GRAD_PROJ
                       PASA_CG_DESCENT
                       PASA_ERROR_TOLERANCE_SATISFIED
                       PASA_POLYHEDRON_INFEASIBLE
                       PASA_INVALID_LINEAR_CONSTRAINT_BOUNDS
                       PASA_LINE_SEARCH_STEPS_EXCEED_MAXSTEPS_IN_ACTIVEGP
                       PASA_OUT_OF_MEMORY_IN_PPROJ
                       PASA_SSOR_ASCENT_FAILURE_IN_PPROJ
                       PASA_SEARCH_DIRECTION_NOT_DESCENT_DIRECTION_IN_ACTIVEGP
                       PASA_SSOR_ITERATIONS_EXCEED_MAX_ITS_IN_PPROJ */
(
    PASAcom *Com  /* common data for PASA */
)
{
    int its_since_test, test_increment, na, PrintLevel, ProblemCompressed,
         status, use_pproj, use_napheap, napchange ;
    PASAFLOAT alpha, delta, deltaf, dQd, f0, fR, fRpert,
              gtd, gtdnew, maxsteps, muAd, pert, s, safe0, safe1, stepdecay, t,
              *d, *g, *gnew, *x, *xnew ;
    PASAINT iter_nochange, j, nf, ni0, *ifree, *order ;
    PASAparm *Parm ;
    PASAstat *Stat ;
    PPprob *Prob ;

    Parm = Com->pasaparm ;  /* parameters for PASA */
    PrintLevel = Parm->PrintLevel ;
    if ( PrintLevel >= 1 )
    {
        printf ("START activeGP, testtol: %e\n", Com->testtol) ;
    }
    /* if there are no free variables, then no active gradient project
       iteration is possible, we need to branch to gradproj, free variables
       and estimate the error */
    if ( !Com->nf ) return (PASA_GRAD_PROJ) ;

    int const    Bounds = Com->Bounds ;
    int const   Aexists = Com->Aexists ;
    int const  location = Com->location ;
    int const        QP = Com->QP ;
    /* nfull = number of components of x after removal of fixed variables */
    PASAINT const nfull = Com->ncol ;
    ifree = Com->ifree ;
    order = Com->order ;
    use_pproj = Com->use_pproj ;
    use_napheap = Com->use_napheap ;
    Com->flagAG = FALSE ; /* currently, search direction not ascent direction */

    /* depending on where we are coming from, the startup process is slightly
       different */
    if ( location == PASA_CG_DESCENT ) /* coming from CG (new active) */
    {
        if ( PrintLevel >= 1 )
        {
            printf ("coming from CG\n") ;
        }
        ProblemCompressed = TRUE ;

        /* if the code comes from CG and an estimate for the bb parameter
           was obtained, then we use it */
        if ( Com->cg_bb_est > PASAZERO )
        {
            Com->bbk = Com->cg_bb_est ;
        }
    }
    else /* coming from grad_proj */
    {
        if ( PrintLevel >= 1 )
        {
            printf ("coming from grad_proj\n") ;
        }
        ProblemCompressed = FALSE ;
        /* if bounds exist, then set ifree */
        if ( Bounds )
        {
            if ( Com->order_with_colperm )
            {
                /* colperm stores the indices in the user's coordinate system
                   of the unfixed variables */
                pasa_copyi (ifree, Com->colperm, nfull) ;
            }
            else
            {
                /* all the variables are free initially */
                for (j = 0; j < nfull; j++)
                {
                    ifree [j] = j ;
                }
            }
        }
        /* Only compress variables and inequalities if coming from grad_proj. */
        if ( use_pproj || use_napheap || (Com->Bounds && !Aexists) )
        {
            Com->location = PASA_ACTIVE_GP ;
            pasa_compress_prob (FALSE, Com) ;
            ProblemCompressed = TRUE ;
#ifndef NDEBUG
            if ( use_pproj ) pasa_checkA (Com) ;
#endif
        }
    }

    /* update Com->location to the active set gradient projection algorithm */
    Com->location = PASA_ACTIVE_GP ;

    if ( use_pproj ) Prob = Com->ppcom->Prob ;
    x    = Com->x ;
    xnew = Com->xnew ;
    gnew = Com->gnew ;
    g    = Com->g ;
    d    = Com->d ;
    Stat = Com->pasastat ; /* statistics for PASA */
    nf   = Com->nf ;

    /* Evaluate starting guess for bb parameter Com->bbk */
    Com->first = TRUE ; /* first iteration */
    status = pasa_update_bb (Com) ;
    /* subsequent calls to update_bb and update_reference are handled
       differently since update_bb sets Com->first = FALSE  */

    if ( status == PASA_ERROR_TOLERANCE_SATISFIED )
    {
        return (PASA_ERROR_TOLERANCE_SATISFIED) ;
    }

    /* Initialize reference value to be the current function value */
    fR = pasa_update_reference (Com) ;
    Com->first = FALSE ; /* end of first iteration */

    /* A nonmonotone Armijo line search is performed. An acceptable stepsize
       alpha satisfies the condition

               fnew <= fR + alpha*deltaf,  deltaf = delta*gk'*dk

       where dk = pk - xk and pk = projection (xk - bbk*gk) onto
       the polyhedron.  Here bbk is the BB parameter and alpha has
       the starting value 1.  If the acceptance condition is not satisfied,
       then we continue to decrease alpha until the acceptance condition
       is satisfied. Due to numerical errors that arise when xnew is near x,
       we may replace the Armijo acceptance condition by the following
       approximation, which can be evaluated more accurately than the
       original version of the acceptance condition when xnew is near x:

                   gnew'*dk <= 2(fR-f)/alpha + deltag,

        where deltag = (2*delta - 1)*gk'*dk.  This is an accurate
        approximation to the Armijo condition when xnew is close to
        x, but when xnew is far from x, it could yield a poor
        approximation. Hence, we also require fnew <= fRpert,
        which is slightly larger than fR. */
    delta = Parm->Armijo_delta ;
    maxsteps = Parm->maxsteps ; /* max number of Armijo backsteps */

    /* The reduction in alpha is done by different methods in different
       situations. One method is to multiply alpha by the factor
       armijo_decay. Another method is to approximate a local minimum of
       the function in the search direction using either a quadratic
       interpolation step or a secant step, with safe guards to ensure
       that the new alpha lies in the interval [safe0, safe1]*alpha */
    stepdecay = Parm->stepdecay ;
    safe0 = Parm->safe0 ;
    safe1 = Parm->safe1 ;

    if ( PrintLevel >= 1 )
    {
        printf ("active set GP Starting function value: %e\n", Com->f) ;
    }
    na = 0 ;
    iter_nochange = 0 ;
    its_since_test = 0 ; /* counts number of iterations since testing error */
    test_increment = 2 ; /* number of iterations until next test */


    while ( iter_nochange < Parm->terminate_agp )
    {
        if ( PrintLevel >= 1 )
        {
            printf("active GP Ref value: %e BB step for grad proj: %e\n",
                    Com->fr, Com->bbk) ;
        }

        /* Project xk - bbk*gk onto the polyhedron. The search direction
           dk = pk - xk is stored in Com->d. If A does not exist (only
           bound constraints are present), then we also compute global
           and local errors. */
        status = pasa_gradproj_step (Com->bbk, Com) ;
        if ( status >= 1 )
        {
            if ( status == PASA_POLYHEDRON_INFEASIBLE ) /* too many active */
            {
                return (PASA_GRAD_PROJ) ;
            }
            return (status) ;
        }

        Stat->agpit++ ;

        /* use ||d|| to estimate the local error at current iterate.
           We obtain an upper bound for the local error. */
        t = pasa_sup_normx (d, nf) ;
        if ( Com->actual_step < PASAONE ) t /= Com->actual_step ;
        if ( t < Com->e ) Com->e = t ;
        if ( PrintLevel )
        {
            printf ("local error: %e\n", Com->e) ;
        }
        /* if the local error is smaller than the test tolerance, then
           obtain a bound for the global error and update test tolerance.
           If the global error is small enough, then terminate. If the
           local error is smaller than the test tolerance, then branch to
           the gradient projection algorithm. */
        if ( Com->e < Com->testtol )
        {
            status = pasa_check_error (Com) ;
            if ( (status == PASA_ERROR_TOLERANCE_SATISFIED) ||
                 (status == PASA_GRAD_PROJ) )
            {
                if ( (status == PASA_GRAD_PROJ) && (Com->gtd > PASAZERO) )
                {
                    Com->flagAG = TRUE ;
                }
                return (status) ;
            }
        }

        /* to deactivate Lagrangian-based line search, change "if 1" to "if 0"*/
        muAd = PASAZERO ;
#if 1
        if ( use_pproj )
        {
            PASAINT   const *Ap = (use_pproj) ? Prob->Ap : NULL ;
            PASAINT   const *Ai = (use_pproj) ? Prob->Ai : NULL ;
            PASAFLOAT const *Ax = (use_pproj) ? Prob->Ax : NULL ;
            PASAFLOAT u, v, *lambda ;
            PASAINT p, q ;
            if ( Aexists )
            {
                lambda = Com->lambda ;
                v = PASAZERO ; /* = (lambda'*A)*d */
                for (j = 0; j < nf; j++)
                {
                    if ( d [j] != PASAZERO )
                    {
                        u = PASAZERO ; /* = (lambda'*A)_j */
                        q = Ap [j+1] ;
                        for (p = Ap [j]; p < q; p++)
                        {
                            u += lambda [Ai [p]]*Ax [p] ;
                        }
                        v += u*d [j] ;
                    }
                }
                muAd = v/Com->actual_step ;
            }
        }
        else if ( use_napheap )
        {
            if ( Com->nap_lambda != PASAZERO ) /* otherwise muAd = 0 (default)*/
            {
                muAd = Com->nap_lambda*pasa_dot (d, Com->nap_a, nf)
                                                             /Com->actual_step ;
            }
        }
#endif
        gtd = Com->gtd ;
        if ( muAd <= PASAZERO )
        {
            muAd = PASAZERO ;
        }
        else
        {
            gtd -= muAd ;
        }

        if ( PrintLevel && (gtd > PASAZERO) )
        {
            printf ("gtd = %e > 0, ||g|| = %e ||d|| = %e, "
                    "branch to grad_proj\n",
                     gtd, pasa_sup_normx (g, nf), pasa_sup_normx (d, nf)) ;
        }
        /* If gtd > 0, then current search direction not descent direction.
           Either the current iterate is infeasible, or we have reached
           a stationary point of the active set subproblem. */
        if ( gtd >= PASAZERO )
        {
            Com->flagAG = TRUE ;
            return (PASA_GRAD_PROJ) ;
        }
#if 0
        if ( gtd >= PASAZERO )
        {
            if ( PrintLevel >= 1 )
            {
                printf ("gtd = %e, no descent, project iterate on feasible "
                        "set\n", gtd) ;
            }
            /* project current iterate x onto feasible set and store the
               projection in xnew */
            status = pasa_project2 (xnew, x, Com) ;
            if ( status >= 0 ) /* an error occurred */
            {
                return (status) ;
            }

            /* Evaluate function and gradient at projection xnew
               computed above. Note that we have the function
               value at x stored in Com->f. */
            if ( PrintLevel )
            {
                printf ("function value before projection: %e\n", Com->f) ;
            }
            pasa_evaluate_x (xnew, x, g, Com->f, nf, order, Com) ;
            if ( PrintLevel )
            {
                printf ("function value after projection: %e\n", Com->f) ;
            }
            /* copy xnew to x; note that the evaluate_x routine
               evaluates the gradient and cost at xnew and stores
               them in Com->g and Com->f */
            pasa_copyx (x, xnew, nf) ;
            /* Attempt another projection of xk - bbk*gk onto the polyhedron.
               The search direction dk = pk - xk is stored in Com->d. */
            status = pasa_gradproj_step (Com->bbk, Com) ;
            if ( status > 1 )
            {
                return (status) ;
            }
            gtd = Com->gtd ;
            /* if the search direction continues to be an ascent direction,
               then return to the gradient projection routine */
            if ( gtd >= PASAZERO ) return (PASA_GRAD_PROJ) ;
            /* otherwise, perform the step */
        }
#endif
 
        /* initial stepsize in line search (backtrack along search direction) */
        alpha = PASAONE ;
        PASAINT nf0 = nf ;
        if ( use_pproj ) ni0 = Prob->ni ;

        /* For a quadratic objective, we first determine whether a unit
           step in the search direction is acceptable. If it is accepted,
           then we take it. Otherwise, the exact minimizer in the search
           direction is used, as long as the stepsize is between 0 and 1. */
        if ( QP )
        {
            /* note that for a quadratic, the evaluation routine only computes
               the product Q*d between the Hessian and the search direction
               Com->d. */
            pasa_evaluate (PASAZERO, &alpha, Com, "fg") ;
            if ( order != NULL )
            {
                dQd = PASAZERO ;
                /* dQd = pasa_dot (d, Com->Qd, n) but, Qd is in user's coor */
                for (j = 0; j < nf; j++)
                {
                    dQd += d [j]*Com->Qd [order [j]] ;
                }
            }
            else /* no bounds, no linear constraints, no fixed variables */
            {
                dQd = pasa_dot (d, Com->Qd, nf) ;
            }
            /* When dQd <= 0, then in theory, both of the acceptance conditions
               that follow should be satisfied, and hence, alpha = 1, the
               default value given above. */
            if ( dQd <= PASAZERO ) ; /* do nothing, alpha = 1 above */
            else if ( Com->approxstep == TRUE ) /* approximate Armijo search */
            {
                gtdnew = pasa_dot (gnew, d, nf) ;
                if ( gtdnew-muAd > 2*(fR-Com->f) + (2*delta-1)*gtd )
                {
                    if ( Com->gtd < PASAZERO ) alpha = -Com->gtd/dQd ;
                    else                       alpha = -gtd/dQd ;
                    alpha = PASAMIN (alpha, PASAONE) ;
                }
            }
            else                              /* ordinary Armijo line search */
            {
                if ( dQd > 2*(fR-Com->f + (delta-1)*Com->gtd) )
                {
                    if ( Com->gtd < PASAZERO ) alpha = -Com->gtd/dQd ;
                    else                       alpha = -gtd/dQd ;
                    alpha = PASAMIN (alpha, PASAONE) ;
                }
            }
            if ( alpha == PASAONE ) /* unit step is acceptable */
            {
                    /* update cost and gtot (xnew and gnew were updated in
                       pasa_evaluate) */

                Com->fnew = Com->f + 0.5*dQd + Com->gtd ;

                /* update total gradient */
                pasa_step (Com->gtot, Com->gtot, Com->Qd, alpha, nfull) ;
            }
            else /* nonunit step */
            {
                /* update cost */
                Com->fnew = Com->f + alpha*(0.5*alpha*dQd + Com->gtd) ;

                /* update xnew */
                pasa_step (xnew, x, d, alpha, nf) ;

                /* update gradient */
                pasa_step (Com->gtot, Com->gtot, Com->Qd, alpha, nfull) ;
    
                if ( order != NULL )
                {
                    /* copy gtot into gnew, adjust for user's coordinates */
                    pasa_convert_to_pproj (gnew, Com->gtot, order, nf) ;
                }
                else
                {
                    pasa_copyx (gnew, Com->gtot, nf) ;
                }
            }
            /* Check whether to switch between ordinary and approximate Armijo
               line search */
            if ( Com->approxstep )
            {
                if ( fabs(Com->f-Com->fnew)>=Parm->ArmijoSwitchFactor*fabs(fR) )
                {
                    if ( Parm->PrintLevel >= 1 )
                    {
                        printf ("-- change to ordinary Armijo line search\n") ;
                    }
                    Com->approxstep = FALSE ;
                }
            }
            else /* ordinary Armijo line search */
            {
                if ( fabs(fR-Com->fnew) < Parm->ArmijoSwitchFactor*fabs(fR) )
                {
                    if ( PrintLevel >= 1 )
                    {
                        printf("-- change to approximate Armijo line search\n");
                    }
                    Com->approxstep = TRUE ;
                }
            }
        }
        else if ( Com->approxstep ) /* approximate Armijo line search */
        {
            if ( PrintLevel >= 1 )
            {
                printf ("approx armijo step in grad proj\n") ;
            }
            /* stopping condition in the approximate Armijo line search is

                   gnew'*d <= 2(fR-f)/alpha + deltaf,

               where deltaf = (2*delta - 1)*gk'*dk. This is an accurate
               approximation to the Armijo condition when xnew is close to
               x, but when xnew is far from x, it could yield a poor
               approximation. Hence, we also require fnew <= fRpert,
               which is slightly larger than fR

               If PertRule = 1, then fRpert = fR + Parm->eps*|fR|,
               if PertRule = 0, then fRpert = fR + Parm->eps */
            if ( Parm->PertRule == 1 )
            {
                pert = Parm->pert_eps*fabs(fR) ;
            }
            else
            {
                pert = Parm->pert_eps ;
            }
            fRpert = fR + pert ;
            deltaf = (2.*delta - 1.)*gtd ;
            s = 2.*(fR - Com->f) + deltaf ; /* 2*(fR-f)/alpha + deltaf */
            for (na = 0; na < maxsteps; na++)
            {
                /* Evaluate func and grad at xnew = x + alpha*d. */
                status = pasa_evaluate (PASAZERO, &alpha, Com, "fg") ;
                if ( status == PASA_FUNCTION_NAN_OR_INF )
                {
                    return (status) ;
                }
                if ( PrintLevel >= 1 )
                {
                    printf ("    na: %i deltaf: %e fR: %e fnew: %e fold: %e\n",
                             na, deltaf, fR, Com->fnew, Com->f) ;
                }
                gtdnew = pasa_dot (gnew, d, nf) ;
                if ( (gtdnew != gtdnew) || (gtdnew <= -PASAINF) )
                {
                    gtdnew = PASAINF ;
                }
                PASAFLOAT Lgtdnew = gtdnew - muAd ;
                if ( (Lgtdnew <= s) && (Com->fnew - alpha*muAd <= fRpert) )
                {
                    break ;
                }
                if ( Lgtdnew > PASAZERO )/* newalpha computed by secant method*/
                {
                    t = gtd/(gtd - Lgtdnew) ;
                    if ( (t >= safe0) && (t <= safe1) )
                    {
                        alpha *= t ;
                    }
                    else
                    {
                        alpha *= stepdecay ;
                    }
                }
                else
                {
                    alpha *= stepdecay ;
                }
                /* change fR to f + fpert after the first iteration */
                s = 2.*pert/alpha + deltaf ;
                fRpert = Com->f + pert ;
            }
            /* Check whether to switch back to ordinary Armijo line search */
            if ( fabs(Com->f - Com->fnew) >= Parm->ArmijoSwitchFactor*fabs(fR) )
            {
                if ( Parm->PrintLevel >= 1 )
                {
                    printf ("-- change to ordinary Armijo line search\n") ;
                }
                Com->approxstep = FALSE ;
            }
        }
        else /* Ordinary Armijo line search */
        {
            if ( PrintLevel >= 1 )
            {
                printf ("ordinary armijo step\n") ;
            }
            /* stopping condition in the line search is
                   fnew <= fR + alpha*deltaf, where deltaf = delta*gk'*dk */
            deltaf = delta*gtd ;
            for (na = 0; na < maxsteps; na++)
            {
                /* evaluate objective at xnew = x + alpha*d */
                status = pasa_evaluate (PASAZERO, &alpha, Com, "f") ;
                if ( status == PASA_FUNCTION_NAN_OR_INF )
                {
                    return (status) ;
                }

                if ( na == 0 )
                {
                    f0 = Com->fnew ;
                }

                if ( PrintLevel >= 1 )
                {
                    printf ("    na: %i deltaf: %e fR: %e fnew: %e fold: %e\n",
                             na, deltaf, fR, Com->fnew, Com->f) ;
                }
                if ( Com->fnew <= fR + alpha*(deltaf+muAd) )
                {
                    break ;
                }
                fR = Com->f ; /* change fR to f after the first iter */
                /* Perform safeguarded quadratic interpolation step */
                t = 2.*(Com->fnew - alpha*(muAd+gtd) - Com->f) ;
                if ( t > PASAZERO )
                {
                    t = -gtd*alpha/t ;
                    /* if the minimizer of the quadratic outside the
                       interval [safe0, safe1]*alpha, then multiply alpha
                       by the decay factor */
                    if ( (t >= safe0) && (t <= safe1) )
                    {
                        alpha *= t ;
                    }
                    else
                    {
                       alpha *= stepdecay ;
                    }
                }
                else /* quadratic not convex, multiply alpha by decay factor */
                {
                    t = -gtd*alpha/t ;
                    alpha *= stepdecay ;
                }
            }
            /* Check whether to switch to approximate Armijo line search */
            if ( fabs(fR-Com->fnew) < Parm->ArmijoSwitchFactor*fabs(fR) )
            {
                if ( PrintLevel >= 1 )
                {
                    printf ("-- change to approximate Armijo line search\n") ;
                }
                Com->approxstep = TRUE ;
            }
            if ( Com->fnew > f0 )
            {
                Com->fnew = f0 ;
                alpha = PASAONE ;
                pasa_step (Com->xnew, Com->x, Com->d, alpha, nf) ;
            }
            /* evaluate grad at xnew computed above */
            t = -PASAONE ;
            pasa_evaluate (PASAZERO, &t, Com, "g") ;
        }
        if ( PrintLevel >= 1 )
        {
            printf ("AG iter: %ld na: %i fnew: %25.15e fold: %25.15e "
                    "alpha: %e Com->Egp: %e Com->E1 %e\n", (LONG) Stat->agpit,
                     na, Com->fnew, Com->f, alpha, Com->Egp, Com->E1) ;
        }
        if ( na == maxsteps )
        {
            return (PASA_LINE_SEARCH_STEPS_EXCEED_MAXSTEPS_IN_ACTIVEGP) ;
        }
        if ( Stat->agpit >= Parm->agpmaxit )
        {
            return (PASA_ITERATIONS_EXCEED_MAXITS_IN_ACTIVE_GRAD_PROJ) ;
        }
        Com->alpha = alpha ;
        iter_nochange++ ;

        /* update the BB parameter if necessary */
        status = pasa_update_bb (Com) ;
        if ( status == PASA_ERROR_TOLERANCE_SATISFIED )
        {
            return (status) ;
        }

#ifndef NDEBUG
        if ( na > 0 ) /* test error decay */
        {
            if ( Com->f != PASAZERO ) /* test relative error */
            {
                if ( (Com->fnew - Com->f)/fabs (Com->f) > Parm->debugtol )
                {
                    printf ("old cost %25.15e < new cost %25.15e\n",
                              Com->f, Com->fnew) ;
                    pasa_error (-1, __FILE__, __LINE__, "stop") ;
                }
            }
        }
        Com->f = Com->fnew ;
        pasa_copyx (x, Com->xnew, nf) ;
        pasa_copyx (g, Com->gnew, nf) ;
        pasa_check_feas (Com) ;
#endif
        pasa_copyx (x, Com->xnew, nf) ;
        pasa_copyx (g, Com->gnew, nf) ;
        Com->f = Com->fnew ;

        if ( use_pproj ) /* either use_pproj or use_napheap */
        {
            /* The factorization at the projection point corresponds to
               alpha = 1. When alpha < 1, the current pproj factorization
               may not correspond to the current iterate x. By projecting
               the current iterate on the feasible set, we will update
               the factorization so as to correspond to the current iterate. */
            if ( alpha < PASAONE )
            {
                /* project current iterate x on the feasible set and store
                   the result in xnew */
                status = pasa_project2 (xnew, x, Com) ;
                if ( status >= 0 ) /* an error occurred */
                {
                    if ( status == PASA_POLYHEDRON_INFEASIBLE )
                    {
                        return (PASA_GRAD_PROJ) ;
                    }
                    return (status) ;
                }

                /* Evaluate function and gradient at projection xnew
                   computed above. Note that we have the function
                   value at x stored in Com->f. */
                if ( PrintLevel )
                {
                    printf ("function value before projection: %e\n", Com->f) ;
                }
                pasa_evaluate_x (xnew, x, g, Com->f, nf, order, Com) ;
                if ( PrintLevel )
                {
                    printf ("function value after projection: %e\n", Com->f) ;
                }
                /* copy xnew to x; note that the evaluate_x routine
                   evaluates the gradient and cost at xnew and stores
                   them in Com->g and Com->f */
                pasa_copyx (x, xnew, nf) ;
            }
            status = pasa_compress_prob (TRUE, Com) ;
#ifndef NDEBUG
            if ( use_pproj ) pasa_checkA (Com) ;
#endif
        }

        else if ( use_napheap )
        {
            /* When napheap is used and constraint inactive, update status,
               save the lambda computed in napheap. When alpha < 1, no change
               in constraint status. */
            napchange = FALSE ;
            if ( alpha == PASAONE )
            {
                if ( Com->nap_constraint == 0)
                {
                    if ( Com->nap_lambda > PASAZERO )
                    {
                        Com->nap_constraint = -1 ;
                        napchange = TRUE ;
                    }
                    else if ( Com->nap_lambda < PASAZERO )
                    {
                        Com->nap_constraint = 1 ;
                        napchange = TRUE ;
                    }
                }
            }
            if ( (alpha == PASAONE) || !ProblemCompressed )
            {
                status = pasa_compress_prob (TRUE, Com) ;
            }
        }
        else if ( !Aexists && Com->Bounds
                           && ((alpha == PASAONE) || !ProblemCompressed) )
        {
            status = pasa_compress_prob (TRUE, Com) ;
        }
        ProblemCompressed = TRUE ;

        nf = Com->nf ; /* number of free variables after compression */
        /* if there are no free variables, then no active gradient project
        iteration is possible, we need to branch to gradproj, free variables
        and estimate the error */
        if ( !nf ) return (PASA_GRAD_PROJ) ;

        /* if new active constraints found, then reset iter_nochange = 0 */
        if ( (nf < nf0) || (use_pproj && (Prob->ni < ni0))
                        || (use_napheap && napchange) )
        {
            iter_nochange = 0 ;
        }
        if ( PrintLevel )
        {
            printf ("new free variables: %ld old: %ld ",
                   (LONG) nf, (LONG) nf0) ;
            if ( use_pproj )
            {
                printf ("new free rows: %ld old: %ld",
                       (LONG) Prob->ni, (LONG) ni0) ;
            }
            printf ("\n") ;
        }

        /* when the number of iterations since a change in the constraint
           activity reaches terminate_agp, break and branch to cg_descent */
        if ( iter_nochange >= Parm->terminate_agp ) break ;

        /* update the reference value */
        fR = pasa_update_reference (Com) ;
        if ( fR < Com->f ) fR = Com->f ;

        /* test the error if the criteria are satisfied */
        its_since_test++ ;
        if ( PrintLevel > 1 )
        {
            printf ("its_since_test: %i test_increment: %i\n",
                    its_since_test, test_increment) ;
        }
        /* Compute local and global error when its_since_test large enough. */
        if ( its_since_test >= test_increment )
        {
            its_since_test = 0 ;
            test_increment++ ;
            test_increment = PASAMIN (test_increment, Parm->testit) ;

            /* Estimate the local error Com->e by projecting gradient into
               the null space associated with current active constraints. */
            pasa_null_project (NULL, g, NULL, TRUE, Com) ;

            /* Obtain a bound for the global error */
            status = pasa_check_error (Com) ;

            if ( (status == PASA_GRAD_PROJ) ||
                 (status == PASA_ERROR_TOLERANCE_SATISFIED) )
            {
                return (status) ;
            }
        }
    }
#if 0
    /* If alpha < 1, then no addition to the active set. The effect
       on the factorization associated with any increase in the active
       set at step alpha needs to be neutralized. */
    if ( (alpha < PASAONE) && (use_pproj == TRUE) )
    {
        int    *ib ;
        PASAINT col, i, l, k, ncol, ncoladd, nrowdel, ni, row, p, q, *ineq_row,
               *AFTp, *AFTnz,*AFTi, *Anz, *Cp, *Cnz, *F, *ir,
               *RLinkDn, *RLinkUp, *RowmodFlag, *RowmodList,
               *ColmodFlag, *ColmodList ;
        PASAFLOAT *AFTx, *lambda, *shift_l ;
        PPwork *Work ;

        Work = Com->ppcom->Work ;
        ncol = Prob->ncol ;
        ni = Prob->ni ;
        ineq_row = Prob->ineq_row ;
        Anz = Prob->Anz ;
        Ap = Prob->Ap ;
        Ax = Prob->Ax ;
        AFTp = Work->AFTp ;
        AFTi = Work->AFTi ;
        AFTnz = Work->AFTnz ;
        AFTx = Work->AFTx ;
        Cp = Work->Cp ;
        Cnz = Work->Cnz ;

        Work = Com->ppcom->Work ;
        ib = Work->ib ;
        ir = Work->ir ;
        lambda = Work->lambda ;
        RowmodFlag = Work->RowmodFlag ;
        RowmodList = Work->RowmodList ;
        ColmodFlag = Work->ColmodFlag ;
        ColmodList = Work->ColmodList ;
        /* list of free variables */
        F = Work->F ;
        /* number of free variables */
        nf = Work->nf ;
        /* Links point to active rows */
        RLinkUp = Work->RLinkUp ;
        RLinkDn = Work->RLinkDn ;
        lambda = Work->lambda ;
        shift_l = Work->shift_l ;
        ASSERT (Work->ncoladd == 0) ;
        ASSERT (Work->ncoldel == 0) ;
        ASSERT (Work->nrowadd == 0) ;
        ASSERT (Work->nrowdel == 0) ;
        ncoladd = 0 ;
        nrowdel = 0 ;
        if ( Com->Bounds == TRUE )
        {
            for (j = 0; j < ncol; j++)
            {
                /* if column was dropped, then it needs to be added */
                if ( ib [j] != 0 )
                {
                    /* add the column since the constraint is free at x [j] */
                    ASSERT (ColmodFlag [j] == EMPTY) ;
                    ib [j] = 0 ;
                    F [nf++] = j ;
                    ColmodList [ncoladd] = j ;
                    ColmodFlag [j] = ncoladd ;
                    ncoladd++ ;
                    q = Ap [j+1] ;
                    for (p = Ap [j]; p < q; p++)
                    {
                        i = Ai [p] ;
                        if ( ir [i] <= ni )
                        {
                            l = AFTp [i] + AFTnz [i]++ ;
                            AFTx [l] = Ax [p] ;
                            AFTi [l] = j ;
                        }
                    }
                }
            }
        }
        for (k = 1; k <= ni; k++)
        {
            /* check if previously dropped inequality is now active */
            row = ineq_row [k] ;
            if ( ir [row] <= ni ) /* row active, drop it */
            {
                ASSERT (RowmodFlag [row] == EMPTY) ;
#ifndef NDEBUG
                /* since we do not resymbol (remove any zeros that arise
                   in the factorization when we delete rows, we initialize
                   the components of lambda_pen, if it exists, to 0 to avoid
                   valgrind errors that would otherwise arise when the zeros
                   in the factorization are multiplied by undefined variables */
                if ( Com->use_penalty == TRUE )
                {
                    Com->lambda_pen [row] = PASAZERO ;
                }
#endif
                RowmodList [nrowdel] = row ;
                RowmodFlag [row] = nrowdel ;
                nrowdel++ ;
                /* remove row from the active row links */
                i = RLinkDn [row] ;
                j = RLinkUp [i] = RLinkUp [row] ;
                RLinkDn [j] = i ;
            }
            ir [row] = k + ni ;
            if ( Work->shiftl_is_zero )
            {
               lambda [row] = PPZERO ;
            }
            else
            {
               lambda [row] = -shift_l [row] ;
            }
        }
        Work->nrowdel = nrowdel ;
        Work->ncoladd = ncoladd ;
        Work->nf = nf ;
        if ( PrintLevel > 0 )
        {
            printf ("exit activeGP, restore factorization:\n") ;
            printf ("nrowdel: %ld ncoladd: %ld\n",
                   (LONG) nrowdel, (LONG) ncoladd) ;
            if ( PrintLevel > 1 )
            {
                printf ("\nadd columns:\n") ;
                for (i = 0; i < ncoladd; i++)
                {
                    printf ("%ld\n", (LONG) ColmodList [i]) ;
                }
                printf ("\ndelete rows:\n") ;
                for (i = 0; i < nrowdel; i++)
                {
                    printf ("%ld\n", (LONG) RowmodList [i]) ;
                }
            }
        }
#ifndef NOPPROJ
        if  ( nrowdel > 0 )
        {
            Com->ppcom->Parm = Com->pprojparm ;
            pproj_modrow (Com->ppcom, 0, TRUE, FALSE, -1, NULL, NULL,NULL,NULL);
            pproj_updateAnz (Com->ppcom, 0) ;
        }
        if ( ncoladd > 0 )
        {
            l = 0 ;
            for (k = 0; k < ncoladd; k++)
            {
                j = ColmodList [k] ;
                if ( Anz [j] > 0 )
                {
                    Cp [l] = Ap [j] ;
                    Cnz [l] = Anz [j] ;
                    l++ ;
                }
                else /* update the list of addition columns by removing j */
                {
                    ncoladd-- ;                         /* one less update */
                    col = ColmodList [ncoladd] ;        /* last col in list*/
                    ColmodList [k] = col ;              /* replace j by col*/
                    ColmodFlag [col] = k ;              /* new col location */
                    Work->ncoladd = ncoladd ;           /* store it in W */
                    k-- ;                               /* adjust k */
                    ColmodFlag [j] = EMPTY ;            /* col j is done */
                }
            }
            Com->ppcom->Parm = Com->pprojparm ;
            pproj_modcol (Com->ppcom, 0, 0, +1, NULL, Cp, Cnz, NULL, NULL, l) ;
        }
#endif
    }
#endif
    /* if the convergence tolerance not satisfied and no new constraints
       are activated, then go to cg_descent */
    return (PASA_CG_DESCENT) ;
}

/* ==========================================================================
   === pasa_gradprojLP ======================================================
   ==========================================================================
    If gradprojLP is called fromm activeLP, then perform one iteration
    of the gradient projection algorithm with a unit step and return to
    pasa_activeLP. Otherwise, perform the gradient projection algorithm
    until the objective value does not improve strictly.
   ========================================================================== */
int pasa_gradprojLP /* return:
                       PASA_ACTIVE_LP
                       PASA_SSOR_ASCENT_FAILURE_IN_PPROJ
                       PASA_SSOR_ITERATIONS_EXCEED_MAX_ITS_IN_PPROJ */
(
    PASAcom *Com  /* common data for PASA */
)
{
    int exponent, status, blks, blk ;
    PASAINT Annz, ATnz, col, iter, i, j, k, l, m, nf, Ll, Ul, Rl,
            nactive, nrowadd, ncoladd, nrowdel, ncoldel, prevblkL, prevblkU,
            p, p0, Q0, Q1, shii, nbrk ;
    PASAFLOAT err, berr, cerr, cerr_old, cerr_decay, cj, epsilon_decay, oj,
              lmax, xmax, fd, fn, sd, st, s, t, scale_l, scale_x,
              scale_l_inc, scale_x_inc, epsilon, epsilon_new, epsilon_inc,
              *singc ;
    PPwork   *Work ;      /* work structure in PPROJ */
    PPprob   *Prob ;      /* problem structure in PPROJ */
    PASAparm *pasaparm ;  /* parameter structure for PASA */
    PPparm   *pprojparm ; /* parameter structure for PPROJ */
    PPcom    *ppcom ;     /* com structure for pproj */

    pprojparm = Com->pprojparm ;
    pasaparm  = Com->pasaparm ;

    int const use_cholmod = pprojparm->cholmod ;
    int const  PrintLevel = pasaparm->PrintLevel ;
    if ( PrintLevel >= 1 )
    {
        printf ("START gradprojLP\n") ;
    }

    /* set the new location */
    Com->location = PASA_GRADPROJ_LP ;

    /* never check if the projection point is feasible */
    pprojparm->check_if_y_feas = 0 ;

    /* extract data from Com */
    PPdata               *ppdata = Com->pasadata->ppdata ; /* PPROJ input data*/
    int       const     loExists = Com->loExists ;
    int       const     hiExists = Com->hiExists ;
    PASAINT   const         ncol = Com->ncol ;
    PASAINT   const         nrow = Com->nrow ;
    PASAFLOAT const     grad_tol = Com->grad_tol ;
    PASAFLOAT                 *x = Com->x ;
    PASAFLOAT                 *c = Com->c ;
    PASAFLOAT                 *b = Com->b ;
    PASAFLOAT                *lo = Com->lo ;
    PASAFLOAT                *hi = Com->hi ;
    PASAFLOAT                *bl = Com->bl ;
    PASAFLOAT                *bu = Com->bu ;
    PASAFLOAT            *lambda = Com->lambda ;

    /* column singletons */
    PASAINT   const           ni = ppdata->ni ;
    PASAINT   const        nsing = ppdata->nsing ;
    PASAINT   const      nsingni = nsing + ni ;
    PASAINT   const *userRowSing = ppdata->row_sing ;
    PASAINT   const     nsingni1 = nsingni + 1 ;
    PASAINT   const     nsingni2 = nsingni + 2 ;
    PASAINT   const         ntot = nsingni1 + ncol ;

    epsilon = pasaparm->epsilon ; /* proximal parameter */
    epsilon_decay = pasaparm->EpsilonDecay ;
    
    if ( use_cholmod )
    {
        if ( epsilon == PASAZERO ) /* assign values if not given */
        {
            if      ( nrow <  100 )  epsilon = ldexp (1, -6) ;
            else if ( nrow < 2500 )  epsilon = ldexp (1, -3) ;
            else                     epsilon = PASAONE ;
        }
        if ( epsilon_decay == PASAZERO )
        {
            epsilon_decay = 0.125 ;
        }
    }
    else /* iterative method, no cholmod */
    {
        if ( epsilon == PASAZERO )
        {
            if      ( nrow <  500 )  epsilon = ldexp (1, -8) ;
            else if ( nrow < 2500 )  epsilon = ldexp (1, -6) ;
            else                     epsilon = PASAONE ;
        }
        if ( epsilon_decay == PASAZERO )
        {
            epsilon_decay = 0.5 ;
        }
        /* require that cerr decay by the factor cerr_decay or else
           let epsilon decay by the factor epsilon_decay in an effort
           to make cerr smaller */
        cerr_decay = sqrt(epsilon_decay) ;
    }

    PASAFLOAT const epsilon_grow  = pasaparm->EpsilonGrow ;
    PASAINT   const       gpmaxit = pasaparm->gpmaxit ;

    err = PASAINF ;
    iter = 0 ;
    scale_x = Com->scale_x ;
    scale_l = Com->scale_l ;

    /* scale c by scale_l */
    if ( scale_l != PASAONE )
    {
        pasa_scale (c, c, scale_l, ncol) ;
        if ( nsing )
        {
            pasa_scale (ppdata->singc, ppdata->singc, scale_l, nsing) ;
        }
        if ( pprojparm->start_guess == 1 )
        {
            pasa_scale (lambda, lambda, scale_l, nrow) ;
        }
    }

    /* scale the polyhedron by scale_x*epsilon */
    t = scale_x*epsilon ;
    if ( t != PASAONE )
    {
        pasa_scale (bl, bl, t, nrow) ;
        if ( ni ) /* if no strict inequalities, bl = bu */
        {
            pasa_scale (bu, bu, t, nrow) ;
        }
        if ( loExists )
        {
            pasa_scale (lo, lo, t, ncol) ;
        }
        if ( hiExists )
        {
            pasa_scale (hi, hi, t, ncol) ;
        }
        if ( nsing )
        {
            pasa_scale (ppdata->singlo, ppdata->singlo, t, nsing) ;
            pasa_scale (ppdata->singhi, ppdata->singhi, t, nsing) ;
        }
    }

    PASAFLOAT *y = Com->userg ;
    pasa_scale (y, c, -PASAONE, ncol) ; /* starting x = 0 */

    ppdata->y = y ;
    ppdata->x = x ;

    /* set some pproj parameters */
    pprojparm->getfactor        = FALSE ;
    pprojparm->use_prior_data   = FALSE ; /* after iteration 1, change to TRUE*/
    pprojparm->return_data      = TRUE ;
    pprojparm->permute          = FALSE ;
    pprojparm->use_startup      = TRUE ;
    pprojparm->LP               = TRUE ;
    /* for the initial projection, the start guess is based on the pasaparm
       parameter use_lambda */
    if ( pasaparm->use_lambda == TRUE )
    {
        pprojparm->start_guess = 1 ;
    }
    else
    {
        pprojparm->start_guess = 0 ;
    }

    /* call pproj, reorder equation and variables, perform phase1 */
    status = pproj (ppdata) ;
    Com->pasastat->nproject++ ;
    ppcom = Com->ppcom = ppdata->priordata ;

    /* solution is stored in ppdata->x = Com->x */

    if ( (status != PPROJ_SOLUTION_FOUND) &&
         (status != PPROJ_ERROR_DECAY_STAGNATES) ) /* error occurred */
    {
        Com->f = PASAINF ;
        return (status) ;
    }

    pprojparm->start_guess    = 3 ;      /* use input lambda for start guess */
    pprojparm->use_prior_data = TRUE ;
    pprojparm->use_startup    = FALSE ;  /* go straight to pproj_dasa */

    /* henceforth, store pproj solution in ppdata->x = Com->d = d */
    PASAFLOAT *dx = ppdata->x = Com->d ;

    /* switch storage format to that of pproj since a permutation was done */
    Prob = ppcom->Prob ;

    /* -c was stored in y, so switch sign to get reordered c */
    pasa_scale (c, Prob->y, -PASAONE, ncol) ;

                     b = Prob->b ;
    if ( loExists ) lo = Prob->lo ;
    if ( hiExists ) hi = Prob->hi ;

    singc  = Prob->singc ;
    bl = (nsing) ? Prob->singlo : Prob->bl ;
    bu = (nsing) ? Prob->singhi : Prob->bu ;

    PASAINT   const         *Ap = Prob->Ap ;
    PASAINT   const         *Ai = Prob->Ai ;
    PASAINT   const        *Anz = Prob->Anz ;
    PASAFLOAT const         *Ax = Prob->Ax ;
    PASAINT   const   *ineq_row = Prob->ineq_row ;
    PASAINT   const   *row_sing = Prob->row_sing ;
    PASAINT   const  *row_sing1 = row_sing+1 ;

    Work = ppcom->Work ;
    PASAINT                 *ir = Work->ir ;
    int                     *ib = Work->ib ;
    PASAINT                  *F = Work->F ;
    PASAINT                 *ns = Work->ns ;
    PASAINT   const        *ATp = Work->ATp ;
    PASAINT   const        *ATi = Work->ATi ;
    PPFLOAT   const        *ATx = Work->ATx ;

    PASAFLOAT           *arrayd = Work->arrayd ;
    PASAFLOAT          *dlambda = arrayd ;  arrayd += nrow ;
    /* g is also used for absAx */
    PASAFLOAT                *g = arrayd ;  arrayd += nrow ;
    PASAFLOAT                *r = arrayd ;  arrayd += nrow ;
    PASAFLOAT               *pA = arrayd ;  arrayd += ncol ;
    PASAFLOAT         *Br_value = arrayd ;  arrayd += ntot ;

    PASAFLOAT               *pC = Com->userg ; /* = ppdata->y */
    PASAINT                *slo = Work->slo ;
    PASAINT                *shi = Work->shi ;

    PASAINT             *arrayi = Work->arrayi ;
    PASAINT               *Heap = arrayi ;  arrayi += ntot + 1 ;

    /* AFT is the transpose of the AF matrix, the submatrix of A associated
       with the free variables */
    PASAINT   const       *AFTp = Work->AFTp ;
    PASAINT              *AFTnz = Work->AFTnz ;
    PASAINT               *AFTi = Work->AFTi ;
    PASAFLOAT             *AFTx = Work->AFTx ;

    PASAINT            *RLinkUp = Work->RLinkUp ;
    PASAINT            *RLinkDn = Work->RLinkDn ;
          /* numbering of strict inequalities (also referred to
             as column singletons) starts at 1 instead of 0 */
    int       const *sol_to_blk = Work->sol_to_blk ;
    PASAINT            *lLinkUp = Work->SLinkUp ;
    PASAINT            *lLinkDn = Work->SLinkDn ;
    PASAINT            *uLinkUp = Work->SLinkUp ;
    PASAINT            *uLinkDn = Work->SLinkDn ;
    PASAINT             *lstart = Work->lstart ;
    PASAINT             *ustart = Work->ustart ;
    PASAINT         *RowmodFlag = Work->RowmodFlag ;
    PASAINT         *RowmodList = Work->RowmodList ;
    PASAINT         *ColmodFlag = Work->ColmodFlag ;
    PASAINT         *ColmodList = Work->ColmodList ;
    PASAFLOAT          *shift_l = Work->shift_l ;
    PASAFLOAT const       sigma = Work->sigma ;

    PASAFLOAT               *Wb = Work->b ;
    PASAFLOAT               *Wc = Work->c ;
    PASAFLOAT               *WD = Work->D ;
    PASAFLOAT              *Wlo = Work->lo ;
    PASAFLOAT              *Whi = Work->hi ;
#ifndef NDEBUG
    /* if the debug mode is used, then create an array for the right side */
    PASAFLOAT *checkb, *checklo, *checkhi ;
    checkb = ppcom->Check->b =
                 (PASAFLOAT *) pasa_malloc (&status, nrow, sizeof (PASAFLOAT)) ;
    if ( loExists ) checklo = ppcom->Check->lo =
                 (PASAFLOAT *) pasa_malloc (&status, ncol, sizeof (PASAFLOAT)) ;
    if ( hiExists ) checkhi = ppcom->Check->hi =
                 (PASAFLOAT *) pasa_malloc (&status, ncol, sizeof (PASAFLOAT)) ;
    if ( status == SOPT_OUT_OF_MEMORY ) return (PASA_OUT_OF_MEMORY) ;
#endif

    while ( (err > grad_tol) && (iter < gpmaxit) )
    {
        iter++ ;
        /* If iter > 1, then call pproj. Otherwise, pproj was called above. */
        if ( iter > 1 )
        {
            status = pproj (ppdata) ;
            Com->pasastat->nproject++ ;

            /* this is normally done in pproj_invert, but since we skip
               the inversion, we do it here */
            if ( use_cholmod )
            {
                pproj_updateAnz (ppcom, 2) ;
                /*also sets AFTnz = 0 for dead rows*/
            }

            /* update x with the solution in d = ppdata->x */
            pasa_step (x, x, dx, PASAONE, ncol) ;

            if ( status == PPROJ_FACTORIZATION_FAILS_IN_CHOLMOD )
            {
                return (PASA_FACTORIZATION_FAILS_IN_CHOLMOD) ;
            }
            else if ( status == PPROJ_INTEGER_OVERFLOW_IN_CHOLMOD )
            {
                return (PASA_INTEGER_OVERFLOW_IN_CHOLMOD) ;
            }
            if ( (status != PPROJ_SOLUTION_FOUND) &&
                 (status != PPROJ_ERROR_DECAY_STAGNATES) ) /* error occurred */
            {
                return (status) ;
            }
        }

        /* Instead of projecting xk - step*ck onto the polyhedron as in
           the usual gradient projection step, the polyhedron is scaled
           by epsilon = 1/step. Also, a scale factor scale_x is applied
           to the polyhedron. This factor is chosen in an effort to achieve
           a solution whose sup-norm is one. The variables scale_x and
           epsilon contain the cummulative scale values, however, in each
           iteration, we only apply the increments to scale_x and
           epsilon determined in the previous iteration. */

        if ( PrintLevel >= 1 )
        {
            printf ("iter: %ld step: %e scale_x: %e scale_l: %e\n",
                   (LONG) iter, 1./epsilon, scale_x, scale_l) ;
        }
#ifndef NDEBUG
        pasa_errLP (epsilon, Com) ;
#endif

        /* Determine the scale factor for lambda and x and see whether the
           stopping criterion is satisfied. Both lmax and xmax should be
           on the order of one if the scaling is working. */
        lmax = Work->norm_l ;
        if ( lmax == PASAZERO ) lmax = PASAONE ;

        xmax = pasa_sup_normx (x, ncol)/epsilon ;
        if ( xmax == PASAZERO ) xmax = PASAONE ;
        /* Work->absAxk = xmax ;*/

        cerr = Work->cerr/lmax ;
        berr = Work->errdual/(xmax*epsilon) ;
        err = PASAMAX (berr, cerr) ;
        if ( PrintLevel >= 1 )
        {
            printf ("err: %e cerr: %e berr: %e xmax: %e lmax: %e\n",
                    err, cerr, berr, xmax, lmax) ;
        }
        if ( (err <= grad_tol) || (iter >= gpmaxit) ) break ;

        /* Determine the adjustments needed for the x and lambda scaling */
        t = frexp (lmax, &exponent) ;        /* exponent for lambda */
        scale_l_inc = ldexp (1, -exponent) ; /* lambda normalization factor */
        scale_l *= scale_l_inc ;             /* cummulative lambda scale */

        t = frexp (xmax, &exponent) ;        /* exponent for x */
        scale_x_inc = ldexp (1, -exponent) ; /* x normalization factor */
        scale_x *= scale_x_inc ;             /* cummulative x scale */

        if ( PrintLevel )
        {
            printf ("scale_x_inc: %e scale_x: %e scale_l_inc: %e scale_l: %e\n",
                     scale_x_inc,    scale_x,    scale_l_inc,    scale_l) ;
        }

        /* Determine the adjustments to epsilon. If cerr is small
           enough relative to either grad_tol or berr, then increase epsilon;
           otherwise, epsilon decreases. After determining these adjustments,
           we evaluate the change that these adjustments to epsilon and the
           scaling of x and lambda have on the solution to the dual problem,
           assuming that the constraints active at the current lambda remain
           active after the change in epsilon and the update to x. Since this
           assumption may not be valid, a line search is performed in this
           search direction using the new value for epsilon.
           After the line search, x and lambda are scaled by the scales
           computed above. */
        if ( (cerr <= grad_tol*pasaparm->cerr_decay1) ||
             (cerr <= berr*pasaparm->cerr_decay2) )
        {
            epsilon_inc = epsilon_grow ;
        }
        else /* epsilon decays */
        {
            if ( use_cholmod )
            {
                epsilon_inc = epsilon_decay ;
            }
            else
            {
                epsilon_inc = PASAONE ;
                if ( iter == (int) 1 )
                {
                    cerr_old = cerr*cerr_decay ;
                }
                else
                {
                    /* when cerr decay linearly, no change to epsilon */
                    if ( cerr <= cerr_old )
                    {
                        cerr_old *= cerr_decay ;
                    }
                    else /* epsilon decays by epsilon_decay => larger step */
                    {
                        epsilon_inc = epsilon_decay ;
                    }
                }
            }
        }
        /* The rationale for the factor scale_x/scale_l below is the
           following: We want to both rescale the problem and multiply
           epsilon by some factor. If we multiply epsilon by this scale
           ratio, then the minimizer of the proximal function with this
           new epsilon is the same as the minimizer of the rescaled
           function with the new epsilon. */
        epsilon_inc *= (scale_x_inc/scale_l_inc) ;
        epsilon_new = epsilon*epsilon_inc ;
        if ( PrintLevel )
        {
            printf ("epsilon: %e epsilon_inc: %e epsilon_new %e\n",
            epsilon, epsilon_inc, epsilon_new) ;
        }

        /* If cholmod is being used and the matrix has been factored,
           then evaluate the adjustment in lambda due to changing epsilon to
           epsilon_new and due to the update of x (x = x + d done above).
           Without the scaling of variable used in lpdasa, the change in
           the right side is given by:

              (eps_new - eps_old)b + AF(eps_old yF_old - eps_new yF_new)
                                   + AB(eps_old xB_old - eps_new xB_new)

           After the first iteration, yF_new = xF/eps_old and yF_old =
           (xF - dF)/eps_old due to the scaling by eps. We assume that
           xB_old = xB_new = code's xB/epsilon.  In the first iteration,
           y_old = 0, yF_new = xF/eps_old, and again
           xB_old = xB_new = code's xB/epsilon.  Also note that b
           in this formula for the right side is the code's b/eps_old.
           Define deps = (eps_new - eps_old)/eps_old. After substituting
           in terms of the code's variables, the right side change after
           the first iteration is

              deps*b - AF*(deps*xF_new + dF) - deps*AB*xB_new

           For the first iteration, let us define rel_eps = eps_new/eps_old.
           The right side change is

              deps*b - rel_eps*AF*xF_new - deps*AB*xB_new) */

#ifndef NDEBUG
        if ( Work->fac == FALSE )
        {
            printf ("warning: Work->fac is FALSE after pproj\n") ;
        }
#endif
        if ( use_cholmod )
        {
            nrowadd = Work->nrowadd ;
            ncoladd = Work->ncoladd ;
            nrowdel = Work->nrowdel ;
            ncoldel = Work->ncoldel ;
        }

        if ( use_cholmod && (Work->fac == TRUE) && (RLinkUp [nrow] < nrow) )
        {
            PASAFLOAT const deps = epsilon_inc - PASAONE ;
            pasa_initx (dlambda, PASAZERO, nrow) ;

            /* build right side change in dlambda */
            if ( nsingni )
            {
                if ( ni )
                {
                    for (i = RLinkUp [nrow]; i < nrow; i = RLinkUp [i])
                    {
                        j = ir [i] ;
                        ASSERT (j <= ni) ;
                        if ( j == 0 )
                        {
                            t = b [i] ;
                        }
                        else if ( j > 0 ) /* for active rows j <= ni */
                        {
                            t = bu [j] ;
                        }
                        else /* j < 0 */
                        {
                            t = bl [-j] ;
                        }
                        dlambda [i] = deps*t ;
                        r [i] = epsilon_inc*t - sigma*lambda [i] ;
                    }
                }
                else /* nsing */
                {
                    for (i = RLinkUp [nrow]; i < nrow; i = RLinkUp [i])
                    {
                        ASSERT (ir [i] <= nsingni) ;
                        t = PASAZERO ;
                        if ( ir [i] ) /* there are bound singletons in row */
                        {
                            PASAINT const qlo = slo [i] ;
                            j = row_sing [i] ;
                            if ( qlo )
                            {
                                for (; j <= qlo; j++)
                                {
                                    t += bl [j] ;
                                }
                            }
                            PASAINT const qhi = row_sing1 [i] ;
                            for (; j < qhi; j++)
                            {
                                t += bu [j] ;
                            }
                        }
                        t += b [i] ;
                        dlambda [i] = deps*t ;
                        r [i] = epsilon_inc*t - sigma*lambda [i] ;
                    }
                }
            }
            else
            {
                /* pure equality constraints:
                       dlambda = eps*b
                       r       = epsilon_inc*b - sigma*lambda */
                
                pasa_scale (dlambda, b, deps,         nrow) ;
                pasa_scale (r,       b, epsilon_inc, nrow) ;
                pasa_daxpy(r, lambda, -sigma, nrow) ;
            }
            if ( iter > 1 )
            {
                for (j = 0; j < ncol; j++)
                {
                    PASAFLOAT const tx = (ib [j]) ?
                                                 /* bound : free */
                                              deps*x [j] : deps*x [j] + dx [j] ;
                    k = Ap [j] ;
                    const PASAINT q = k + Anz [j] ;
                    for (; k < q; k++)
                    {
                        dlambda [Ai [k]] -= tx*Ax [k] ;
                    }
                }
            }
            else /* iter = 1 */
            {
                for (j = 0; j < ncol; j++)
                {
                    PASAFLOAT const tx = (ib [j]) ?
                                                 /* bound : free */
                                               deps*x [j] : epsilon_inc*x [j] ;
                    k = Ap [j] ;
                    PASAINT const q = k + Anz [j] ;
                    for (; k < q; k++)
                    {
                        dlambda [Ai [k]] -= tx*Ax [k] ;
                    }
                }
            }
            pproj_lsol (Work->L, dlambda, RLinkUp [nrow], nrow, RLinkUp) ;
            i = RLinkUp [nrow] ;
            /* momentarily set the initial RLinkDn to -1, this simplifies
               indexing in dltsolve */
            RLinkDn [i] = -1 ;
            j = RLinkDn [nrow] ;
            pproj_dltsol (Work->L, dlambda, dlambda, j, i, RLinkDn) ;
            /* dlambda stores the change in the solution to the dual problem
               associated with the change in epsilon to epsilon_new and the
               update of x (x = x + d), assuming there is no change in the
               active constraints. Next, a line search is performed in
               the direction of dlambda, starting from the current solution
               of the dual problem. */
            RLinkDn [i] = nrow ; /* restore RLinkDn */

            /* scale the variables since the line search should be based
               on the new epsilon */
            if ( epsilon_inc != PASAONE )
            {
                pasa_scale (x , x , epsilon_inc, ncol) ;
                pasa_scale (b , b , epsilon_inc, nrow) ;
                if ( loExists ) pasa_scale (lo, lo, epsilon_inc, ncol) ;
                if ( hiExists ) pasa_scale (hi, hi, epsilon_inc, ncol) ;
                if ( nsingni )
                {
                    pasa_scale (bl+1, bl+1, epsilon_inc, nsingni) ;
                    pasa_scale (bu+1, bu+1, epsilon_inc, nsingni) ;
                }
            }

            pasa_step (pC, c, x, -PASAONE, ncol) ;
            pasa_initx (pA, PASAZERO, ncol) ;

            p = 0 ;
            for (i = 0; i < nrow; i++)
            {
                const PASAFLOAT lambdai = lambda [i] ;
                const PASAINT q = ATp [i+1] ;
                const PASAINT iri = ir [i] ;
                if ( iri <= nsingni ) /* active inequality */
                {
                    g [i] = PASAZERO ;
                    const PASAFLOAT dlambdai = dlambda [i] ;
                    for (; p < q; p++)
                    {
                        const PASAINT colj = ATi [p] ;
                        const PASAFLOAT ax = ATx [p] ;
                        pC [colj] -= ax* lambdai ;
                        pA [colj] += ax*dlambdai ;
                    }
                }
                else
                {
                    for (; p < q; p++)
                    {
                        pC [ATi [p]] -= ATx [p]*lambdai ;
                    }
                }
            }

            nbrk = 0 ;
            sd = PASAZERO ;
            for (j = 0; j < ncol; j++)
            {
                p = Ap [j] ;
                const PASAINT q = p + Anz [j] ;
                const PASAFLOAT pAj = pA [j] ;
                if ( ib [j] < 0 )      /* at lower bound */
                {
                    if ( q > p )
                    {
                        const PASAFLOAT loj = lo [j] ;
                        for (; p < q; p++)
                        {
                            r [Ai [p]] -= loj*Ax [p] ;
                        }
                        if ( pAj > PASAZERO )
                        {
                            Br_value [j] = (loj + pC [j])/pAj ;
                            nbrk++ ;
                            Heap [nbrk] = j ;
                        }
                    }
                }
                else if ( ib [j] > 0 ) /* at upper bound */
                {
                    if ( q > p )
                    {
                        const PASAFLOAT hij = hi [j] ;
                        for (; p < q; p++)
                        {
                            r [Ai [p]] -= hij*Ax [p] ;
                        }
                        if ( pAj < PASAZERO )
                        {
                            Br_value [j] = (hij + pC [j])/pAj ;
                            nbrk++ ;
                            Heap [nbrk] = j ;
                        }
                    }
                }
                else
                {
                    sd += pAj*pAj ;
                    const PASAFLOAT pCj = pC [j] ;
                    for (; p < q; p++)
                    {
                        g [Ai [p]] += Ax [p]*pCj ;
                    }
                }
            }

            if ( Work->shiftl_is_zero )
            {
                for (j = lLinkUp [nsingni1]; j <= nsingni; j = lLinkUp [j])
                {
                    PASAINT const row = ineq_row [j] ;
                    if ( (t = dlambda [row]) < PASAZERO )
                    {
                        /* lambda > 0 for lLinkUp */
                        if ( ni ) s = -lambda [row]/t ;
                        else      s = (singc [j] - lambda [row])/t ;

                        k = j + ncol ;
                        Br_value [k] = s ;
                        nbrk++ ;
                        Heap [nbrk] = k ;
                    }
                }

                for (j = uLinkUp [nsingni2]; j <= nsingni; j = uLinkUp [j])
                {
                    PASAINT const row = ineq_row [j] ;
                    if ( (t = dlambda [row]) > PASAZERO )
                    {
                        /* lambda < 0 for uLinkUp */
                        if ( ni ) s = -lambda [row]/t ;
                        else      s = (singc [j] - lambda [row])/t ;
                        k = j + ncol ;
                        Br_value [k] = s ;
                        nbrk++ ;
                        Heap [nbrk] = k ;
                    }
                }
            }
            else /* shift_l is nonzero */
            {
                for (j = lLinkUp [nsingni1]; j <= nsingni; j = lLinkUp [j])
                {
                    PASAINT const row = ineq_row [j] ;
                    if ( (t = dlambda [row]) < PASAZERO )
                    {
                        /* lambda > 0 for lLinkUp */
                        if ( ni ) s = -(lambda [row]+shift_l [row])/t ;
                        else      s = (singc[j]-(lambda [row]+shift_l [row]))/t;
                        k = j + ncol ;
                        Br_value [k] = s ;
                        nbrk++ ;
                        Heap [nbrk] = k ;
                    }
                }

                for (j = uLinkUp [nsingni2]; j <= nsingni; j = uLinkUp [j])
                {
                    PASAINT const row = ineq_row [j] ;
                    if ( (t = dlambda [row]) > PASAZERO )
                    {
                        /* lambda < 0 for uLinkUp */
                        if ( ni ) s = -(lambda [row]+shift_l [row])/t ;
                        else      s = (singc[j]-(lambda [row]+shift_l [row]))/t;
                        k = j + ncol ;
                        Br_value [k] = s ;
                        nbrk++ ;
                        Heap [nbrk] = k ;
                    }
                }
            }

            fd = t = PASAZERO ;                   /* 1st and 2nd derivatives */
            for (i = RLinkUp [nrow]; i < nrow; i = RLinkUp [i])
            {
                PASAFLOAT const dlambdai = dlambda [i] ;
                t += dlambdai*dlambdai ;         /* d is search direction*/
                fd += dlambdai*(r [i] + g [i]) ; /* g + r is current gradient */
            }
            sd += t*sigma ;

            nf = Work->nf ;
            st = PASAZERO ;
            PASAINT const nbrk0 = nbrk ;
            if ( (nbrk > 0) && (fd > PASAZERO) )
            {
                pproj_minheap_build (Heap, Br_value, nbrk) ;
                for (k = 1; k <= nbrk; k++)
                {
                    ns [Heap [k]] = k ;
                }
                while ( nbrk > 0 )
                {
                    if ( sd > PPZERO )
                    {
                        col = Heap [1] ;
                        PASAFLOAT const Br = PASAMAX (Br_value [col], PASAZERO);
                        pproj_minheap_delete (Heap, ns, Br_value, &nbrk, 1) ;
                        fn = fd - sd * (Br - st) ;
                        if ( PrintLevel > 2 )
                        {
                            printf("    brk: %ld col: %ld fn: %9.3e fd: %9.3e"
                                   " sd: %9.3e st: %9.3e Br_value: %9.3e\n",
                                  (LONG) nbrk, (LONG) col, fn, fd, sd, st, Br) ;
                        }

                        if ( fn <= PPZERO )
                        {
                            if ( fd != fn )
                            {
                                st += (fd/(fd-fn))*(Br - st);
                            }
                            fd = PPZERO ;
                            break ;
                        }
                        else
                        {
                            fd = fn ;
                            st = Br ;
                        }
                    }
                    else
                    {
                        break ;
                    }

                    if ( col < ncol ) /* free column */
                    {
                        F [nf++] = col ;
                        sd += pA [col]*pA [col] ;
                        if ( ib [col] < 0 )
                        {
                            t = lo [col] ;
                        }
                        else
                        {
                            t = hi [col] ;
                        }
                        PASAFLOAT const bndcol = t ;
                        ib [col] = 0 ;
                        if ( use_cholmod )
                        {
                            ASSERT (ColmodFlag [col] == EMPTY) ;
                            ColmodList [ncoladd] = col ;
                            ColmodFlag [col] = ncoladd ;
                            ncoladd++ ;
                        }
                        p = Ap [col] ;
                        PASAINT const q = p + Anz [col] ;
                        for (; p < q; p++)
                        {
                            i = Ai [p] ;
                            /* if the row is active, then add to AFT matrix */
                            if ( ir [i] <= nsingni )
                            {
                                PASAINT   const ki = AFTp [i] + AFTnz [i]++ ;
                                PASAFLOAT const ax = Ax [p] ;
                                AFTx [ki] = ax ;
                                r [i] += bndcol*ax ;
                                AFTi [ki] = col ;
                            }
                        }
                    }
                    else              /* drop row */
                    {
                        PASAINT const sing = col - ncol ;
                        PASAINT const row = ineq_row [sing] ;
                        Work->nactive-- ;
                        ASSERT ((ir [row] != 0) && (ir [row] <= nsingni)) ;
                        /* row dropped due to singleton sing */
                        ir [row] = sing + nsingni ;
                        i = RLinkDn [row] ;
                        k = RLinkUp [row] ;
                        RLinkUp [i] = k ;
                        RLinkDn [k] = i ;
                        if ( use_cholmod )
                        {
                            ASSERT (RowmodFlag [row] == EMPTY) ;
                            RowmodList [nrowdel] = row ;
                            RowmodFlag [row] = nrowdel ;
                            nrowdel++ ;
                        }
                        if ( Work->shiftl_is_zero )
                        {
                            if ( ni )
                            {
                                t = -lambda [row] ;
                                lambda [row] = PASAZERO ;
                            }
                            else /* nsing */
                            {
                                t = singc [sing] - lambda [row] ;
                                lambda [row] = singc [sing] ;
                            }
                        }
                        else
                        {
                            if ( ni )
                            {
                                t = -(lambda [row]+shift_l [row]) ;
                                lambda [row] = -shift_l [row] ;
                            }
                            else
                            {
                                t = singc [sing] - (lambda [row]+shift_l [row]);
                                lambda [row] = singc [sing] ;
                            }
                        }
                        PASAFLOAT const dlnew = t ;
                        PASAFLOAT const dlambda_row = dlambda [row] ;
                        fd -= dlambda_row*r [row] ;
                        sd -= sigma*dlambda_row*dlambda_row ;
                        if ( dlambda_row > PPZERO ) /* drop upper bound bu */
                        {
                            m = uLinkUp [sing] ;
                            l = uLinkDn [sing] ;
                            uLinkUp [l] = m ;
                            uLinkDn [m] = l ;
                            if ( use_cholmod )
                            {
                                blk = sol_to_blk [sing] ;
                                if ( sing == ustart [blk] )
                                {
                                    ustart [blk] = m ;
                                }
                            }
                            if ( nsing )
                            {
                                shi [row] = 0 ;
                                j = slo [row] ;
                                if ( j )
                                {
                                    m = lLinkUp [j] ;
                                    l = lLinkDn [j] ;
                                    lLinkUp [l] = m ;
                                    lLinkDn [m] = l ;
                                    slo [row] = 0 ;
                                    if ( use_cholmod )
                                    {
                                        if ( j == lstart [blk] )
                                        {
                                            lstart [blk] = m ;
                                        }
                                    }
                                }
                            }
                        }
                        else /* drop lower bound bl */
                        {
                            m = lLinkUp [sing] ;
                            l = lLinkDn [sing] ;
                            lLinkUp [l] = m ;
                            lLinkDn [m] = l ;
                            if ( use_cholmod )
                            {
                                blk = sol_to_blk [sing] ;
                                if ( sing == lstart [blk] )
                                {
                                    lstart [blk] = m ;
                                }
                            }
                            if ( nsing )
                            {
                                slo [row] = 0 ;
                                j = shi [row] ;
                                if ( j )
                                {
                                    m = uLinkUp [j] ;
                                    l = uLinkDn [j] ;
                                    uLinkUp [l] = m ;
                                    uLinkDn [m] = l ;
                                    shi [row] = 0 ;
                                    if ( use_cholmod )
                                    {
                                        if ( j == ustart [blk] )
                                        {
                                            ustart [blk] = m ;
                                        }
                                    }
                                }
                            }
                        }
                        PASAINT const q = ATp [row+1] ;
                        for (p = ATp [row]; p < q; p++)
                        {
                            PASAINT const colj = ATi [p] ;
                            PASAFLOAT const ax = ATx [p] ;
                            PASAFLOAT const pAj = pA [colj] ;
                            /* Cj = c after taking the step */
                            PASAFLOAT const Cj = pC [colj] - st*pAj ;
                            /* Modify pC to account for later update. That is,
                               when we drop the row, the contribution of the
                               elements in the row to pC need to be removed.
                               The previous contribution of row i to pC was
                               -a_{ij}lambda_i. So now we need to add this
                               to pC. Note that dlnew = -lambda_i. */
                            pC [colj] -= dlnew*ax ;
                            t = dlambda_row*ax ;
                            const PASAFLOAT pAjnew = pA [colj] = pAj - t ;
                            /* update c, pA, sd, fd */

                            if ( ib [colj] == 0 )
                            {
                                fd -= Cj*t ;
                                sd -= t*(pAj + pAjnew) ;
                            }
                            else if ( ib [colj] < 0 )
                            {
                                if ( pAjnew > PASAZERO )
                                {
                                    Br_value [colj] = st+(lo [colj]+Cj)/pAjnew ;
                                    if ( ns [colj] != EMPTY )
                                    {
                                        pproj_minheap_update (Heap,
                                           ns, Br_value, nbrk, ns [colj]) ;
                                    }
                                    else
                                    {
                                       pproj_minheap_add (colj, Heap,
                                           ns, Br_value, &nbrk) ;
                                    }
                                }
                                else
                                {
                                    if ( ns [colj] != EMPTY )
                                    {
                                        pproj_minheap_delete (Heap,
                                           ns, Br_value, &nbrk, ns [colj]);
                                    }
                                }
                            }
                            else /* ib [colj] > 0 */
                            {
                                if ( pAjnew < PASAZERO )
                                {
                                    Br_value [colj] = st+(hi [colj]+Cj)/pAjnew ;
                                    if ( ns [colj] != EMPTY )
                                    {
                                        pproj_minheap_update (Heap,
                                           ns, Br_value, nbrk, ns [colj]) ;
                                    }
                                    else
                                    {
                                       pproj_minheap_add (colj, Heap,
                                           ns, Br_value, &nbrk) ;
                                    }
                                }
                                else
                                {
                                    if ( ns [colj] != EMPTY )
                                    {
                                        pproj_minheap_delete (Heap,
                                           ns, Br_value, &nbrk, ns [colj]);
                                    }
                                }
                            }
                        }
                    }
                    if ( PrintLevel > 2 )
                    {
                        if ( col < ncol )
                        {
                            PRINTF ("    free col: %ld\n", (LONG) col) ;
                        }
                        else
                        {
                            PRINTF ("    drop row, sing: %ld\n",
                                                 (LONG) col-ncol) ;
                        }
                    }
                    if ( fd <= PASAZERO )
                    {
                        break ;
                    }
                }
            }
            if ( fd < PASAZERO )
            {
                fd = PASAZERO ;
            }
            if ( sd > PASAZERO )
            {
                st += fd/sd ;
            }
            if ( (nbrk0 == 0) && (fd > PASAZERO) )
            {
                st = PASAONE ;
            }
            if ( PrintLevel >= 1 )
            {
                printf ("line search step after epsilon update: %e\n", st) ;
            }
            if ( st > PASAZERO )
            {
                for (i = RLinkUp [nrow]; i < nrow; i = RLinkUp [i])
                {
                    lambda [i] += st*dlambda [i] ;
                }
            }
            if ( nbrk > 0 )
            {
                for (k = 1; k <= nbrk; k++)
                {
                    ns [Heap [k]] = EMPTY ;
                }
            }
            Work->nf = nf ;
            Work->ncoladd = ncoladd ;
            Work->nrowdel = nrowdel ;

            if ( scale_l_inc != PASAONE )
            {
                if ( nsing ) pasa_scale (singc+1, singc+1, scale_l_inc, nsing) ;
                pasa_scale (c, c, scale_l_inc, ncol) ;
                pasa_scale (lambda, lambda, scale_l_inc, nrow) ;
                pasa_scale (x , x , scale_l_inc, ncol) ;
                pasa_scale (b , b , scale_l_inc, nrow) ;
                if ( loExists ) pasa_scale (lo, lo, scale_l_inc, ncol) ;
                if ( hiExists ) pasa_scale (hi, hi, scale_l_inc, ncol) ;
                if ( nsingni )
                {
                    pasa_scale (bl+1, bl+1, scale_l_inc, nsingni) ;
                    pasa_scale (bu+1, bu+1, scale_l_inc, nsingni) ;
                }
            }
        }
        else /* either no cholmod or the matrix was not factored or no rows */
        {
            if ( scale_l_inc != PASAONE )
            {
                pasa_scale (c, c, scale_l_inc, ncol) ;
                pasa_scale (lambda, lambda, scale_l_inc, nrow) ;
                if ( nsing ) pasa_scale (singc+1, singc+1, scale_l_inc, nsing) ;
            }
            t = epsilon_inc*scale_l_inc ;
            if ( t != PASAONE )
            {
                pasa_scale (x , x , t, ncol) ;
                pasa_scale (b , b , t, nrow) ;
                if ( loExists ) pasa_scale (lo, lo, t, ncol) ;
                if ( hiExists ) pasa_scale (hi, hi, t, ncol) ;
                if ( nsingni )
                {
                    pasa_scale (bl+1, bl+1, t, nsingni) ;
                    pasa_scale (bu+1, bu+1, t, nsingni) ;
                }
            }
        }

        epsilon = epsilon_new*(scale_l_inc/scale_x_inc) ;

        if ( PrintLevel )
        {
            printf ("epsilon after scale adjustment: %e epsilon_inc2: %e\n",
                     epsilon, scale_l_inc/scale_x_inc) ;
        }

        /* perform translation and check error */

        /* nonzero diagonal for ssor algorithms */
        pasa_initx (WD, Work->SSORsigma, nrow) ;

        /* start building right side for proximal problem */
        pasa_copyx  (Wb, b, nrow) ; /* NOTE: bi = 0 for inequalities*/

        /* initial an array absAx = g (malloc'd above using arrayd) to zero */
        PASAFLOAT *absAx = g ;
        pasa_initx (absAx, PASAZERO, nrow) ;

        /* below dx become x inside of pproj, the updated x in pproj
           is returned in dx and then added to x in gradprojLP to
           obtain the primal solution approximation  */
        berr = cerr = PASAZERO ;
        /* When the if below is TRUE, check if a bound on a variable is active
           and if so, then bind the variable. Likewise, if a row has dropped
           and it is now active, then add the row. In this case, the AFT
           matrix is rebuilt. If update_bind is FALSE, then no binding of
           variable or adding of rows is done. */
        int const update_bind = FALSE ;
        if ( update_bind )
        {
            Annz = nf = p = 0 ;
            /* rebuild AFT, initialize nnz's to zero */
            pproj_initi (AFTnz, (PPINT) 0, nrow) ;
            for (j = 0; j < ncol; j++)
            {
                cj = -c [j] ;
                p0 = p ;
                PASAINT const q = Ap [j+1] ;
                for (; p < q; p++)
                {
                    cj += lambda [Ai [p]]*Ax [p] ;
                }
                int const ibj = ib [j] ;
                PASAFLOAT const loj   = (loExists) ? lo [j] : -PPINF ;
                PASAFLOAT const hij   = (hiExists) ? hi [j] :  PPINF ;
                PASAFLOAT const xj    = x [j] ;
                PASAFLOAT const xjnew = xj + cj ;
    
                if ( (xjnew > hij) || (xjnew < loj) ) /* xj at a bound */
                {
                    if ( use_cholmod && !ibj ) /*xj free previously, now bound*/
                    {
                        if ( ColmodFlag [j] == EMPTY ) /* add to bound list */
                        { 
                            ncoldel++ ;
                            ColmodList [ncol-ncoldel] = j ;
                            ColmodFlag [j] = ncol-ncoldel ;
                        }
                        else /* remove j from add list since it is now bound */
                        {
                            l = ColmodFlag [j] ;
                            ncoladd-- ;
                            m = ColmodList [ncoladd] ;
                            ColmodList [l] = m ;
                            ColmodFlag [m] = l ;
                            ColmodFlag [j] = EMPTY ;
                        }
                    }
                    if ( PrintLevel )
                    {
                        printf ("free %ld -> bound\n", (LONG) j) ;
                    }
                    if ( xjnew < loj ) /* xj at lower bound */
                    {
                        cerr = PASAMAX (cerr, cj) ;
                        ib [j] = -1 ;
                        Wc [j] = xjnew - loj ;
                        Wlo [j] = PPZERO ;
                        if ( hiExists ) Whi [j] = hij - loj ;
                        for (p = p0; p < q; p++)
                        {
                            Wb [Ai [p]] -= loj*Ax [p] ;
                        }
                        dx [j] = loj - xj ;
                    }
                    else               /* xj at upper bound */
                    {
                        cerr = PASAMAX (cerr, -cj) ;
                        ib [j] = 1 ;
                        Wc [j] = xjnew - hij ;
                        Whi [j] = PPZERO ;
                        if ( loExists ) Wlo [j] = loj - hij ;
                        for (p = p0; p < q; p++)
                        {
                            Wb [Ai [p]] -= hij*Ax [p] ;
                        }
                        dx [j] = hij - xj ;
                    }
                }
                else /* xj is free */
                {
                    if ( use_cholmod && ibj ) /* xj previously bound, now free*/
                    {
                        k = ColmodFlag [j] ;
                        /*If j is not scheduled to be free or bound, free it.*/
                        if ( k == EMPTY )
                        {
                            ColmodList [ncoladd] = j ;
                            ColmodFlag [j] = ncoladd ;
                            ncoladd++ ;
                        }
                        else /* xj to be bound, remove from bind list */
                        {
                            l = ColmodList [ncol-ncoldel] ;
                            ColmodList [k] = l ;
                            ColmodFlag [l] = k ;
                            ColmodFlag [j] = EMPTY ;
                            ncoldel-- ;
                        }
                    }
                    if ( PrintLevel )
                    {
                        printf ("bound %ld ->free\n", (LONG) j) ;
                    }
                    F [nf++] = j ;
                    ib [j] = 0 ;
                    Wc [j] = PPZERO ; 
                    dx [j] = cj ;
                    cerr = PASAMAX (cerr, fabs (cj)) ;
                    /* translate bounds and determine bound violation error */
                    if ( loExists ) Wlo [j] = loj - xjnew ;
                    if ( hiExists ) Whi [j] = hi [j] - xjnew ;
                    for (p = p0; p < q; p++)
                    {
                        PASAINT   const ai = Ai [p] ;
                        PASAFLOAT const ax = Ax [p] ;
                        Wb [ai] -= xjnew*ax ;
                        if ( ir [ai] <= nsingni ) /* row is active */
                        {
                            Annz++ ;
                            WD [ai] += ax*ax ;
                            k = AFTp [ai] + AFTnz [ai]++ ;
                            AFTi [k] = j ;
                            AFTx [k] = ax ;
                        }
                    }
                }
            }
    
            Work->nf = nf ;
            ATnz = 0 ;    /* number of nonzeros in active rows of full matrix */
            nactive = 0 ; /* number of active rows */
            Ll = nsingni1 ;
            Ul = nsingni2 ;
            Rl = nrow ;
            prevblkL = prevblkU = EMPTY ;
            blks = Work->blks ;
            if ( use_cholmod )
            {
                for (k = 0; k < blks; k++)
                {
                    lstart [k] = nsingni1 ;
                    ustart [k] = nsingni2 ;
                }
            }
            /* continue to build Wb by accounting for inequalites/singletons */
            j = 1 ;
            if ( nsingni )
            {
                for (i = 0; i < nrow; i++)
                {
                    t = Wb [i] ;
                    PASAINT const iri = ir [i] ;
                    if ( iri <= nsingni )        /* row is active */
                    {
                        nactive++ ;
                        ATnz += ATp [i+1] - ATp [i] ;
                        Rl = i ;
                        if ( ni )
                        {
                            if ( iri < 0 )
                            {
                                PASAINT const sing = -iri ;
                                t += bl [sing] ;
                                Ll = sing ;
                                SET_LSTART(sing) ;
                            }
                            else if ( iri > 0 ) /* iri > 0 at upper bound */
                            {
                                t += bu [iri] ;
                                Ul = iri ;
                                SET_USTART(iri) ;
                            }
                        }
                        else if ( iri ) /* active row with column singletons*/
                        {
                            PASAINT const qlo = slo [i] ;
                            PASAINT const qhi  = row_sing1 [i] ;
                            if ( qlo ) /* existing lower bounds active */
                            {
                                for (; j <= qlo; j++) t += bl [j] ;
                                Ll = qlo ;
                                SET_LSTART(qlo) ;
                            }
                            /* j < qhi => shi not 0, existing upper bounds */
                            if ( j < qhi)
                            {
                                /* singleton associated with upper bound */
                                Ul = j ;
                                SET_USTART(j) ;
                                for (; j < qhi; j++) t += bu [j] ;
                            }
                        }
                        /* this also handles the case iri = 0, simple equation*/
                        berr = PASAMAX (berr, fabs (t)) ;
                        Wb [i] = t - sigma*lambda [i] ;
                    }
                    else /* currently dropped equation */
                    {
                        PASAINT addeq = FALSE ;
                        PASAINT const sing = iri - nsingni ;
                        if ( nsing ) /* add column singleton terms into t */
                        {
                            Q1 = row_sing1 [i] ;
                            Q0 = j ;
                            ASSERT (j == row_sing [i]) ;
                            for (; j < sing; j++) t += bl [j] ;
                            shii = ++j ; /* skip singleton */
                            for (; j < Q1; j++)   t += bu [j] ;
                        }
                        if ( (s = t + bl [sing]) > PASAZERO )
                        {
                            berr = PASAMAX (berr, s) ;
                        }
                        else if ( (s = t + bu [sing]) < PASAZERO )
                        {
                            berr = PASAMAX (berr, -s) ;
                        }
    
                        /* when activating constraints, also need to take into
                           account the proximal term */
                        t -= sigma*lambda [i] ;
    
                        /* check if increasing lambda_i increases dual */
                        if ( (s = t + bl [sing]) > PPZERO )
                        {
                            if ( PrintLevel )
                            {
                                printf ("dropped equation %ld, "
                                        "singleton %ld -> active lower "
                                        "(violation %e bl: %e)\n",
                                        (LONG) i, (LONG) sing, s, bl [sing]);
                            }
                            /* activate lower bound */
                            t = s ;
                            ADD_SING_IN_LLINK(sing) ;
                            SET_LSTART(sing) ;
                            addeq = TRUE ;
                            /* for column singletons, also update upper bound */
                            if ( nsing )
                            {
                                ir [i] = 1 ;
                                slo [i] = sing ;
                                if ( shii == Q1 ) shi [i] = 0 ;
                                else
                                {
                                    shi [i] = shii ;
                                    ADD_SING_IN_ULINK(shii) ;
                                    SET_USTART_SIMPLE(shii) ;
                                }
                            }
                            else
                            {
                                ir [i] = -sing ;
                            }
                        }
                        /* check if decreasing lambda_i increases dual */
                        else if ( (s = t + bu [sing]) < PPZERO )
                        {
                            if ( PrintLevel )
                            {
                                printf ("dropped equation %ld, "
                                        "singleton %ld -> active upper "
                                        "(violation %e, bu: %e)\n",
                                        (LONG) i, (LONG) sing, -s, bu [sing]);
                            }
                            /* activate upper bound */
                            t = s ;
                            ADD_SING_IN_ULINK(sing) ;
    
                            SET_USTART(sing) ;
                            addeq = TRUE ;
                            /* for column singletons, also update lower bound */
                            if ( nsing )
                            {
                                ir [i] = 1 ;
                                shi [i] = sing ;
                                if ( sing == Q0 ) slo [i] = 0 ;
                                else
                                {
                                    k = slo [i] = sing - 1 ;
                                    ADD_SING_IN_LLINK(k) ;
                                    SET_LSTART_SIMPLE(k) ;
                                }
                            }
                            else
                            {
                                ir [i] = sing ;
                            }
                        }
                        Wb [i] = t ; /* the final right side */
                        if ( addeq ) /* the equation is active */
                        {
                            nactive++ ;
                            ATnz += ATp [i+1] - ATp [i] ;
                            ADD_ROW_IN_RLINK(i) ;
                            if ( use_cholmod )
                            {
                                l = RowmodFlag [i] ;
                                /* if row missing from factor, add it */
                                if ( l == EMPTY )
                                {
                                    nrowadd++ ;
                                    RowmodList [nrow-nrowadd] = i ;
                                    RowmodFlag [i] = nrow-nrowadd ;
                                }
                                else /* remove row from delete list */
                                {
                                    nrowdel-- ;
                                    m = RowmodList [nrowdel] ;
                                    RowmodList [l] = m ;
                                    RowmodFlag [m] = l ;
                                    RowmodFlag [i] = EMPTY ;
                                }
                            }
                            t = PPZERO ;
                            PASAINT const q = ATp [i+1] ;
                            PASAINT l0 = l = AFTp [i] ;
                            for (p = ATp [i]; p < q; p++)
                            {
                                PPINT   const aj = ATi [p] ;
                                if ( !ib [aj] ) /* the column is free */
                                {
                                    Annz++ ;
                                    PPFLOAT const ax = ATx [p] ;
                                    t += ax*ax ;
                                    AFTx [l] = ax ;
                                    AFTi [l] = aj ;
                                    l++ ;
                                }
                            }
                            WD [i] += t ;  /* new diagonal element for SSOR */
                            AFTnz [i] = l - l0 ;
                        }
                    }
                }
            }
            else /* pure equality constraints */
            {
                berr = pasa_sup_normx (Wb, nrow) ;
                pasa_daxpy (Wb, lambda, -sigma, nrow) ;
            }
            Work->ATnz = ATnz ;
            Work->Annz = Annz ;
            Work->nactive = nactive ;
            Work->nrowadd = nrowadd ;
            Work->nrowdel = nrowdel ;
            Work->ncoladd = ncoladd ;
            Work->ncoldel = ncoldel ;
        }
        else /* do not bind constraints and add rows, instead recompute
                W->b, c, lo, and hi for pproj */
        {
            p = 0 ;
            for (j = 0; j < ncol; j++)
            {
                cj = -c [j] ;
                PASAINT   const  q = Ap [j+1] ;
                PASAFLOAT const xj = x [j] ;
                int const ibj = ib [j] ;
                if ( ibj < 0 )
                {
                    PASAFLOAT const loj = lo [j] ;
                    for (; p < q; p++)
                    {
                        PASAINT   const ai = Ai [p] ;
                        PASAFLOAT const ax = Ax [p] ;
                        cj += lambda [ai]*ax ;
                        PASAFLOAT const AX = loj*ax ;
                        Wb [ai] -= AX ;
                        absAx [ai] += fabs (AX) ;
                    }
                    cerr = PASAMAX (cerr, cj) ;
                    Wc  [j] = cj - xj + loj ;
                    Wlo [j] = PPZERO ;
                    if ( hiExists ) Whi [j] = hi [j] - loj ;
                    dx [j] = loj - xj ;
                }
                else if ( ibj > 0 )
                {
                    PASAFLOAT const hij = hi [j] ;
                    for (; p < q; p++)
                    {
                        PASAINT   const ai = Ai [p] ;
                        PASAFLOAT const ax = Ax [p] ;
                        cj += lambda [ai]*ax ;
                        PASAFLOAT const AX = hij*ax ;
                        Wb [ai] -= AX ;
                        absAx [ai] += fabs (AX) ;
                    }
                    cerr = PASAMAX (cerr, -cj) ;
                    Wc  [j] = cj - xj + hij ;
                    Whi [j] = PPZERO ;
                    if ( loExists ) Wlo [j] = lo [j] - hij ;
                    dx [j] = hij - xj ;
                }
                else
                {
                    p0 = p ;
                    for (; p < q; p++)
                    {
                        cj += lambda [Ai [p]]*Ax [p] ;
                    }
                    dx [j] = cj ;
                    cerr = PASAMAX (cerr, fabs (cj)) ;
                    PASAFLOAT const xjnew = xj + cj ;
                    if ( loExists ) Wlo [j] = lo [j] - xjnew ;
                    if ( hiExists ) Whi [j] = hi [j] - xjnew ;
                    Wc [j] = PPZERO ; 
                    for (p = p0; p < q; p++)
                    {
                        PASAINT   const ai = Ai [p] ;
                        PASAFLOAT const ax = Ax [p] ;
                        PASAFLOAT const AX = xjnew*ax ;
                        Wb [ai] -= AX ;
                        absAx [ai] += fabs (AX) ;
                        if ( ir [ai] <= nsingni ) /* row is active */
                        {
                            WD [ai] += ax*ax ;
                        }
                    }
                }
            }

            /* continue to build Wb by accounting for inequalites/singletons */
            j = 1 ;
            if ( nsingni )
            {
                PASAFLOAT R = PASAZERO ;
                for (i = 0; i < nrow; i++)
                {
                    t = Wb [i] ;
                    PASAINT const iri = ir [i] ;
                    if ( iri <= nsingni )        /* row is active */
                    {
                        if ( absAx [i] > R ) R = absAx [i] ;
                        if ( ni )
                        {
                            if ( iri < 0 )
                            {
                                PASAINT const sing = -iri ;
                                t += bl [sing] ;
                            }
                            else if ( iri > 0 ) /* iri > 0 at upper bound */
                            {
                                t += bu [iri] ;
                            }
                        }
                        else if ( iri ) /* active row with column singletons*/
                        {
                            PASAINT const qlo = slo [i] ;
                            PASAINT const qhi  = row_sing1 [i] ;
                            if ( qlo ) /* existing lower bounds active */
                            {
                                for (; j <= qlo; j++) t += bl [j] ;
                            }
                            /* j < qhi => shi not 0, existing upper bounds */
                            if ( j < qhi)
                            {
                                /* singleton associated with upper bound */
                                for (; j < qhi; j++) t += bu [j] ;
                            }
                        }
                        /* this also handles the case iri = 0, simple equation*/
                        berr = PASAMAX (berr, fabs (t)) ;
                        Wb [i] = t - sigma*lambda [i] ;
                    }
                    else /* currently dropped equation */
                    {
                        PASAINT const sing = iri - nsingni ;
                        if ( nsing ) /* add column singleton terms into t */
                        {
                            ASSERT (j == row_sing [i]) ;
                            for (; j < sing; j++)
                            {
                                t += bl [j] ;
                            }
                            PASAINT const qhi = row_sing1 [i] ;
                            /* skip singleton */
                            for (j++; j < qhi; j++)
                            {
                                t += bu [j] ;
                            }
                        }
                        if ( (s = t + bl [sing]) > PASAZERO )
                        {
                            berr = PASAMAX (berr, s) ;
                        }
                        else if ( (s = t + bu [sing]) < PASAZERO )
                        {
                            berr = PASAMAX (berr, -s) ;
                        }
                        /* the final right side */
                        Wb [i] = t - sigma*lambda [i] ;
                    }
                }
                if ( R == PASAZERO ) R = PASAONE ;
                if ( use_cholmod ) Work->absAx = R ;
                else               Work->absAxk= R ;
            }
            else /* pure equality constraints */
            {
                berr = pasa_sup_normx (Wb, nrow) ;
                pasa_daxpy (Wb, lambda, -sigma, nrow) ;
                t = pasa_sup_normx (absAx, nrow) ;
                if ( t == PASAZERO ) t = PASAONE ;
                if ( use_cholmod ) Work->absAx = t ;
                else               Work->absAxk= t ;
            }
            if ( !use_cholmod ) Work->absAx = PASAZERO ;
        }
        berr /= (epsilon*xmax) ;
        cerr /= lmax ;
        err = PASAMAX (cerr, berr) ;
        if ( PrintLevel >= 1 )
        {
            printf ("after line search cerr: %e berr: %e xmax: %e lmax: %e "
                    "active rows: %ld nf: %ld\n", cerr, berr, xmax, lmax,
                    (LONG) Work->nactive, (LONG) Work->nf) ;
        }
        if ( err <= grad_tol )
        {
            /* include the update to x computed above, when error computed */
            pasa_step (x, x, dx, PASAONE, ncol) ;
            break ;
        }
        Work->errdual = err ;
        /* the point we project onto the polyhedron is y = -c which has been
           scaled above */
        pasa_scale (Prob->y, c, -PASAONE, ncol) ;
#ifndef NDEBUG
        pasa_copyx (checkb, b, nrow) ;
        p = 0 ;
        for (j = 0; j < ncol; j++)
        {
            PASAINT   const  q = Ap [j+1] ;
            PASAFLOAT const xj = x [j] ;
            for (; p < q; p++)
            {
                checkb [Ai [p]] -= Ax [p]*xj ;
            }
        }
        if ( loExists ) pasa_step (checklo, lo, x, -PASAONE, ncol) ;
        if ( hiExists ) pasa_step (checkhi, hi, x, -PASAONE, ncol) ;
#endif
    }

#ifndef NDEBUG
    pasa_free (checkb) ;
    if ( loExists ) pasa_free (checklo) ;
    if ( hiExists ) pasa_free (checkhi) ;
#endif
    Com->E = err ;
    Com->pasastat->gpit = iter ;

    /* convert to user's problem (remove scale factors and permutations) */
    PASAFLOAT add, Hij, Loj ;
    scale_x = PASAONE/(scale_x*epsilon) ;
    scale_l = PASAONE/scale_l ;
    PASAINT *rowperm = ppcom->Prob->rowperm ;
    PASAINT *colperm = ppcom->Prob->colperm ; /* after fixed removed */
    /* if there were fixed variables, then temp_ifree contains the
       reordering associated with the removal of the fixed variables */
    PASAINT *ifree = Com->temp_ifree ;
    PASAFLOAT *lambdatemp = ppcom->Work->arrayd ;
    PASAFLOAT *userx = Com->userx ;

    /* compute the dual multipliers */
    if ( use_cholmod )
    {
        for (i = 0; i < nrow; i++)
        {
            k = rowperm [i] ;
            lambdatemp [k] = scale_l * lambda [i] * Com->RowScale [k] ;
        }
        pasa_copyx (lambda, lambdatemp, nrow) ;
    }
    else /* iterative, no row permutation */
    {
        for (i = 0; i < nrow; i++)
        {
            lambda [i] *= scale_l * Com->RowScale [i] ;
        }
    }

    /* compute the part of x not associated with column singletons */
    PPFLOAT const *userlo = Com->pasadata->lo ;
    PPFLOAT const *userhi = Com->pasadata->hi ;
    if ( use_cholmod )
    {
        if ( ifree == NULL )
        {
            for (j = 0; j < ncol; j++)
            {
                i = colperm [j] ;
                t = Com->offset [i] + x [j]*Com->ColScale [i]*scale_x ;
                if      ( loExists && (t < userlo [i]) ) t = userlo [i] ;
                else if ( hiExists && (t > userhi [i]) ) t = userhi [i] ;
                userx [i] = t ;
            }
        }
        else
        {
            for (j = 0; j < ncol; j++)
            {
                /* user variable for pproj's column j */
                i = colperm [j] ;
                /* user location accounting for bound variables*/
                k = ifree [i] ;
                t = Com->offset [i] + x [j]*Com->ColScale [i]*scale_x ;
                if      ( loExists && (t < userlo [k]) ) t = userlo [k] ;
                else if ( hiExists && (t > userhi [k]) ) t = userhi [k] ;
                userx [k] = t ;
            }
        }
    }
    else /* iterative method */
    {
        if ( ifree != NULL ) /* fixed variables were removed */
        {
            for (j = 0; j < ncol; j++)
            {
                /* put the free variables back into userx */
                k = ifree [j] ;
                t = Com->offset [j] + x [j]*Com->ColScale [j]*scale_x ;
                if      ( loExists && (t < userlo [k]) ) t = userlo [k] ;
                else if ( hiExists && (t > userhi [k]) ) t = userhi [k] ;
                userx [k] = t ;
            }
        }
        else
        {
            for (j = 0; j < ncol; j++)
            {
                /* put the free variables back into userx */
                userx [j] = Com->offset [j] + x [j]*Com->ColScale [j]*scale_x ;
                if      ( loExists && (t < userlo [j]) ) t = userlo [j] ;
                else if ( hiExists && (t > userhi [j]) ) t = userhi [j] ;
                userx [j] = t ;
            }
        }
    }

    /* now compute the part of x associated with column singletons
       The statement SET_USERX1 below represents the following block of code.
       Note that the variable col corresponds to a column of the original
       matrix. If col < 0, it is not a column of the matrix; it corresponds
       to the right and left side bounds for an equation and hence there is
       no corresponding component of userx to compute.

           col = Com->sj [l] ;
           if ( col >= 0 )
           {
               if ( Com->sx [l] > PASAZERO )
               {
                   userx [col] = Com->shi[l] ;
               }
               else
               {
                   userx [col] = Com->slo[l] ;
               }
           }

        SET_USERX2 is the same except that shi and slo are switched. */

    PASAINT const *singperm = Com->singperm ;
    for (j = 1; j <= nsing; )
    {
        i = ineq_row [j] ;                /* row in the permuted matrix */
        PASAINT const ksing = ir [i] ;
        /* row in original matrix depends on whether cholmod was used */
        PASAINT const row = (use_cholmod) ? rowperm [i] : i ;

        if ( ksing == 1 ) /* row is active, all singletons at bounds */
        {
            k = userRowSing [row] - j ;
            /* !! NOTE: Work->slo is a singleton index while Com->slo is the
                        lower bound for a column singleton.
                        Similar for Prob->shi and Com->shi !! */
            PASAINT const q = slo [i] ;
            for (; j <= q; j++) /*variables at lower bounds */
            {
                /* l = singleton number in original matrix when l > 0 */
                l = singperm [j+k] ;
                if ( l < 0 ) /* compression has occurred */
                {
                    l = -(l+1) ; /* singleton number in original matrix */
                    while ( l != EMPTY )    /* l empty <=> last compression */
                    {
                        SET_USERX1
                        l = Com->si [l] ;
                    }
                }
                else /* no compression */
                {
                    SET_USERX1
                }
            }
            PASAINT const qtop = row_sing1 [i] ;
            for (; j < qtop; j++) /*variable at upper bounds */
            {
                /* l = singleton number in original matrix when l > 0 */
                l = singperm [j+k] ;
                if ( l < 0 ) /* compression has occurred */
                {
                    l = -(l+1) ; /* singleton number in original matrix */
                    while ( l != EMPTY )    /* l empty <=> last compression */
                    {
                        SET_USERX2 ;
                        l = Com->si [l] ;
                    }
                }
                else /* no compression */
                {
                    SET_USERX2 ;
                }
            }
        }
        else /* row inactive */
        {
            /* compute the singleton associated with inactive row */
            PASAINT const sing = ksing - nsing ;
            /* in pproj problem, evaluate the bound singletons and their sum */
            t = PASAZERO ;
            k = userRowSing [row] - j ;
            PASAINT const q = sing ;
            for (; j < q; j++) /*variables at lower bounds */
            {
                /* l = singleton number in original matrix when l > 0 */
                l = singperm [j+k] ;
                t += bl [j] ;
                if ( l < 0 ) /* compression has occurred */
                {
                    l = -(l+1) ; /* singleton number in original matrix */
                    while ( l != EMPTY )    /* l empty <=> last compression */
                    {
                        SET_USERX1 ;
                        l = Com->si [l] ;
                    }
                }
                else /* no compression */
                {
                    SET_USERX1 ;
                }
            }
            PASAINT const qtop = row_sing1 [i] ;
            for (j++; j < qtop; j++) /*variable at upper bounds */
            {
                /* l = singleton number in original matrix when l > 0 */
                l = singperm [j+k] ;
                t += bu [j] ;
                if ( l < 0 ) /* compression has occurred */
                {
                    l = -(l+1) ; /* singleton number in original matrix */
                    while ( l != EMPTY )    /* l empty <=> last compression */
                    {
                        SET_USERX2 ;
                        l = Com->si [l] ;
                    }
                }
                else /* no compression */
                {
                    SET_USERX2 ;
                }
            }

            /* take into account b and the nonsingleton part of the matrix */
            t += Prob->b [i] ;
            PASAINT const q2 = ATp [i+1] ;
            for (p = ATp [i]; p < q2; p++)
            {
                t -= ATx [p]*x [ATi [p]] ;
            }
            /* remove the scaling from the right side */
            t *= scale_x ;

            /* we now have the right side of the permuted problem; the only
               scaling is RowScale, however, this scaling was previously
               applied to Com->sx [l]. Now solve for the column singleton. */
            l = singperm [sing+k] ;
            if ( l < 0 ) /* compression has occurred */
            {
                l = -(l+1) ; /* singleton number in original matrix */
                   /* The relationship between the solution yj in the
                   transformed problem that we compute, and the solution xj
                   in the original problem is yj = -sx*(xj - oj), oj = offset.
                   Also, in the transformed equation, the term appears as -yj.
                   If t denotes the right side of the transformed equation
                   after moving all terms to the right except for the -yj term,
                   then we have -yj = t = sx*(xj-oj).
                   Hence, xj = oj + t/sx. When compression occurs, we have
                   sum_j sx_j(xj-oj) = t because each yj term in the
                   transformed problem is a sum of terms in the original
                   problem. Here xj lies between its bounds loj and hij,
                   and oj is either 0 or loj or hij, where oj is chosen so
                   xj - oj has 0 in its range of possible values. To solve
                   for xj, proceed as follows: If t < 0, then in the
                   successive terms in sum_j, choose xj to make the j-th
                   term as small as possible, until the partial sum reaches t.
                   Then set xj = oj for the remaining terms in the sum.
                   If t > 0, then choose xj to make the j-th term as large as
                   possible, until the partial sum reaches t, then set xj = oj
                   for the remaining terms in the sum.  */
                while ( l != EMPTY )
                {
                    col = Com->sj [l] ; /* column in original matrix */
                    Loj = Com->slo [l] ;
                    Hij = Com->shi [l] ;
                    /* compute the offset denoted oj below */
                    if      ( Loj > PASAZERO ) oj = Loj ;
                    else if ( Hij < PASAZERO ) oj = Hij ;
                    else                       oj = PASAZERO ;

                    if ( t < PASAZERO )
                    {
                        PASAFLOAT const sx  = Com->sx  [l] ;
                        /* increase yj until t + yj >= 0 (use singhi,
                           which depends on sign of ax, to boost yj) */
                        if ( sx > PASAZERO )
                        {
                            add = sx*(oj-Loj) ; /* >= 0 since oj in [Loj, Hij]*/
                        }
                        else /* sx < 0 */
                        {
                            add = sx*(oj-Hij) ; /* >= 0 since oj in [Loj, Hij]*/
                            Loj = Hij ;
                        }
                        /* if we have added enough positive terms to t to
                           achieve positivity, then we are done */
                        if ( add + t >= PASAZERO )
                        {
                            /* If col < 0, then this variable corresponds to
                               a strict inequality that was converted
                               to an equality by creating a new variable.
                               Since this newly created variable is not one
                               of the user's variables, there is no need to
                               assign it a value. */
                            if ( col >= 0 )
                            {
                                /* compute the value of xj that makes this
                                   equation an equality */
                                t = oj + t/sx ;
                                /* chop x if it is outside the bounds due to
                                   rounding errors */
                                if ( loExists && (t < userlo [col]) )
                                {
                                    t = userlo [col] ;
                                }
                                else if ( hiExists && (t > userhi [col]) )
                                {
                                    t = userhi [col] ;
                                }
                                userx [col] = t ;
                            }
                            l = Com->si [l] ;
                            break ;
                        }
                    }
                    else /* t >= 0 */
                    {
                        /* decrease yj until t + yj <= 0 (use singlo,
                           which depends on sign of ax, to lower yj) */
                        PASAFLOAT const sx  = Com->sx  [l] ;
                        if ( sx > PASAZERO )
                        {
                            add = sx*(oj-Hij) ; /* <= 0 since oj in [Loj, Hij]*/
                            Loj = Hij ;
                        }
                        else /* sx < 0 */
                        {
                            add = sx*(oj-Loj) ; /* <= 0 since oj in [Loj, Hij]*/
                        }
                        /* if we have added enough negative terms to t to
                           achieve negativity, then we are done */
                        if ( add + t <= PASAZERO ) /* done, t has been reached*/
                        {
                            /* If col < 0, then this variable corresponds to
                               a strict inequality that was converted
                               to an equality by creating a new variable.
                               Since this newly created variable is not one
                               of the user's variables, there is no need to
                               assign it a value. */
                            if ( col >= 0 )
                            {
                                t = oj + t/sx ;
                                if ( loExists && (t < userlo [col]) )
                                {
                                    t = userlo [col] ;
                                }
                                else if ( hiExists && (t > userhi [col]) )
                                {
                                    t = userhi [col] ;
                                }
                                userx [col] = t ;
                            }
                            l = Com->si [l] ;
                            break ;
                        }
                    }
                    t += add ;
                    /* the correct bound was set to Loj above */
                    if ( col >= 0 ) userx [col] = Loj ;
                    l = Com->si [l] ;
                }

                /* set the remaining variables to the offset */
                while ( l != EMPTY )
                {
                    oj = PASAZERO ;
                    if      ( Com->slo [l] > PASAZERO ) oj = Com->slo [l] ;
                    else if ( Com->shi [l] < PASAZERO ) oj = Com->shi [l] ;

                    col = Com->sj [l] ;
                    if ( col >= 0 ) userx [col] = oj ;
                    l = Com->si [l] ;
                }
            }
            else /* no compression oj */
            {
                /* The problem fed to pproj involved an offset. We need
                   to compute that offset in order to solve for the
                   column singleton.  The offset is denoted oj below. */
                oj = Com->slo [l] ;
                if      (           oj > PASAZERO ) ;
                else if ( Com->shi [l] < PASAZERO ) oj = Com->shi [l] ;
                else                                oj = PASAZERO ;
                /* xj = offset + rhsj/sx where rhsj denote all the terms in
                   the equation moved to the right side of the equation. This
                   is stored in the variable t above. */
                col = Com->sj [l] ;
                if ( col >= 0 )
                {
                    t = oj + t/Com->sx [l] ;
                    if ( loExists && (t < userlo [col]) )
                    {
                        t = userlo [col] ;
                    }
                    else if ( hiExists && (t > userhi [col]) )
                    {
                        t = userhi [col] ;
                    }
                    userx [col] = t ;
                }
#ifndef NDEBUG
                if ( col >= 0 )
                {
                    t = userx [col] ;
                    if ( (t > Com->shi [l]) || (t < Com->slo [l]) )
                    {
                        printf ("Computed singleton outside bounds\n") ;
                        printf ("Computed: %e lo: %e hi: %e\n",
                                 t, Com->slo [l], Com->shi [l]) ;
                        /*pproj_error (-1, __FILE__, __LINE__, "stop\n") ;*/
                    }
                }
#endif
            }
        }
    }

    /* store objective value */
    Com->f = pasa_dot (Com->userc, Com->userx, Com->ucol) ;

    /* store iteration count and test for error */
    Com->pasastat->gpit = iter ;
    if ( iter >= gpmaxit )
    {
        return (PASA_ITERATIONS_EXCEED_MAXITS_IN_GRAD_PROJ) ;
    }
    return (PASA_ERROR_TOLERANCE_SATISFIED) ;
}

/* ==========================================================================
   === pasa_errLP ===========================================================
   ==========================================================================
   Consider the following LP:

       min c'x subject to bl <= Ax <= bu, lo <= x <= hi

   At optimality, the constraints are satisfied and there exists lambda
   with the following properties:

       1. If A_j' lambda < c_j, then x_j = hi_j
       2. If A_j' lambda > c_j, then x_j = lo_j
       3. If bl_i < A_i'x < bu_i, then lambda_i = 0
       4. If A_i'x = bu_i, then lambda_i <= 0
       5. If A_i'x = bl_i, then lambda_i >= 0
   ========================================================================== */
void pasa_errLP
(
    PASAFLOAT epsilon, /* some data is scaled by epsilon in gradprojLP */
    PASAcom      *Com  /* common data for PASA */
)
{
    PASAINT i, j, k, p ;
    PASAFLOAT d, e, Eprimal, Elambda, Emax, loj, hij, Lmax, max_absAx, Xmax, t,
             *absAx ;
    PPwork   *Work ;  /* work structure in PPROJ */
    PASAparm *Parm ;  /* parameters for PASA */
    PPprob   *Prob ;

    Prob = Com->ppcom->Prob ;
    Parm = Com->pasaparm ; /* parameters */
    int       const  PrintLevel = Parm->PrintLevel ;
    int       const    loExists = Com->loExists ;
    int       const    hiExists = Com->hiExists ;
    PASAINT   const        ncol = Com->ncol ;
    PASAINT   const        nrow = Com->nrow ;
    PASAINT   const       nsing = Com->pasadata->ppdata->nsing ;
    PASAINT   const          ni = Com->pasadata->ppdata->ni ;
    PASAFLOAT const          *x = Com->x ;
    PASAFLOAT const          *c = Com->c ;
    PASAFLOAT const     *lambda = Com->lambda ;
    PASAINT   const         *Ap = Prob->Ap ;
    PASAINT   const         *Ai = Prob->Ai ;
    PASAFLOAT const         *Ax = Prob->Ax ;
    PASAFLOAT const          *b = Prob->b ;
    PASAFLOAT const         *lo = Prob->lo ;
    PASAFLOAT const         *hi = Prob->hi ;
    PASAFLOAT const         *bl = Prob->bl ;
    PASAFLOAT const         *bu = Prob->bu ;
    PASAINT   const   *row_sing = Prob->row_sing ;
    PPFLOAT   const      *singc = Prob->singc ;
    PPFLOAT   const     *singlo = Prob->singlo ;
    PPFLOAT   const     *singhi = Prob->singhi ;
    PASAINT   const  *row_sing1 = row_sing+1 ;

    Work   = Com->ppcom->Work ;
    PASAINT   const         *ir = Work->ir ;
    PASAFLOAT                *y = Work->arrayd ;
    pasa_initx (y, PASAZERO, nrow) ;
    absAx = y+nrow ;
    pasa_initx (absAx, PASAZERO, nrow) ;
    Xmax = PASAZERO ;
    Emax = PASAZERO ;
    if ( PrintLevel > 1 ) printf ("Primal Bound Errors:\n") ;
    k = 0 ;
    for (j = 0; j < ncol; j++)
    {
        e = PASAZERO ;
        if ( loExists ) loj = lo [j] ;
        else            loj = -PASAINF ;
        if ( hiExists ) hij = hi [j] ;
        else            hij = PASAINF ;
        if ( x [j] < loj )
        {
            t = loj ;
            e = t - x [j] ;
        }
        else if ( x [j] > hij )
        {
            t = hij ;
            e = x [j] - t ;
        }
        else
        {
            t = x [j] ;
        }
        if ( PrintLevel > 2 )
        {
            printf ("%ld %e %e %e %e\n",
                  (LONG) j, e/epsilon, loj/epsilon, x [j]/epsilon, hij/epsilon);
        }
        if ( e > Emax ) Emax = e ;
        if ( fabs (t) > Xmax )
        {
            Xmax = fabs (t) ;
        }
        PASAINT const l = Ap [j+1] ;
        for (; k < l; k++)
        {
            y [Ai [k]] += t*Ax [k] ;
            absAx [Ai [k]] += fabs (t*Ax [k]) ;
        }
    }

    if ( PrintLevel > 1 )
    {
        printf ("Total: %e relative: %e Xmax: %e\n\n",
                 Emax/epsilon, Emax/Xmax, Xmax/epsilon) ;
        printf ("Primal Equation Errors:\n") ;
    }
    max_absAx = pasa_sup_normx (absAx, nrow) ;
    k = 0 ;
    Eprimal = PASAZERO ;
    Elambda = PASAZERO ;
    for (i = 0; i < nrow; i++)
    {
        e = PASAZERO ;
        if ( ir [i] == 0 ) /* equality constraint, no singletons */
        {
            e = fabs (b [i] - y [i]) ;
            if ( PrintLevel > 2 )
            {
                printf ("%ld %e %e %e %e\n", (LONG) i, e/epsilon,
                        b [i]/epsilon, y [i]/epsilon, b [i]/epsilon) ;
            }
        }
        else if ( ni ) /* there are strict inequality constraints */
        {
            k++ ;
            if ( bl [k] >= y [i] )
            {
                e = bl [k] - y [i] ;
            }
            else if ( y [i] >= bu [k] )
            {
                e = y [i] - bu [k] ;
            }
            if ( PrintLevel > 2 )
            {
                printf ("p: %ld %e %e %e %e\n", (LONG) i, e/epsilon,
                         bl [k]/epsilon, y [i]/epsilon, bu [k]/epsilon) ;
            }
            d = PASAZERO ;
            if ( lambda [i] > PASAZERO )
            {
                d = PASAMIN (lambda [i], fabs (bl [k] - y [i])/epsilon) ;
            }
            else if ( lambda [i] < PASAZERO )
            {
                d = PASAMIN (-lambda [i], fabs (bu [k] - y [i])/epsilon) ;
            }
            if ( PrintLevel > 2 )
            {
                printf ("d: %ld %e %e %e %e\n",
                  (LONG) i, e, bl [k]/epsilon, lambda [i], bu [k]/epsilon) ;
            }
            if ( d > Elambda ) Elambda = d ;
        }
        else if ( nsing ) /* potential column singletons */
        {
            if ( ir [i] == 1 ) /* all singletons at bounds */
            {
                t = b [i] - y [i] ;
                PASAINT const q = Work->slo [i] ;
                /* singletons at lower bounds */
                for (k = row_sing [i]; k <= q; k++)
                {
                    t += singlo [k] ;
                }
                PASAINT const q1 =row_sing1 [i] ;
                /* singletons at upper bounds */
                for (; k < q1; k++)
                {
                    t += singhi [k] ;
                }
                e = fabs (t) ;
                if ( PrintLevel > 2 )
                {
                    printf ("%ld %e %e %e\n",
                    (LONG) i, e/epsilon, b [i]/epsilon, y [i]/epsilon) ;
                }
            }
            else
            {
                PASAINT const ksing = ir [i] - nsing ;
                t = PASAZERO ;
                /* singletons at lower bounds */
                for (k = row_sing [i]; k < ksing; k++)
                {
                    t += singlo [k] ;
                }
                /* singletons at upper bounds */
                for (k++; k < row_sing1 [i]; k++)
                {
                    t += singhi [k] ;
                }
                t += b [i] - y [i] ;
                t = -t ;
                /* t should lie between singlo [ksing] and singhi [ksing] */
                if ( t < singlo [ksing] ) e = singlo [ksing] - t ;
                else                      e = PASAZERO ;
                if ( t > singhi [ksing] )
                {
                    e = PASAMAX (e, t - singhi [ksing]) ;
                }
                if ( PrintLevel > 2 )
                {
                    printf ("psing: %ld %e %e %e %e\n", (LONG) i, e/epsilon,
                         singlo [ksing]/epsilon, t/epsilon,
                         singhi [ksing]/epsilon) ;
                }
                /* dual error */
                d = fabs (lambda [i] - singc [ksing]) ;
                if ( PrintLevel > 2 )
                {
                    printf ("dsing: %ld %e %e %e\n",
                            (LONG) i, d, lambda [i],singc [ksing]) ;
                }
                if ( d > Elambda ) Elambda = d ;
            }
        }
        if ( e > Eprimal ) Eprimal = e ;
    }

    if ( PrintLevel > 1 )
    {
        printf ("Primal Total: %e Relative: %e absAx: %e\n",
                        Eprimal/epsilon, Eprimal/max_absAx, max_absAx/epsilon) ;
        printf ("Lambda Total: %e Relative: %e\n\n",
                                           Elambda, Elambda*epsilon/max_absAx) ;
        printf ("Dual Equation Errors:\n") ;
    }
    Emax = PASAZERO ;
    p = 0 ;
    for (j = 0; j < ncol; j++)
    {
        PASAINT const q = Ap [j+1] ;
        t = c [j] ;
        for (; p < q; p++)
        {
            t -= Ax [p]*lambda [Ai [p]] ;
        }
        e = PASAZERO ;
        if ( loExists ) loj = lo [j] ;
        else            loj = -PASAINF ;
        if ( hiExists ) hij = hi [j] ;
        else            hij = PASAINF ;
        if ( t > PASAZERO )
        {
            e = PASAMIN (t, fabs (loj - x [j])/epsilon) ;
        }
        else if ( t < PASAZERO )
        {
            e = PASAMIN (-t, fabs (hij - x [j])/epsilon) ;
        }
        if ( e > Emax ) Emax = e ;
        if ( PrintLevel > 2 )
        {
            printf ("%ld %e %e %e %e\n",
                  (LONG) j, e, loj/epsilon, x [j]/epsilon, hij/epsilon) ;
        }
    }
    Lmax = pasa_sup_normx (lambda, nrow) ;
    if ( PrintLevel > 1 )
    {
        printf ("Total: %e Relative: %e lmax: %e\n\n", Emax, Emax/Lmax, Lmax) ;
    }
}

/* ==========================================================================
   === pasa_bound_compress (code not used, can be deleted) ==================
   ==========================================================================
    Check for new active bounds, and compress the problem when they are found.
    Only used during a partial step towards the projection point so potentially
    ib does not correspond to the active constraints at the current iterate.
    This is only employed when use_pproj is TRUE.
   ========================================================================== */
int pasa_bound_compress /* return:
                                   PASA_NEW_ACTIVE
                                   PASA_TRY_PROJECT2
                                   PASA_NO_CHANGE_IN_ACTIVE   */
(
    PASAcom    *Com  /* common variables */
)
{
    int *ib ;
    PASAINT change_in_active, cols, j, k, nc, ncol, ni, row,
           *bound_cols, *ir, *ineq_row ;
    PASAFLOAT *lo, *hi, *x ;
    PPprob *Prob ;
    PPwork *Work ;

    const int PrintLevel = Com->pasaparm->PrintLevel ;
    const int loExists = Com->loExists ;
    const int hiExists = Com->hiExists ;
    if ( PrintLevel >= 2 )
    {
        printf ("bound compress\n") ;
    }

    if ( Com->Aexists == FALSE )
    {
        return (PASA_NO_CHANGE_IN_ACTIVE) ;
    }
    /* otherwise Aexists */

    Prob = Com->ppcom->Prob ;
    ncol = Prob->ncol ;
    ni = Prob->ni ;
    ineq_row = Prob->ineq_row ;

    Work = Com->ppcom->Work ;
    ib = Work->ib ;
    ir = Work->ir ;

    lo = Com->lo ;
    hi = Com->hi ;
    x = Com->x ;

    change_in_active = FALSE ;
    if ( Com->Bounds == FALSE )
    {
        cols = ncol ; /* no bound constraints are present */
    }
    else /* bound constraints are present */
    {
        /* check for new active bounds */
        for (cols = 0; cols < ncol; cols++)
        {
            if ( ib [cols] != 0 )
            {
                /* if a bound was active at the projection point,
                   then change_in_active is TRUE */
                change_in_active = TRUE ;
                if ( ib [cols] < 0 )
                {
                    if ( x [cols] == lo [cols] )
                    {
                        break ;
                    }
                }
                else if ( x [cols] == hi [cols] )
                {
                    break ;
                }
            }
        }
    }
    if ( cols == ncol )
    {
        if ( change_in_active == FALSE )
        {
            /* check whether an inequality became active */
            for (k = 1; k <= ni; k++)
            {
                row = ineq_row [k] ;
                if ( ir [row] <= ni ) /* active inequality */
                {
                    break ;
                }
            }
            if ( k <= ni ) /* an inequality became active */
            {
                /* Project the current iterate onto the feasible set.
                   After this projection, the current factorization
                   corresponds to the current iterate. */
                return (PASA_TRY_PROJECT2) ;
            }
            /* otherwise, there was no change in the active set */
            return (PASA_NO_CHANGE_IN_ACTIVE) ;
        }
        else
        {
            /* else, there was a change in the active bounds, try project2 */
            return (PASA_TRY_PROJECT2) ;
        }
    }

    {
        int blks ;
        PASAINT blk, i, iri, l, nrow, pp, p, q,
               *Ap, *Ai, *Anz, *ATp, *ATi, *colmap, *col_start, *ifree ;
        PASAFLOAT *Ax, *ATx, *b, *bl, *bu, *g ;
        nrow = Com->nrow ;
        colmap = Work->arrayi ; /* borrow temporary space from pproj */
        ifree = Com->ifree ;
        g = Com->g ;
        b  = Com->b ;
        bl = Com->bl ;
        bu = Com->bu ;
        nc = Com->nc ;
        bound_cols = Com->bound_cols ;
        Ap = Prob->Ap ;
        Ai = Prob->Ai ;
        Ax = Prob->Ax ;
        Anz = Prob->Anz ;
        ATp = Work->ATp ;
        ATi = Work->ATi ;
        ATx = Work->ATx ;
#ifndef NDEBUG
        pasa_check_feas (Com) ;
#endif

        /* cols < ncol => at least one constraint is activated.
           delete from A the columns associated with bound variables
           and update the start of each column in the multilevel
           decomposition. cols will count the number of remaining columns in A*/
        blks = Work->blks ;
        col_start = Work->col_start ;

        /* the initial columns are mapped to themselves */
        for (i = 0; i < cols; i++)
        {
            colmap [i] = i ;
        }
        /* When columns are deleted, the column start of each block is
           reduced by the number of proceeding columns that were deleted.
           Find the first block with column start > j. The following code
           terminates since col_start [blks] = number of columns
           in matrix. */
        for (blk = 1; blk <= blks; blk++)
        {
            if ( col_start [blk] > cols )
            {
                break ;
            }
        }
        l = col_start [blk] ;

        /* start of storage for this first dropped column, k increments
           over the compressed matrix, while p increments over the
           original matrix */
        p = k = Ap [cols] ;
        for (j = cols; j < ncol; j++)
        {
            if ( j == l )
            {
                col_start [blk] = cols ;
                blk++ ;
                while ( col_start [blk] == l )
                {
                    col_start [blk] = cols ;
                    blk++ ;
                }
                l = col_start [blk] ;
            }
            q = Ap [j+1] ;
            /* retained column in the following cases: */
            if ( (ib [j] == 0) || ((ib [j] < 0) && (x [j] != lo [j]))
                               || ((ib [j] > 0) && (x [j] != hi [j])) )
            {
                /* retain associated x, g, lo, hi, and ifree entries */
                x [cols] = x [j] ;
                g [cols] = g [j] ;
                if ( loExists ) lo [cols] = lo [j] ;
                if ( hiExists ) hi [cols] = hi [j] ;
                ib[cols] = ib[j] ;
                ifree [cols] = ifree [j] ;
                colmap [j] = cols ;
                Anz [cols] = Anz [j] ;

                /* save column j in matrix */
                Ap [cols+1] = Ap [cols] + q - p ;
                cols++ ;
                for (; p < q; p++)
                {
                    Ai [k] = Ai [p] ;
                    Ax [k] = Ax [p] ;
                    k++ ;
                }
            }
            else /* bound column */
            {
                i = ifree [j] ;
                PASAFLOAT const xj = x [j] ;
                Com->userx [i] = xj ;
                colmap [j] = -1 ;
                if ( ib [j] > 0 )
                {
                    bound_cols [nc] = i ;
                }
                else
                {
                    bound_cols [nc] = -(i + 1) ;
                }
                nc++ ;
                /* The replacement column is free in the reduced matrix,
                   and the right side of constraints is adjusted */
                for (; p < q; p++)
                {
                    i = Ai [p] ;
                    PASAFLOAT const Axj = Ax [p]*xj ;
                    iri = ir [i] ;
                    if ( iri == 0 ) /* equality constraint */
                    {
                        b [i] -= Axj ;
                    }
                    else            /* inequality constraint */
                    {
                        if ( iri < 0 )
                        {
                            iri = -iri ;
                        }
                        else if ( iri > ni )
                        {
                            iri -= ni ;
                        }
                        bl [iri] -= Axj ;
                        bu [iri] -= Axj ;
                    }
                }
            }
        }
        Prob->ncol = cols ;
        Work->A->ncol = cols ;
        Work->A->nzmax = Ap [cols] ;
        Com->nf = cols ;
        Com->nc = nc ;
        col_start [blk] = cols ;
        while ( blk < blks )
        {
            blk++ ;
            col_start [blk] = cols ;
        }

        if ( PrintLevel >= 1 )
        {
            printf ("new bounds activated: %ld "
                    "(free before: %ld after: %ld)\n",
                   (LONG) (ncol-cols), (LONG) ncol, (LONG) cols);
        }
        /* Update AT in one pass by exploiting colmap, the
           map from the columns of the original matrix to the
           to the columns in the compressed matrix
           (col = -1 if the column was deleted). */
        pp = 0 ;
        p = 0 ;
        for (i = 0; i < nrow; i++)
        {
            q = ATp [i+1] ;
            for (; p < q; p++)
            {
                k = colmap [ATi [p]] ;
                if ( k >= 0 ) /* keep the element */
                {
                    ATi [pp] = k ;
                    ATx [pp] = ATx [p] ;
                    pp++ ;
                }
            }
            ATp [i+1] = pp ;
        }
        pasa_copyi (Work->AFTp, ATp, nrow+1) ;

        /* check whether an inequality became active */
        for (k = 1; k <= ni; k++)
        {
            row = ineq_row [k] ;
            if ( ir [row] <= ni ) /* active inequality */
            {
                break ;
            }
        }
        if ( k <= ni ) /* an inequality became active */
        {
            /* Project the current iterate onto the feasible set.
               After this projection, the current factorization
               corresponds to the current iterate. */
            return (PASA_TRY_PROJECT2) ;
        }
        return (PASA_NEW_ACTIVE) ;
    }
}

/* ==========================================================================
   === pasa_compress_prob ===================================================
   ==========================================================================
    Check for new active bounds and inequalities, and compress the problem
    when they are found. This is only called from activeGP.
   ========================================================================== */
int pasa_compress_prob /* return:
                                  PASA_NEW_ACTIVE
                                  PASA_NO_CHANGE_IN_ACTIVE
                                  PASA_GRAD_PROJ */
(
    int      use_nc, /* use nc_temp to determine the active bounds */
    PASAcom    *Com  /* common variables */
)
{
    PASAINT nc, nf, *bound_cols, *ifree ;
    PASAFLOAT *lo, *hi, *g, *x ;
    const int loExists = Com->loExists ;
    const int hiExists = Com->hiExists ;
    lo = Com->lo ;
    hi = Com->hi ;
    g = Com->g ;
    x = Com->x ;
    ifree = Com->ifree ;
    bound_cols = Com->bound_cols ;
    nc = Com->nc ;

    const int PrintLevel = Com->pasaparm->PrintLevel ;
    if ( PrintLevel >= 1 )
    {
        printf ("compress, nc: %ld\n", (LONG) nc) ;
    }

    if ( Com->use_napheap )
    {
        int DelCol, AddRow ;
        PASAINT cols, i, j ;
        PASAFLOAT a2, s, *a ;
        a = Com->nap_a ;
        DelCol = FALSE ;
        AddRow = FALSE ;
        if ( abs (Com->nap_constraint) == 1 )
        {
            if ( Com->nap_bl != Com->nap_bu )
            {
                AddRow = TRUE ;
                Com->nr = 1 ;
                Com->bound_rows [0] = Com->nap_constraint ;
                if ( Com->nap_constraint > 0 )
                {
                    Com->nap_bl = Com->nap_bu ;
                }
                else
                {
                    Com->nap_bu = Com->nap_bl ;
                }
            }
        }
        nf = Com->nf ;
        for (cols = 0; cols < nf; cols++)
        {
            if ( (loExists && (x [cols] == lo [cols])) ||
                 (hiExists && (x [cols] == hi [cols])) )
            {
                break ;
            }
        }
        if ( cols < nf ) /* at least one bound constraint is activated */
        {
            a2 = s = PASAZERO ;
            DelCol = TRUE ;
            for (j = cols; j < nf; j++)
            {
                PASAFLOAT const xj = x [j] ;
                PASAFLOAT const aj = a [j] ;
                /* a bound xj is stored in userx and removed from the problem */
                if ( loExists && (xj == lo [j]) )
                {
                    i = ifree [j] ;
                    Com->userx [i] = xj ;
                    bound_cols [nc] = -(i + 1) ;
                    nc++ ;

                    /* update the bounds for the linear inequality */
                    s += aj*xj ;
                    a2 += aj*aj ;
                }
                else if ( hiExists && (xj == hi [j]) )
                {
                    i = ifree [j] ;
                    Com->userx [i] = xj ;
                    bound_cols [nc] = i ;
                    nc++ ;

                    /* update the bounds for the linear inequality */
                    s += aj*xj ;
                    a2 += aj*aj ;
                }
                else /* xj is free and retained */
                {
                    /* retain associated x, g, lo, hi, and ifree entries */
                    x [cols] = xj ;
                    g [cols] = g [j] ;
                    a [cols] = aj ;
                    if ( loExists ) lo [cols] = lo [j] ;
                    if ( hiExists ) hi [cols] = hi [j] ;
                    ifree [cols] = ifree [j] ;
                    cols++ ;
                }
            }
            nf = cols ;
            Com->nf = nf ;
            Com->nc = nc ;
            Com->nap_bl -= s ;
            Com->nap_bu -= s ;
            if ( PrintLevel >= 1 )
            {
                printf ("nf: %ld nc: %ld\n", (LONG) nf, (LONG) nc) ;
            }
        }
        if ( (AddRow == TRUE) || (DelCol == TRUE) )
        {
            if ( AddRow == TRUE )
            {
                Com->nap_a2save = Com->nap_a2 = pasa_dot (a, a, nf) ;
            }
            else if ( DelCol == TRUE )
            {
                Com->nap_a2 -= a2 ;
                /* recompute ||a||^2 when it drops by the given factor */
                if ( a2 <= .01*Com->nap_a2save )
                {
                    Com->nap_a2save = Com->nap_a2 = pasa_dot (a, a, nf) ;
                }
            }
        }

        if ( AddRow || DelCol )
        {
            return (PASA_NEW_ACTIVE) ;
        }
        else
        {
            return (PASA_NO_CHANGE_IN_ACTIVE) ;
        }
    }
    else if ( Com->use_pproj )
    {
        int blks, *ib, *sol_to_blk ;
        PASAINT blk, cols, iri, i, j, k, l, nr, ni, ncol, nrow, pp, p, q, row,
                *Ap, *Ai, *Anz, *ATp, *ATi, *bound_rows, *colmap,
               *col_start, *ir, *ineq_row ;
        PASAFLOAT t, *Ax, *ATx, *b, *bl, *bu ;
        PPprob *Prob ;
        PPwork *Work ;

        Prob = Com->ppcom->Prob ;
        Work = Com->ppcom->Work ;
        nrow = Com->nrow ;
        colmap = Work->arrayi ; /* borrow temporary space from pproj */
        ifree = Com->ifree ;
        ib = Work->ib ;
        ir = Work->ir ;
        ineq_row = Prob->ineq_row ;
        b  = Com->b ;
        bl = Com->bl ;
        bu = Com->bu ;
        Ap = Prob->Ap ;
        Ai = Prob->Ai ;
        Ax = Prob->Ax ;
        Anz = Prob->Anz ;
        ATp = Work->ATp ;
        ATi = Work->ATi ;
        ATx = Work->ATx ;
        sol_to_blk = Work->sol_to_blk ;
        blks = Work->blks ;
        bound_rows = Com->bound_rows ;
        /* Check to see if an inactive inequality constraint became active.
           If so, then make it an equality constraint and remove it from
           the list of inequality constraints. */
        ni = Prob->ni ;
        nr = Com->nr ;
        PASAINT newrows = 0 ;
        for (k = 1; k <= ni; k++)
        {
            row = ineq_row [k] ;
            if ( ir [row] <= ni ) /* active inequality */
            {
                /* shift bl and bu down if newrows > 0 */
                if ( newrows > 0 )
                {
                    for (; j < k; j++)
                    {
                        bl [j-newrows] = bl [j] ;
                        bu [j-newrows] = bu [j] ;
                        sol_to_blk [j-newrows] = sol_to_blk [j] ;
                    }
                }
                j = k + 1 ;
                if ( ir [row] > 0 )
                {
                    b [row] = bu [k] ;
                    bound_rows [nr] = row + 1 ;
                }
                else
                {
                    b [row] = bl [k] ;
                    bound_rows [nr] = -(row+1) ;
                }
                ir [row] = 0 ;
                nr++ ;
                newrows++ ;
            }
            else if ( newrows > 0 ) /* keep the inactive row */
            {
                ineq_row [k-newrows] = row ;
                ir [row] -= newrows ; /* adjust for number of deleted rows */
            }
        }

        if ( newrows > 0 )
        {
            if ( PrintLevel >= 1 )
            {
                printf ("%ld new inequalities activated\n", (LONG) newrows) ;
                if ( PrintLevel >= 3 )
                {
                    printf ("The active rows are:\n") ;
                    for (i = 0; i < nr; i++)
                    {
                        if ( bound_rows [i] > 0 )
                        {
                            printf ("%10ld upper\n", (LONG) bound_rows [i] - 1);
                        }
                        else
                        {
                            printf ("%10ld lower\n", (LONG) -(bound_rows[i]+1));
                        }
                    }
                }
            }
#ifndef NDEBUG
            int *ActiveRows = Com->ppcom->Check->ActiveRows ;
            for (i = 0; i < nr; i++)
            {
                row = (bound_rows [i] > 0) ?
                       bound_rows [i] - 1 : -(bound_rows [i] + 1) ;
                if ( ActiveRows [row] != 1 )
                {
                    printf ("row %ld is active in compress_prob but is missing "
                            "from the factorization\n", (LONG) row) ;
                    pasa_error (-1, __FILE__, __LINE__, "stop") ;
                }
            }
#endif
            /* the remaining inequalities are inactive, shift down */
            for (; j <= ni; j++)
            {
                bl [j-newrows] = bl [j] ;
                bu [j-newrows] = bu [j] ;
                sol_to_blk [j-newrows] = sol_to_blk [j] ;
            }
            sol_to_blk [j-newrows] = Work->blks ;
            Com->nr = nr ;
            ni -= newrows ;
            Prob->ni = ni ;
            ineq_row [ni+1] = nrow ;
            for (k = 1; k <= ni; k++)
            {
                ir [ineq_row [k]] -= newrows ;
            }
        }

#ifndef NDEBUG
        pasa_check_feas (Com) ;
#endif

        /* Check for new active bounds. If they are found,
           then delete from A the columns associated with bound variables
           and update the start of each column in the multilevel
           decomposition. */
        ncol = Prob->ncol ;
        for (cols = 0; cols < ncol; cols++)
        {
            if ( ib [cols] != 0 )
            {
                break ;
            }
        }
        if ( cols < ncol ) /* at least one constraint is activated */
        {
            /* cols will count the number of remaining columns in A */
            blks = Work->blks ;
            col_start = Work->col_start ;

            /* the initial columns are mapped to themselves */
            for (i = 0; i < cols; i++)
            {
                colmap [i] = i ;
            }
            /* When columns are deleted, the column start of each block is
               reduced by the number of proceeding columns that were deleted.
               Find the first block with column start > j. The following code
               terminates since col_start [blks] = number of columns
               in matrix. */
            for (blk = 1; blk <= blks; blk++)
            {
                if ( col_start [blk] > cols )
                {
                    break ;
                }
            }
            l = col_start [blk] ;

            /* start of storage for this first dropped column, k increments
               over the compressed matrix, while p increments over the
               original matrix */
#ifndef NDEBUG
            int *ActiveCols = Com->ppcom->Check->ActiveCols ;
#endif
            p = k = Ap [cols] ;
            for (j = cols; j < ncol; j++)
            {
                if ( j == l )
                {
                    col_start [blk] = cols ;
                    blk++ ;
                    while ( col_start [blk] == l )
                    {
                        col_start [blk] = cols ;
                        blk++ ;
                    }
                    l = col_start [blk] ;
                }
                q = Ap [j+1] ;
                if ( ib [j] == 0 ) /* retained column */
                {
#ifndef NDEBUG
                    if ( ActiveCols [j] == 0 )
                    {
                        printf ("column %ld is missing from factor but ib "
                                "vanishes\n", (LONG) j) ;
                        pasa_error (-1, __FILE__, __LINE__, "stop") ;
                    }
                    ActiveCols [j]    = 0 ;
                    ActiveCols [cols] = 1 ;
#endif
                    /* retain associated x, g, lo, hi, and ifree entries */
                    x [cols] = x [j] ;
                    g [cols] = g [j] ;
                    if ( loExists ) lo [cols] = lo [j] ;
                    if ( hiExists ) hi [cols] = hi [j] ;
                    ifree [cols] = ifree [j] ;
                    colmap [j] = cols ;
                    Anz [cols] = Anz [j] ;

                    /* save column j in matrix */
                    Ap [cols+1] = Ap [cols] + q - p ;
                    cols++ ;
                    for (; p < q; p++)
                    {
                        Ai [k] = Ai [p] ;
                        Ax [k] = Ax [p] ;
                        k++ ;
                    }
                }
                else /* bound column */
                {
#ifndef NDEBUG
                    if ( ActiveCols [j] == 1 )
                    {
                        printf ("column %ld appears in the factor but is "
                                "supposed to be absent\n", (LONG) j) ;
                        pasa_error (-1, __FILE__, __LINE__, "stop") ;
                    }
#endif
                    i = ifree [j] ;
                    colmap [j] = -1 ;
                    if ( ib [j] > 0 )
                    {
                        Com->userx [i] = hi [j] ;
                        bound_cols [nc] = i ;
                    }
                    else
                    {
                        Com->userx [i] = lo [j] ;
                        bound_cols [nc] = -(i + 1) ;
                    }

                    nc++ ;
                    /* The replacement column is free in the reduced matrix,
                       and the right side of constraints is adjusted */
                    ib [j] = 0 ;
                    PASAFLOAT const xj = x [j] ;
                    for (; p < q; p++)
                    {
                        i = Ai [p] ;
                        iri = ir [i] ;
                        t = Ax [p]*xj ;
                        if ( iri == 0 ) /* equality constraint */
                        {
                            b [i] -= t ;
                        }
                        else /* inactive inequality, above ir = 0 if active */
                        {
                            iri -= ni ;
                            bl [iri] -= t ;
                            bu [iri] -= t ;
                        }
                    }
                }
            }
            Prob->ncol = cols ;
            Work->A->ncol = cols ;
            Work->A->nzmax = Ap [cols] ;
            Com->nf = cols ;
            Com->nc = nc ;
            col_start [blk] = cols ;
            while ( blk < blks )
            {
                blk++ ;
                col_start [blk] = cols ;
            }

            if ( PrintLevel >= 1 )
            {
                printf ("new bounds activated: %ld "
                        "(free before: %ld after: %ld)\n",
                        (LONG) (ncol-cols), (LONG) ncol, (LONG) cols);
            }
            /* Update AT in one pass by exploiting colmap, the
               map from the columns of the original matrix to the
               to the columns in the compressed matrix
               (col = -1 if the column was deleted). */
            pp = 0 ;
            p = 0 ;
            for (i = 0; i < nrow; i++)
            {
                q = ATp [i+1] ;
                for (; p < q; p++)
                {
                    k = colmap [ATi [p]] ;
                    if ( k >= 0 ) /* keep the element */
                    {
                        ATi [pp] = k ;
                        ATx [pp] = ATx [p] ;
                        pp++ ;
                    }
                }
                ATp [i+1] = pp ;
            }
            pasa_copyi (Work->AFTp, ATp, nrow+1) ;
        }
#ifndef NDEBUG
        pasa_check_feas (Com) ;
#endif
        if ( (cols < ncol) || (newrows > 0) )
        {
            return (PASA_NEW_ACTIVE) ;
        }
        else
        {
            return (PASA_NO_CHANGE_IN_ACTIVE) ;
        }
    }
    /* At the top of activeGP, or at the end with step < 1. In either case,
       we cannot use the bounds array to deduce the potential bound variables */
    else if ( !use_nc )
    {
        PASAINT j, ncol ;
        ncol = Com->ncol ;
        for (j = 0; j < ncol; j++)
        {
            /* variable at lower bound */
            if ( loExists && (x [j] == lo [j])
                          && (g [j] >= PASAZERO) )
            {
                Com->userx [ifree [j]] = lo [j] ;
                bound_cols [nc] = -(j+1) ;
                nc++ ;
            }
            /* variable at upper bound */
            else if ( hiExists && (x [j] == hi [j])
                               && (g [j] <= PASAZERO) )
            {
                Com->userx [ifree [j]] = hi [j] ;
                bound_cols [nc] = j ;
                nc++ ;
            }
            else
            {
                if ( nc > 0 )
                {
                    ifree [j-nc] = ifree [j] ;
                    x [j-nc] = x [j] ;
                    g [j-nc] = g [j] ;
                    if ( loExists ) lo [j-nc] = lo [j] ;
                    if ( hiExists ) hi [j-nc] = hi [j] ;
                }
            }
        }
        if ( PrintLevel >= 1 )
        {
            printf ("compress at top of active GP, final nc: %ld\n",
                    (LONG) nc);
        }
        if ( nc > 0 )
        {
            Com->nf -= nc ;
            Com->nc = nc ;
            return (PASA_NEW_ACTIVE) ;
        }
        else
        {
            return (PASA_NO_CHANGE_IN_ACTIVE) ;
        }
    }
    else /* only bound constraints, bottom part of active GP, use nc */
    {
        PASAINT i, i0, i1, j, k, l, nc_temp ;
        nc_temp = Com->nc_temp ;
        if ( nc == nc_temp )
        {
            return (PASA_NO_CHANGE_IN_ACTIVE) ;
        }

        int const search = ((Com->nf == Com->ncol) && (Com->fixed == FALSE)) ?
                           FALSE : TRUE ;
        /* i0 = first available location
            i = element to delete
           i1 = first element to move */
        i0 = -1 ; /* indicates the start of search */
        i = 0 ;   /* look for first delete element starting at i = 0 */
        i1 = 1 ;
        if ( PrintLevel >= 1 )
        {
            printf ("compress in bottom half of active GP\n") ;
            printf ("search: %i, nc_temp: %ld\n", search, (LONG) nc_temp) ;
        }
        for (; nc <= nc_temp; nc++)
        {
            if ( nc < nc_temp )
            {
                /* variable to compress in user coordinates */
                k = j = bound_cols [nc] ;
                if ( j < 0 )
                {
                    j = -(j+1) ;
                }

                /* find the pasa index corresponding to j */
                if ( !search ) /* ifree [j] = j for each j */
                {
                    i = j ;
                    ASSERT (ifree [j] == j) ;
                    if ( i0 >= 0 )
                    {
                        pasa_copyi (ifree+i0, ifree+i1, i-i1) ;
                    }
                }
                else if ( i0 < 0 )     /* search for first j index */
                {
                    while ( ifree [i] != j )
                    {
                        i++ ;
                    }
                }
                else                   /* search for subsequent j indices */
                {
                    l = i0 - i ;
                    while ( ifree [i] != j )
                    {
                        ifree [i+l] = ifree [i] ; /* move ifree down */
                        i++ ;
                    }
                }

                /* store the bound value in userx */
                if ( k < 0 )
                {
                    Com->userx [j] = lo [i] ;
                    if ( PrintLevel >= 2 )
                    {
                        printf ("lower bind %ld (userx) using %ld (pasa)\n",
                                 (LONG) j, (LONG) i) ;
                    }
                }
                else
                {
                    Com->userx [j] = hi [i] ;
                    if ( PrintLevel >= 2 )
                    {
                        printf ("upper bind %ld (userx) using %ld (pasa)\n",
                                 (LONG) j, (LONG) i) ;
                    }
                }
                /* determine number k of elements to move */
                k = i - i1 ;
            }
            else
            {
                k = Com->nf - i1 ;
                pasa_copyi (ifree+i0, ifree+i1, k) ;
            }
            if ( i0 < 0 ) /* first pass, do not copy */
            {
                i0 = i ;
                i++ ;
                i1 = i ;
            }
            else /* move stuff from locations i1 to i - 1
                            or from locations i1 to nf - 1 */
            {
                if ( k > 0 )
                {
                    if ( PrintLevel >= 2 )
                    {
                        printf ("do copy i0: %ld i1: %ld k: %ld\n",
                                 (LONG) i0, (LONG) i1, (LONG) k) ;
                    }
                    pasa_copyx_noblas (x+i0, x+i1, k) ;
                    pasa_copyx_noblas (g+i0, g+i1, k) ;
                    if ( loExists )
                    {
                        pasa_copyx_noblas (lo+i0, lo+i1, k) ;
                    }
                    if ( hiExists )
                    {
                        pasa_copyx_noblas (hi+i0, hi+i1, k) ;
                    }
                    i0 += k ;
                }
                i++ ;
                i1 = i ;
            }
        }
        Com->nf -= nc_temp - Com->nc ;
        Com->nc = nc_temp ;
        return (PASA_NEW_ACTIVE) ;
    }
#ifndef NDEBUG
    pasa_check_feas (Com) ;
#endif
}

/* =========================================================================
   ==== pasa_check_error ===================================================
   =========================================================================
   This routine computes the global error, and if needed, the local error.
   If check_error is called from grad_proj, then it will determine if the
   error tolerance is satisfied, or we should branch to the active set
   routines. If it is called from the active set routines, then it will
   determine if the error tolerance is satisfied, or we should branch to
   grad_proj.

   Recall that the optimization problem is

   min f(x) subject to lo <= x <= hi, bl <= Ax <= bu

   We reformulate this as

   (P) min f(x) subject to lo <= x <= hi, bl <= b <= bu, Ax - b = 0.

   Let L(x, y) = f(x) + y'(Ax - b) be the Lagrangian.
   The first-order optimality conditions are that

  (1)  bl <= Ax <= bu,  lo <= x <= hi,
  (2)  L'_j (x, y)  = 0 if lo_j < x_j < hi_j,
  (3)  L'_j (x, y) <= 0 if        x_j = hi_j,
  (4)  L'_j (x, y) >= 0 if lo_j = x_j,
  (5)      y_i      = 0 if bl_i < (Ax)_i < bu_i,
  (6)      y_i     <= 0 if bl_i = (Ax)_i
  (7)      y_i     >= 0 if        (Ax)_i = bu_i

   A measure of the feasibility error (1) is

       feas = max { max {0, (Ax)_i - bu_i} i = 1, 2, ... m,
                    max {0, bl_i - (Ax)_i} i = 1, 2, ... m,
                    max {0, x_j - hi_j   } j = 1, 2, ... n,
                    max {0, lo_j - x_j   } j = 1, 2, ... n.

   A measure of the violation of (2)-(7), assuming x is feasible, is given by

   E = maxsignerr = max { min { y_i, |(Ax)_i - bu_i|} for y_i >= 0,
                          min {-y_i, |(Ax)_i - bl_i|} for y_i <= 0,
                          min { L'_j (x, y), |x_j - lo_j|} for L'_j (x, y) >= 0,
                          min {-L'_j (x, y), |x_j - hi_j|} for L'_j (x, y) <= 0}

   If x is feasible and E is 0, then the KKT conditions are satisfied.
   
   In the active set approach, the iterates are assumed to be feasible.
   Also, in the null space projection, we estimated the local error e by
   projecting the gradient onto the manifold associated with the active
   constraints. The optimization problem solved in null_project is

   (L0)   min 0.5 |g_F - z_F|^2 s. t. A_{BF} z_F = 0,

   where g = nabla f(x), F denotes the free indices for which
   lo_j < x_j < hi_j, and B is the indices of the binding inequality
   constraints: (A x)_i = bl_i or bu_i.  The local error e is |z_F|
   where z_F is the unique solution of (L0).  Let y_B denote a least
   squares KKT multiplier for the constraint in (L).  The equation
   satisfied by y_B is

   (L1)        A_{BF}'y_B = g_F - z_F.

   An estimate for the least square multiplier is obtained in PPROJ by solving

   (L2)        (A_{BF} A_{BF}' + sigma I) y_B = A_{BF} g_F ,

   where sigma is a small positive number.  The components of y corresponding
   to indicies in B complement are set to 0.  It is this multiplier,
   whose nonzero components are computed in null_project using (L2),
   that is used to estimate the error in the check_error routine. By (L1)
   we have

                z_F = g_F - A_{BF}'y_B.

   Since the components of y corresponding to indicies in B complement are
   set to 0, it follows that z_j = L'_j (x, y) for j in F. Since the local
   error e = |z_F| and since |L'_j (x, y)| is embedded in the global
   error estimator E, the inequality E >= e often holds.

   Based on E and e, check_error returns an integer flag which triggers
   one of the following:

   1. If E <= grad_tol, then terminate, the specified
      error tolerance was achieved.
   2. Switch from the active set routines to grad_proj
      when e < switchfactor * E.
   3. We switch from grad_proj to the active set GP when
      e >= switchfactor * E.
   ========================================================================= */
int pasa_check_error
(
    PASAcom *Com  /* pasa com structure */
)
{
    PASAINT i, j, k, l, m, p, *RLinkUp ;
    PASAFLOAT t, u, *lambda, *userg ;
    PPprob *Prob ;
    PPwork *W ;
    PASAparm  const       *Parm = Com->pasaparm ; /* parameters */
    int       const  PrintLevel = Parm->PrintLevel ;
    int       const   use_pproj = Com->use_pproj ;
    int       const use_napheap = Com->use_napheap ;
    int       const    location = Com->location ;
    int       const     Aexists = Com->Aexists ;
    int       const      Bounds = Com->Bounds ;
    int       const    loExists = Com->loExists ;
    int       const    hiExists = Com->hiExists ;
    PASAINT   const        ncol = Com->ncol ;
    PASAINT   const        nrow = Com->nrow ;
    PASAFLOAT const          *x = Com->x ;
    PASAFLOAT const          *g = Com->g ;
    PASAFLOAT const         *lo = Com->lo ;
    PASAFLOAT const         *hi = Com->hi ;
    PPcom                *ppcom = Com->ppcom ;
    PASAFLOAT              gmax = Com->gmax ;
    PASAFLOAT                 E = PASAZERO ;    /* global error */
    PASAFLOAT                 e = PASAZERO ;    /* local error */

    /* get gmax if iterates are printed or error is scaled by gmax */
    int const get_gmax = (PrintLevel || Parm->scale_error_by_gmax ) ;
    if ( PrintLevel ) printf ("check_error\n") ;
    if ( Com->QP )
    {
        userg = Com->gtot ;
    }
    else
    {
        userg = Com->userg ;
    }

#ifndef NOPPROJ
    if ( use_pproj )
    {
        Prob    = ppcom->Prob ;
        W       = ppcom->Work ;
        lambda  = Com->lambda ; /* dual multiplier (y above in comments) */
        RLinkUp = W->RLinkUp ;
    }
#endif

    /* If there are no linear constraints, then NoRows = TRUE */
    int NoRows = !Aexists ;

    /* If all the linear constraints are inactive, then NoRows
       is set to TRUE even though A exists in the problem. */
    if ( use_napheap )
    {
        if ( (Com->nap_constraint == 0) || (Com->nap_a2 == PASAZERO) )
        {
            NoRows = TRUE ;
        }
    }
#ifndef NOPPROJ
    else if ( use_pproj && (RLinkUp [nrow] == nrow) )
    {
        NoRows = TRUE ;
    }
#endif
    if ( PrintLevel > 1 )
    {
        printf ("NoRows: ") ;
        sopt_print_TF (NoRows) ;
    }

    /* local error estimated previously in null_project */
    e = Com->e ;
    if ( location == PASA_GRAD_PROJ )
    {
        /* NOTE: error estimation for pure bound constrained problems handled
                 in gradproj_step, so ignore pure bound constraint case here */

        if ( NoRows && !Bounds ) /* the problem is unconstrained */
        {
            ; /* do nothing, e = E was already computed in null_project */
        }
        else if ( (NoRows && Aexists) || use_napheap )
        {
            /* a_exists is TRUE if the linear constraint a'x = b exists and
               it is active */
            int const a_exists = (!NoRows && use_napheap) ? TRUE : FALSE ;
            PASAFLOAT const *a = (a_exists) ? Com->nap_a : NULL ;
            PASAFLOAT aa, ag ;
            if ( a_exists )
            {
                /* estimate the multiplier */
                if ( !Bounds ) /* no bound constraints */
                {
                    aa = Com->nap_a2full ;
                    ag = pasa_dot (a, g, ncol) ;
                }
                else if ( !loExists || !hiExists ) /* either lo or hi missing */
                {
                    aa = PASAZERO ;
                    ag = PASAZERO ;
                    PASAFLOAT const *lower = (loExists) ? lo : x ;
                    PASAFLOAT const *upper = (hiExists) ? hi : x ;
                    for (j = 0; j < ncol; j++)
                    {
                        /* bound constaint inactive if upper [j] > lower [j] */
                        if ( upper [j] > lower [j] )
                        {
                            aa += a [j]*a [j] ;
                            ag += a [j]*g [j] ;
                        }
                    }
                }
                else /* both lo and hi exist */
                {
                    aa = PASAZERO ;
                    ag = PASAZERO ;
                    for (j = 0; j < ncol; j++)
                    {
                        if ( (x [j] > lo [j]) && (x [j] < hi [j]) )
                        {
                            aa += a [j]*a [j] ;
                            ag += a [j]*g [j] ;
                        }
                    }
                }
            }
            int const a_exists_pos = (a_exists && (aa > PASAZERO)) ? TRUE:FALSE;
            PASAFLOAT const naplambda = (a_exists_pos) ?  ag/aa : PASAZERO ;
            for (j = 0; j < ncol; j++)
            {
                t = (a_exists_pos) ? g [j] - naplambda*a [j] : g [j] ;
                if ( t > E ) /* also implies that t > 0 since E >= 0 */
                {
                    if ( loExists )
                    {
                        if (((u = fabs (x [j] - lo [j])) < t) && (u > E)) E = u;
                        else E = t ;
                    }
                    else     E = t ;
                }
                else if ( (t = -t) > E ) /* t < 0 */
                {
                    if ( hiExists )
                    {
                        if (((u = fabs (x [j] - hi [j])) < t) && (u > E)) E = u;
                        else E = t ;
                    }
                    else     E = t ;
                }
            }
            if ( a_exists_pos )
            {
                /* since the linear constraint is active, include the multiplier
                   error in the global error */
                if ( Com->nap_constraint < 0 ) /* at lower bound */
                {
                    if ( E < -naplambda ) E = -naplambda ;
                    if ( (PrintLevel >= 2) && (naplambda < PASAZERO) )
                    {
                        printf ("lower ineqn %i wrong sign: %e\n",
                                                                0, -naplambda) ;
                    }
                }
                else if ( Com->nap_constraint == 1 ) /* at upper bound */
                {
                    if ( E < naplambda ) E = naplambda ;
                    if ( (PrintLevel >= 2) && (naplambda > PASAZERO) )
                    {
                        printf ("upper ineqn %i wrong sign: %e\n",
                                                                 0, naplambda) ;
                    }
                }
            }
        }
#ifndef NOPPROJ
        else /* use_pproj, linear constraints present */
        {
            PASAINT   const      *Ap = Prob->Ap ;
            PASAINT   const      *Ai = Prob->Ai ;
            PASAINT   const     *Anz = Prob->Anz ;
            PASAFLOAT const      *Ax = Prob->Ax ;
            int       const      *ib = W->ib ;
            PASAINT   const      *ir = W->ir ;

            for (i = RLinkUp [nrow]; i < nrow; i = RLinkUp [i])
            {
                /* first put sign errors in lambda into the global error */
                PASAINT const iri = ir [i] ;
                if ( iri < 0 ) /* constraint at lower bound, need lambda >= 0 */
                {
                    if ( E < (t = -lambda [i]) )
                    {
                        E = t ;
                        if ( PrintLevel >= 2 )
                        {
                            printf ("i: %ld wrong lower: %e\n", (LONG) i, E) ;
                        }
                    }
                }
                else if ( (iri > 0) && (E < (t = lambda [i])) )
                {
                    /* constraint at upper bound, need lambda <= 0 */
                    ASSERT ( ir [i] < Com->ni ) ;
                    E = t ;
                    if ( PrintLevel >= 2 )
                    {
                        printf ("i: %ld wrong upper: %e\n", (LONG) i, E) ;
                    }
                }
            }
            for (j = 0; j < ncol; j++)
            {
                int const ibj = ib [j] ;
                if ( ibj ) /* variable is at bound */
                {
                    /* compute g_j - column j of A dot lambda */
                    t = PASAZERO ;
                    k = Ap [j] ;
                    PASAINT const L = k + Anz [j] ;
                    for (; k < L; k++)
                    {
                        t -= Ax [k]*lambda [Ai [k]] ;
                    }
                    t += g [j] ;
                    if ( ibj < 0 ) /* variable at lower bound, need t >= 0 */
                    {
                        if ( -t > E )
                        {
                            E = -t ;
                            if ( PrintLevel >= 2 )
                            {
                                printf ("j: %ld wrong lower: %e\n",
                                       (LONG) j, E) ;
                            }
                        }
                    }
                    else if ( t > E ) /* ibj > 0, variable at upper bound */
                    {
                        /* variable at upper bound, need t <= 0 */
                        E = t ;
                        if ( PrintLevel >= 2 )
                        {
                            printf ("j: %ld wrong upper: %e\n", (LONG) j, t) ;
                        }
                    }
                }
            }
        }
    }
#endif
    else /* location is PASA_ACTIVE_GP or CG_DESCENT */
    {
        PASAINT const nc = Com->nc ;
        PASAINT const *bound_cols = Com->bound_cols ;
        if ( NoRows && !Bounds ) /* the variables are unconstrained */
        {
            ; /* do nothing */
        }
        else if ( NoRows ) /* pure bound constraints or napheap or pproj */
        {
            for (k = 0; k < nc; k++)
            {
                j = bound_cols [k] ;
                if ( j < 0 )     /* at lower bound */
                {
                    j = -(j+1) ; /* user coordinates */
                    PASAFLOAT const gj = userg [j] ;
                    if ( E < -gj ) E = -gj ;
                    if ( get_gmax )
                    {
                        if ( gmax < fabs (gj) ) gmax = fabs(gj) ;
                    }
                }
                else             /* at upper bound */
                {
                    PASAFLOAT const gj = userg [j] ;
                    if ( E < gj ) E = gj ;
                    if ( get_gmax )
                    {
                        if ( gmax < fabs (gj) ) gmax = fabs(gj) ;
                    }
                }
            }
            if ( get_gmax ) Com->gmax = gmax ;
        }
        else if ( use_napheap ) /* knapsack constraint with active equality */
        {
            /* naplambda computed previously in null_project */
            PASAFLOAT const naplambda = Com->nap_lambda ;
            PASAFLOAT const *usera = Com->nap_auser ;
            for (k = 0; k < nc; k++)
            {
                j = bound_cols [k] ;
                if ( j < 0 )      /* at lower bound */
                {
                    j = -(j+1) ;  /* user coordinates */
                    PASAFLOAT const T = userg [j] - naplambda*usera [j] ;
                    if ( E < -T ) E = -T ;
                }
                else              /* at upper bound */
                {
                    PASAFLOAT const T = userg [j] - naplambda*usera [j] ;
                    if ( E < T ) E = T ;
                }
                if ( get_gmax )
                {
                    if ( gmax < fabs (userg [j]) ) gmax = fabs (userg [j]) ;
                }
            }
            if ( get_gmax ) Com->gmax = gmax ;

            /* since the linear constraint is active, include the error
               in the multiplier when there are active inequalities */
            if ( Com->nr )
            {
                ASSERT (Com->nr == 1) ;
                if ( Com->bound_rows [0] < 0 ) /* at lower bound */
                {
                    if ( E < -naplambda ) E = -naplambda ;
                    if ( (PrintLevel >= 2) && (naplambda < PASAZERO) )
                    {
                        printf ("lower ineqn 0 wrong sign: %e\n", -naplambda) ;
                    }
                }
                else                           /* at upper bound */
                {
                    if ( E < naplambda ) E = naplambda ;
                    if ( (PrintLevel >= 2) && (naplambda > PASAZERO) )
                    {
                        printf ("upper ineqn 0 wrong sign: %e\n", naplambda) ;
                    }
                }
            }
        }
#ifndef NOPPROJ
        else /* use pproj, location = PASA_ACTIVE_GP or CG_DESCENT */
        {
            /* Com->lambda computed previously in null_project */
            PASAcopy               *Copy = Com->Copy ;
            PASAINT   const         *ApC = Copy->Ap ;
            PASAINT   const         *AiC = Copy->Ai ;
            PASAFLOAT const         *AxC = Copy->Ax ;
            PASAINT   const     *invperm = Com->invperm ;
            PASAFLOAT const lambda_scale = PASAONE/Com->lambda_scale ;
            for (k = 0; k < nc; k++)
            {
                j = bound_cols [k] ;
                if ( j < 0 )         /* at lower bound */
                {
                    m = -(j+1) ;      /* user col */
                    l = invperm [m] ; /* convert from user col to pproj col */
                }
                else                  /* at upper bound */
                {
                    m = j ;
                    l = invperm [m] ;
                }
                t = PASAZERO ;
                PASAINT const q = ApC [l+1] ;
                for (p = ApC [l]; p < q; p++)
                {
                    t -= lambda [AiC [p]]*AxC [p] ;
                }
                t = lambda_scale*t + userg [m] ;

                if ( j < 0 ) /* variable at lower bound */
                {
                    if ( E < -t ) E = -t ;
                }
                else
                {
                    if ( E  < t ) E = t ;
                }
                if ( get_gmax )
                {
                    if ( gmax < fabs (userg [m]) ) gmax = fabs (userg [m]) ;
                }
                if ( PrintLevel >= 2 )
                {
                    if ( j < 0 )
                    {
                        if ( t < PASAZERO )
                        {
                            printf("lower bound %ld (pproj index) wrong sign:"
                                   " %e\n", (LONG) l, -t) ;
                        }
                    }
                    else
                    {
                        if ( t > PASAZERO )
                        {
                            printf("upper bound %ld (pproj index) wrong sign: "
                                   "%e\n", (LONG) l, t) ;
                        }
                    }
                }
            }
            if ( get_gmax ) Com->gmax = gmax ;
            /* include sign errors for lambda in the global error */
            PASAINT const          nr = Com->nr ;
            PASAINT const *bound_rows = Com->bound_rows ;
            for (k = 0; k < nr; k++)
            {
                j = bound_rows [k] ;
                if ( j < 0 ) /* at lower bound */
                {
                    m = -(j+1) ;      /* row index */ 
                    if ( E < -lambda [m] ) E = -lambda [m] ;
                }
                else
                {
                    m = j - 1 ;
                    if ( E < lambda [m] ) E = lambda [m] ;
                }
/* printf ("k: %i E: %e\n", k, E) ;*/
                if ( PrintLevel >= 2 )
                {
                    t = lambda [m] ;
                    if ( j < 0 )
                    {
                        if ( t < PASAZERO )
                        {
                            printf ("lower ineqn %ld wrong sign: %e\n",
                                   (LONG) m, -t) ;
                        }
                    }
                    else
                    {
                        if ( t > PASAZERO )
                        {
                            printf ("upper ineqn %ld wrong sign: %e\n",
                                   (LONG) m, t) ;
                        }
                    }
                }
            }
        }
#endif
    }
    if ( Aexists || Bounds )
    {
        Com->E1 = PASAMAX (e, E) ;
        /* we have two different ways to measure the global error:

           1. violation of the KKT conditions (E1)
           2. ||proj(x-g) - x||               (Egp)
       
           The smaller of these is taken as the estimate of the global error */
        E = PASAMIN (Com->E1, Com->Egp) ;
        if ( Parm->scale_error_by_gmax )
        {
            if ( Com->gmax > PASAZERO ) E /= Com->gmax ;
            else                        E = PASAZERO ;
        }
    }
    else
    {
        E = Com->e ;
    }
    if ( E <= Com->grad_tol )
    {
        Com->E = E ;
        if ( PrintLevel )
        {
            if ( Aexists || Bounds )
            {
                printf ("error tol OK, E = %e e = %e KKT = %e Egp = %e"
                        " gmax = %e\n", E, e, Com->E1, Com->Egp, Com->gmax) ;
            }
            else
            {
                printf ("error tol OK, E = %e\n", E) ;
            }
        }
        return (PASA_ERROR_TOLERANCE_SATISFIED) ;
    }
    Com->e = e ;
    /* t = Parm->switchfactor * Com->E1 ;*/
    t = Com->switchfactor * E ;
    Com->testtol = PASAMAX (t, Com->grad_tol) ;
    if ( PrintLevel )
    {
        printf ("KKT = %e Egp: %e e = %e testtol= %e gmax = %e (check_error)\n",
                 Com->E1, Com->Egp, Com->e, Com->testtol, Com->gmax) ;
    }
    if ( (location == PASA_GRAD_PROJ) && (e >= Com->testtol) )
    {
        return (PASA_ACTIVE_GP) ;
    }
    if ( (location == PASA_ACTIVE_GP) || (location == PASA_CG_DESCENT) )
    {
        if ( e < Com->testtol ) return (PASA_GRAD_PROJ) ;
    }
    /* continue the same routine (grad_proj, activeGP, cg_descent) */
    return (PASA_OK) ;
}

/* =========================================================================
   ==== pasa_restoration ===================================================
   =========================================================================
    Compute the minimum norm perturbation dx such that A(x + dx) = b where A
    is the active constraint matrix and b is the associated right side
    vector.  It assumed that pproj has already been run and a factorization
    for (A*A' + sigma * I) already exists, where sigma is relatively small.
    The minimum norm solution has the form dx = A'*dy.  Basically, we solve
    the equation A*A'*dy = b - A*x, but with A*A' replaced by
    (A*A' + sigma * I), and then add A'*dy to x. The new x could violate
    either the bound constraints or the linear inequality constraints.
    If one of these constraints is violated, then the perturbation dx
    is multiplied by the largest stepsize s <= 1 such that x + s*dx
    satisfies the constraints.
   ========================================================================= */
int pasa_restoration
(
    PASAcom *pasacom, /* pasa com structure */
    CGcom     *cgcom  /* cg com structure */
)
{
    PASAINT i, j, p, maxbndindex, maxconindex ;
    PPFLOAT t, u, maxbndstep, maxconstep, maxstep, *dx, *xold ;
    int       const   use_pproj = pasacom->use_pproj ;
    int       const    loExists = pasacom->loExists ;
    int       const    hiExists = pasacom->hiExists ;
    int       const      Bounds = pasacom->Bounds ;
    int       const  PrintLevel = pasacom->cgparm->PrintLevel ;
    PASAINT   const        nrow = pasacom->nrow ;
    PASAINT   const          nf = pasacom->nf ;
    CGFLOAT                  *x = cgcom->x ;
    PASAFLOAT const         *lo = pasacom->lo ;
    PASAFLOAT const         *hi = pasacom->hi ;
    maxstep = maxconstep = maxbndstep = PASAONE ;
    maxconindex = maxbndindex = 0 ;
    if ( use_pproj )
    {
        PPwork              *W = pasacom->ppcom->Work ;
        PPFLOAT             *mem = W->arrayd ; /* some space */
        PPFLOAT   const     *ATx = W->ATx ;
        PPINT     const     *ATi = W->ATi ;
        PPINT     const     *ATp = W->ATp ;
        PPINT           *RLinkUp = W->RLinkUp ;
        PPINT           *RLinkDn = W->RLinkDn ;
        PPprob    const    *Prob = pasacom->ppcom->Prob ;
        PPINT     const      *Ap = Prob->Ap ;
        PPINT     const      *Ai = Prob->Ai ;
        PPINT     const     *Anz = Prob->Anz ;
        PPFLOAT   const      *Ax = Prob->Ax ;
        PASAFLOAT const       *b = pasacom->b ;
        PASAFLOAT const      *bl = pasacom->bl ;
        PASAFLOAT const      *bu = pasacom->bu ;

        /* memory allocation from pproj */
        PPFLOAT    *r = mem ;  mem += nrow ;
                   dx = mem ;  mem += nf ;
                 xold = mem ;  mem += nf ;

        /* for a QP, we need to save a copy of x */
        if ( pasacom->QP ) pasa_copyx (xold, x, nf) ;

        pasa_initx (r, PASAZERO, nrow) ;
        for (i = RLinkUp [nrow]; i < nrow; i = RLinkUp [i])
        {
            t = PPZERO ;
            p = ATp [i] ;
            PPINT const q = ATp [i+1] ;
            for (; p < q; p++)
            {
                t -= ATx [p]*x [ATi [p]] ;
            }
            r [i] = t + b [i] ; /* = b - A*x */
        }
        if ( PrintLevel )
        {
            printf ("violation of active constraints: %e\n",
                     pasa_sup_normx (r, nrow)) ;
        }

        /* solve (A*A' + sigma I)*dy = r, store the result in r */
        pproj_lsol (W->L, r, RLinkUp [nrow], nrow, RLinkUp) ;
        j = RLinkUp [nrow] ;
        /* momentarily set the initial RLinkDn to -1, this simplifies
           indexing in dltsolve */
        RLinkDn [j] = -1 ;
        i = RLinkDn [nrow] ;
        pproj_dltsol (W->L, r, r, i, j, RLinkDn) ; /* r stores dy */
        RLinkDn [j] = nrow ; /* restore RLinkDn */

        /* Compute dx = A'*dy and then determine the minimum stepsize
           (maxbndstep) along dx with the property that x + s*dx satisfies
           the bound constraints.  Since the columns of A have been
           compressed to remove the bound columns, we index over all
           the columns of A. */
        for (j = 0; j < nf; j++)
        {
            t = PPZERO ;
            p = Ap [j] ;
            PPINT const q = p + Anz [j] ;
            for (; p < q; p++)
            {
                t += Ax [p]*r [Ai [p]] ;
            }
            dx [j] = t ;
            if ( Bounds )
            {
                if ( loExists && (t < PASAZERO) )
                {
                    PASAFLOAT const s = (lo [j] - x [j])/t ;
                    if ( s < maxbndstep )
                    {
                        maxbndstep = s ;
                        maxbndindex = -(j+1) ;
                    }
                }
                else if ( hiExists && (t > PASAZERO) ) /* check upper bound */
                {
                    PASAFLOAT const s = (hi [j] - x [j])/t ;
                    if ( s < maxbndstep )
                    {
                        maxbndstep = s ;
                        maxbndindex = j + 1 ;
                    }
                }
            }
        }
        if ( maxbndstep < PASAZERO ) maxbndstep = PASAZERO ;
        /* check if an inactive inequality became active */
        PPINT const ni = Prob->ni ;
        if ( ni > 0 )
        {
            PPINT const *ineq_row = Prob->ineq_row ;
            for (i = 1; i <= ni; i++)
            {
                PPINT row = ineq_row [i] ;
                t = u = CGZERO ;
                PPINT const q = ATp [row+1] ;
                for (p = ATp [row]; p < q; p++)
                {
                    PPFLOAT const atx = ATx [p] ;
                    PPINT   const ati = ATi [p] ;
                    t += atx* x [ati] ;
                    u += atx*dx [ati] ;
                }
                if ( u < PASAZERO )
                {
                    t = (bl [i] - t)/u ;
                    if ( t < maxconstep )
                    {
                        maxconstep = t ;
                        maxconindex = -i ;
                    }
                }
                else if ( u > PASAZERO )
                {
                    t = (bu [i] - t)/u ;
                    if ( t < maxconstep )
                    {
                        maxconstep = t ;
                        maxconindex = i ;
                    }
                }
            }
            if ( maxconstep < PASAZERO ) maxconstep = PASAZERO ;
        }
        maxstep = PASAMIN (maxconstep, maxbndstep) ;
        pasa_daxpy (x, dx, maxstep, nf) ;
        if ( PrintLevel )
        {
            u = PPZERO ;
            for (i = RLinkUp [nrow]; i < nrow; i = RLinkUp [i])
            {
                t = PPZERO ;
                p = ATp [i] ;
                PPINT const q = ATp [i+1] ;
                for (; p < q; p++)
                {
                    t -= ATx [p]*x [ATi [p]] ;
                }
                t += b [i] ;
                if ( fabs (t) > u ) u = fabs (t) ;
            }
            if ( PrintLevel )
            {
                printf ("violation ... after restoration: %e\n", u) ;
            }
        }
    }
    else /* use_napheap */
    {
        /* there is nothing to check if the linear constraint is inactive */
        if ( pasacom->nap_constraint == 0 ) return (CG_OK) ;

        PASAFLOAT rhs ;
        rhs = pasacom->nap_bl ;
        PASAFLOAT const *a = pasacom->nap_a ;
        /* compute A*x */
        t = pasa_dot (a, x, nf) ;
        rhs -= t ;             /* rhs - a'*x */
        rhs /= pasacom->nap_a2 ; /* (rhs - a'*x)/a'*a */
        if ( pasacom->Bounds )
        {
            for (j = 0; j < nf; j++)
            {
                t = rhs*a [j] ;
                if ( loExists && (t < PASAZERO) )
                {
                    PASAFLOAT const s = (lo [j] - x [j])/t ;
                    if ( s < maxbndstep )
                    {
                        maxbndstep = s ;
                        maxbndindex = -(j+1) ;
                    }
                }
                else if ( hiExists && (t > PASAZERO) ) /* check upper bound */
                {
                    PASAFLOAT const s = (hi [j] - x [j])/t ;
                    if ( s < maxbndstep )
                    {
                        maxbndstep = s ;
                        maxbndindex = j + 1 ;
                    }
                }
            }
            if ( maxbndstep < PASAZERO ) maxbndstep = PASAZERO ;
        }
        if ( pasacom->QP )
        {
            xold = pasacom->nap_xold ;
            pasa_copyx (xold, x, nf) ;
            dx = pasacom->nap_dx ;
            pasa_scale (dx, a, maxbndstep*rhs, nf) ;
            pasa_daxpy (x, dx, PASAONE, nf) ;
        }
        else
        {
            pasa_daxpy (x, a, maxbndstep*rhs, nf) ;
        }
        maxstep = maxbndstep ;
    }
    pasacom->maxbndstep = maxbndstep ;
    pasacom->maxconstep = maxconstep ;
    pasacom->maxbndindex = maxbndindex ;
    pasacom->maxconindex = maxconindex ;
    pasacom->maxstep = maxstep ;

    /* update objective value and gradient at x */
    if ( PrintLevel )
    {
        printf ("obj. value   before restoration: %21.15e\n", pasacom->f) ;
    }
    if ( pasacom->QP )
    {
        pasa_evaluate_x (dx, xold, pasacom->g, pasacom->f, nf, pasacom->order,
                         pasacom) ;
    }
    else /* not a QP */
    {
        pasa_evaluate_x (x, NULL, NULL, PASAZERO, nf, pasacom->order, pasacom) ;
    }
    if ( PrintLevel )
    {
        printf ("...           after restoration: %21.15e\n", pasacom->f) ;
    }
    if ( maxstep < PASAONE ) return (CG_HITS_BOUNDARY) ;
    return (CG_OK) ;
}
#if 0
/* =========================================================================
   ==== pasaAG_checktol ====================================================
   =========================================================================
   This routine is only called from the pasa_activeGP routine.
   It is entered when the nominal stopping criterion
   Com->e <= Com->testtol (=switchfactor*Com->E) is satisfied
   where Com->E is the previous global error bound.
   We assume that the local error Com->e was previously computed.
   The global error and testtol are recomputed.
   The goal is to decide what is done next. There are three possibilities:

   1. Terminate with status = PASA_ERROR_TOLERANCE_SATISFIED if the global
      error is sufficiently small.
   2. Terminate AG and return to grad_proj where bound variables are freed.
   3. Continue the iteration.

   Recall that the optimization problem is

   min f(x) subject to lo <= x <= hi, bl <= Ax <= bu

   We reformulate this as

   min f(x) subject to lo <= x <= hi, bl <= b <= bu, Ax - b = 0.

   Let L(x, y) = f(x) + y'(Ax - b) be the Lagrangian where b_i = 0 if
   bl_i < (Ax)_i < bu_i, b_i = bu_i if (Ax)_i = bu_i, and b_i = bl_i otherwise.
   The first-order optimality conditions are that

       L'_j (x, y)  = 0 if lo_j < x_j < hi_j,
       L'_j (x, y) <= 0 if        x_j = hi_j,
       L'_j (x, y) >= 0 if lo_j = x_j,
           y_i      = 0 if bl_i < (Ax)_i < bu_i,
           y_i     <= 0 if bl_i = (Ax)_i
           y_i     >= 0 if        (Ax)_i = bu_i

   In null_project (previously executed) we determined y to minimize
   ||nabla f(x) - B'y||_2 and then stored in Com->e the quantity
   ||nabla f(x) - B'y||_inf (the absolute largest component), where B
   is the submatrix of A corresponding to the active constraints at x.
   Thus Com->e stores the local error in the manifold associated with
   the active constraints at x. To obtain an (over) estimate for the
   global error at x, set the components of y corresponding to the
   inactive inequalities to 0.  Let maxsignerr denote the maximum
   violation of the sign constraints in the first-order optimality
   conditions. It is assumed that the current iterate is feasible.
   Our decision about what to do next is based on the following rules:

   A. If MAX (Com->e, maxsignerr) <= Com->grad_tol,
      then we terminate with status = PASA_ERROR_TOLERANCE_SATISFIED.
   B. If Com->e <= switchfactor*maxsignerr, then we branch to grad_proj.
      The rationale is that we wish to determine the active constraints
      at the solution and reduce maxsignerr.
   C. Otherwise, continue the active gradient projection iteration.

   NOTE: This routine is basically a simplified version of cg_checktol.
   ========================================================================= */
int pasaAG_checktol
(
    PASAcom    *Com  /* pasa com structure */
)
{
    PASAINT j, k, nc, nrow, *bound_cols, *RLinkUp ;
    PASAFLOAT E, maxsignerr, *userg, *lambda ;
    PASAparm *Parm ;      /* parameters for PASA */
    PPwork *Work ;

    Parm = Com->pasaparm ; /* parameters */
    const int PrintLevel = Parm->PrintLevel ;
    const int use_pproj = Com->use_pproj ;
    const int use_napheap = Com->use_napheap ;
    int Aexists = Com->Aexists ;

    if ( Com->QP )
    {
        userg = Com->gtot ;
    }
    else
    {
        userg = Com->userg ;
    }

    maxsignerr = PASAZERO ;
    nc = Com->nc ;
    bound_cols = Com->bound_cols ;
    nrow = Com->nrow ;
    if ( use_pproj == TRUE )
    {
        Work = Com->ppcom->Work ;
        RLinkUp = Work->RLinkUp ;
        /* the problem is effectively bound constrained if there are no
           active equations */
        if ( RLinkUp [nrow] == nrow )
        {
            Aexists = FALSE ;
        }
    }
    else if ( use_napheap == TRUE )
    {
        if ( (Com->nap_constraint == 0) || (Com->nap_a2 == PASAZERO) )
        {
            Aexists = FALSE ;
        }
    }

    if ( Aexists == FALSE ) /* only bound constraints present */
    {
        for (k = 0; k < nc; k++)
        {
            j = bound_cols [k] ;
            if ( j < 0 ) /* at lower bound */
            {
                j = -(j+1) ;
                maxsignerr = PASAMAX (maxsignerr, -userg [j]) ;
            }
            else
            {
                maxsignerr = PASAMAX (maxsignerr, userg [j]) ;
            }
        }
    }
    else if ( use_pproj == TRUE ) /* linear equations/inequalities present*/
    {
        PASAINT l, m, nr, p, q,
             *Ap, *Ai, *bound_rows, *invperm ;
        PASAFLOAT t, *Ax ;

        /* lambda was computed inside null_project */
        lambda = Com->lambda ;

        Ap = Com->Copy->Ap ;
        Ax = Com->Copy->Ax ;
        Ai = Com->Copy->Ai ;
        invperm = Com->invperm ;
        for (k = 0; k < nc; k++)
        {
            j = bound_cols [k] ;
            if ( j < 0 ) /* at lower bound */
            {
                m = -(j+1) ;      /* user indexing */
                l = invperm [m] ; /* convert from user index to pproj index */
            }
            else
            {
                m = j ;
                l = invperm [j] ;
            }
            t = PASAZERO ;
            q = Ap [l+1] ;
            for (p = Ap [l]; p < q; p++)
            {
                t -= lambda [Ai [p]]*Ax [p] ;
            }
            t += userg [m] ;
            if ( j < 0 ) /* variable at lower bound */
            {
                maxsignerr = PASAMAX (maxsignerr, -t) ;
            }
            else
            {
                maxsignerr = PASAMAX (maxsignerr, t) ;
            }
            if ( PrintLevel >= 2 )
            {
                if ( j < 0 )
                {
                    if ( t < PASAZERO )
                    {
                        printf("lower bound %ld (pproj index) wrong sign: %e\n",
                              (LONG) l, -t) ;
                    }
                }
                else
                {
                    if ( t > PASAZERO )
                    {
                        printf("upper bound %ld (pproj index) wrong sign: %e\n",
                              (LONG) l, t) ;
                    }
                }
            }
        }
        nr = Com->nr ;
        bound_rows = Com->bound_rows ;
        for (k = 0; k < nr; k++)
        {
            j = bound_rows [k] ;
            if ( j < 0 ) /* at lower bound */
            {
                m = -(j+1) ;      /* row index */
                maxsignerr = PASAMAX (maxsignerr, -lambda [m]) ;
            }
            else
            {
                m = j - 1 ;
                maxsignerr = PASAMAX (maxsignerr, lambda [m]) ;
            }
            if ( PrintLevel >= 2 )
            {
                t = lambda [m] ;
                if ( j < 0 )
                {
                    if ( t < PASAZERO )
                    {
                        printf ("lower ineqn %ld wrong sign: %e\n",
                               (LONG) m, -t) ;
                    }
                }
                else
                {
                    if ( t > PASAZERO )
                    {
                        printf ("upper ineqn %ld wrong sign: %e\n",
                               (LONG) m, t) ;
                    }
                }
            }
        }
    }
    else /* use_napheap = TRUE */
    {
        PASAINT m ;
        const PASAFLOAT Lambda = Com->nap_lambda ;
        for (k = 0; k < nc; k++)
        {
            j = bound_cols [k] ;
            if ( j < 0 ) /* at lower bound */
            {
                m = -(j+1) ;      /* user indexing */
            }
            else
            {
                m = j ;
            }
            const PASAFLOAT t = userg [m] - Lambda*Com->nap_auser [m] ;
            if ( j < 0 ) /* variable at lower bound */
            {
                maxsignerr = PASAMAX (maxsignerr, -t) ;
                if ( (PrintLevel >= 2) && (t < PASAZERO) )
                {
                    printf("lower bound %ld (user index) wrong sign: %e\n",
                          (LONG) m, -t) ;
                }
            }
            else         /* variable at upper bound */
            {
                maxsignerr = PASAMAX (maxsignerr, t) ;
                if ( (PrintLevel >= 2) && (t > PASAZERO) )
                {
                    printf("upper bound %ld (user index) wrong sign: %e\n",
                          (LONG) m, t) ;
                }
            }
        }
        if ( Com->nr > 0 )
        {
            if ( Com->bound_rows [0] < 0 ) /* at lower bound */
            {
                maxsignerr = PASAMAX (maxsignerr, -Lambda) ;
                if ( (PrintLevel >= 2) && (Lambda < PASAZERO) )
                {
                    printf ("lower ineqn %i wrong sign: %e\n", 0, -Lambda) ;
                }
            }
            else                           /* at upper bound */
            {
                maxsignerr = PASAMAX (maxsignerr, Lambda) ;
                if ( (PrintLevel >= 2) && (Lambda > PASAZERO) )
                {
                    printf ("upper ineqn %i wrong sign: %e\n", 0, Lambda) ;
                }
            }
        }
    }

    /* TEST A: If MAX (Com->e, maxsignerr) <= Com->grad_tol,
       then we terminate with status = PASA_ERROR_TOLERANCE_SATISFIED. */
    E = PASAMAX (maxsignerr, Com->e) ;
    if ( E < Com->E )
    {
        Com->E = E ;
        E *= Com->switchfactor ;
        Com->testtol = PASAMAX (E, Com->grad_tol) ;
    }
    if ( PrintLevel >= 1 )
    {
        printf ("maxsignerr: %e Com->E: %e testtol: %e\n",
                 maxsignerr, Com->E, Com->testtol) ;
    }
    if ( Com->E <= Com->grad_tol )
    {
        if ( PrintLevel >= 1 )
        {
            printf ("checktol: A. Error tolerance satisfied\n") ;
        }
        return (PASA_ERROR_TOLERANCE_SATISFIED) ;
    }

    /* TEST B: If Com->e <= switchfactor*maxsignerr, then we branch
       to grad_proj.  The rationale is that we wish to determine the active
       constraints at the solution and reduce maxsignerr. */
    if ( Com->e <= Com->testtol )
    {
        if ( PrintLevel >= 1 )
        {
            printf ("AG checktol: B. Branch to grad_proj\n") ;
        }
        return (PASA_GRAD_PROJ) ;
    }

    /* TEST C. Otherwise, continue the active gradient projection iteration. */
    if ( PrintLevel >= 1 )
    {
        printf ("AG checktol: C. Continue AG\n") ;
    }
    return (PASA_OK) ;
}
#endif

/* ==========================================================================
   === pasa_update_bb =======================================================
   ==========================================================================
    A cycle version of the BB algorithm is employed. This means that we
    reuse the BB approximation bbk*I to the Hessian for one or more
    iterations before updating its value. This function attempts to update
    bbk using the BB formula (g_k - g_k-1)'(x_k - x_k-1)/||x_k - x_k-1||^2.
    If either the numerator is <= 0 or the denominator = 0, then there are
    two possible outcomes: If the BB cycle length has reached the maximum
    allowed cycle length, then bbk is reinitialized using the startup
    procedure of iteration 1. Before the bb cycle length is reached, the
    function returns -1, which means to retain the current bbk.
   ========================================================================== */
PASAINT pasa_update_bb /* return lambda_k */
(
    PASAcom    *Com  /* common variables */
)
{
    PASAINT k, n, nf, nrow ;
    PASAFLOAT dk, gmax, sy, ss, normx, f, fnew, sy1, sy2, t,
              *x, *d, *g, *xnew, *gnew ;
    PASAparm *Parm ;
    PASAstat *Stat ;

    Parm = Com->pasaparm ;
    Stat = Com->pasastat ;
    x = Com->x ;
    g = Com->g ;

    n = Com->ncol ;
    nf = Com->nf ; /* number of free variables */

    int convex = TRUE ;

    /* At startup, we use the given bbk if provided, otherwise we
       estimate bbk using ||x|| and ||g|| */
    if ( (Stat->gpit == 0) && (Stat->agpit == 0) )
    {
        if ( Parm->bbk == PASAZERO )
        {
            gmax = pasa_sup_normx (g, n) ;
            if ( gmax == PASAZERO ) /* gmax = ||g|| */
            {
                Com->E = PASAZERO ;
                if ( Com->use_napheap )
                {
                    Com->nap_lambda = PASAZERO ;
                    Com->lambda_scale = PASAONE ;
                }
                else if ( Com->use_pproj )
                {
                    nrow = Com->nrow ;
                    for (k = 0; k < nrow; k++) Com->lambda [k] = PASAZERO ;
                    Com->lambda_scale = PASAONE ;
                }
                return (PASA_ERROR_TOLERANCE_SATISFIED) ;
            }
            normx = pasa_sup_normx (x, n) ;    /* ||x_k||_inf */
            /* check if x = 0 */
            if ( normx > PASAZERO )
            {
                t = PASAMAX (Parm->lambda0Factor*normx/gmax,Parm->lambda0);
            }
            else /* x = 0 */
            {
                t = PASAMAX (PASAONE/gmax, Parm->lambda0) ;
            }
            Com->bbk = t ;
        }
        else
        {
            Com->bbk = Parm->bbk ;
        }
        /* stimulate a new evaluation of the BB parameter */
        Com->its_in_cycle = Parm->NominalCycle ;
        return (PASA_OK) ;
    }

    if ( Com->first == TRUE )
    {
        /* Reuse the previous choice for the BB parameter.
           Next iteration, we want to try to compute a new value
           for the BB parameter, so we inialize its_in_cycle =
           NominalCycle. */
        Com->its_in_cycle = Parm->NominalCycle ;
        return (PASA_OK) ;
    }

    t = -PASAONE ; /* initialize BB parameter */

    /* Try to compute a new value for the BB parameter in any of the
       following cases: 1. The iteration did not employ a unit step.
                        2. The number of iterations in the current cycle
                           exceeds the nominal cycle length. */
    Com->its_in_cycle++ ;
    if ( (Com->alpha < PASAONE) || (Com->its_in_cycle >= Parm->NominalCycle) )
    {
        if ( Parm->PrintLevel )
        {
            printf ("update bb step (alpha: %e it in cycle: %i "
                    "nominal cycle: %i)\n", Com->alpha, Com->its_in_cycle,
                     Parm->NominalCycle) ;
        }
        xnew = Com->xnew ;
        gnew = Com->gnew ;
        d = Com->d ;
        sy = PASAZERO ;
        ss = PASAZERO ;
        gmax = PASAZERO ;
        for (k = 0; k < nf; k++)
        {
            if ( gmax < fabs (gnew [k]) ) gmax = fabs (gnew [k]) ;
            dk = d [k] ;                     /* dk = kth component of d */
            if ( dk != PASAZERO )
            {
                ss += dk*dk ;                /* ss = ||s||^2 (neglect alpha)*/
                sy += dk*(gnew [k] - g [k]) ;/* update sy (BB denominator)*/
            }
        }
        ss *= Com->alpha ;      /* adjust for neglected alpha */

        /* check if sty is nonpositive OR if ss = 0 */
        if ( sy <= PASAZERO || ss == PASAZERO )
        {
            convex = FALSE ;
            if ( Parm->PrintLevel > 0 )
            {
                printf ("BB step: sy = %e <= 0 or ss = %e\n", sy, ss) ;
            }
            /* check if current cycle >= MaximumCycle */
            if ( Com->its_in_cycle >= Parm->MaximumCycle )
            {
                if ( Parm->PrintLevel > 0 )
                {
                    printf ("its_in_cycle = %i >= MaximumCycle = %i\n",
                             Com->its_in_cycle, Parm->MaximumCycle) ;
                }
                if ( gmax == PASAZERO )
                {
                    Com->E = PASAZERO ;
                    if ( Com->use_napheap )
                    {
                        Com->nap_lambda = PASAZERO ;
                        Com->lambda_scale = PASAONE ;
                    }
                    else if ( Com->use_pproj )
                    {
                        nrow = Com->nrow ;
                        for (k = 0; k < nrow; k++) Com->lambda [k] = PASAZERO ;
                        Com->lambda_scale = PASAONE ;
                    }
                    return (PASA_ERROR_TOLERANCE_SATISFIED) ;
                }
                normx = pasa_sup_normx (xnew, nf) ;    /* ||xnew_k||_inf */
                /* check if x = 0 */
                if ( normx > PASAZERO )
                {
                    t = PASAMAX(Parm->lambda0Factor*normx/gmax,
                                Parm->lambda0);
                }
                else /* x = 0 */
                {
                    t = PASAMAX (PASAONE/gmax, Parm->lambda0) ;
                }
                if ( t < Parm->bbexpand*Com->bbk )
                {
                    t = Parm->bbexpand*Com->bbk ;
                }
            }
        }
        else /* lambda_k possibly updated by BB formula */
        {
            /* If the BB formula is not used in this iteration, then we want to
               try it in the next iteration. Hence, we inialize its_in_cycle =
               NominalCycle. Whenever we use the BB formula, we set
               its_in_cycle = 0. */
            Com->its_in_cycle = Parm->NominalCycle ;
            fnew = Com->fnew ;
            f    = Com->f ;
            /* If the relative change in the function value is small enough,
               then we always use the BB step */
            if ( fabs (f-fnew) < Parm->bbSwitchFactor*fabs (f) )
            {
                t = ss/sy ; /* s's/s'y */
                if ( t >= Parm->lambda0 )
                {
                    Com->its_in_cycle = 0 ;
                }
                else
                {
                    t = Parm->lambda0 ;
                }
                if ( Parm->PrintLevel > 0 )
                {
                    printf ("use BB formula for step: %e\n", t) ;
                }
            }
            else /* further investigate the convexity of f */
            {
                /* If function is not convex along the line segment between x
                   and xnew, then we do not use BB step. Above, we already
                   checked if the denominator sy < 0 in the BB formula.
                   In addition, we now consider additional tests for a loss
                   of convexity based on the formulas from COAP, 22 (2002),
                   pp. 103-109 by Dai and Yuan^2. Let sy1 and sy2 denote the
                   numerator of (2.7) and (2.10) respectively. If sy1 < 0 or
                   sy2 < 0, the function is not convex between x and xnew.
                   Also, if sy1 > 2*sy, then the function is not convex.
                   This is because
                       sy1 = 2*s_{k-1}'*
                       int_0^1 nabla^2 f(x_k + t(x_{k-1}-x_k))(1-t)dt s_{k-1}
                   while sy is the same thing but with the factor 2 deleted
                   as well as the (1-t) factor in the integral. */

                sy1 = 2.*((f - fnew)/Com->alpha + sy + Com->gtd) ;
                sy2 = 6.*(Com->gtd + (f - fnew)/Com->alpha) + 4.*sy ;

                if ( (sy1 > PASAZERO) && (sy2 > PASAZERO) && (sy1 <= 2*sy) )
                {
                    t = ss/sy ; /* s's/s'y */
                    if ( Parm->PrintLevel > 0 )
                    {
                        printf ("function passes convexity test, use BB "
                                "formula for step: %e\n", t) ;
                    }
                    if ( t >= Parm->lambda0 )
                    {
                        Com->its_in_cycle = 0 ;
                    }
                    else
                    {
                        t = Parm->lambda0 ;
                    }
                }
                else
                {
                    convex = FALSE ;
                    normx = pasa_sup_normx (xnew, nf) ;    /* ||xnew_k||_inf */
                    /* check if x = 0 */
                    if ( normx > PASAZERO )
                    {
                        t = PASAMAX(Parm->lambda0Factor*normx/gmax,
                                    Parm->lambda0);
                    }
                    else /* x = 0 */
                    {
                        t = PASAMAX (PASAONE/gmax, Parm->lambda0) ;
                    }
                    if ( Parm->PrintLevel > 0 )
                    {
                        printf ("function not convex, use %e\n", t) ;
                    }
                }
            }
        }
    }
    if ( t > PASAZERO )
    {
        if      ( !convex )           Com->bbk = t ;
        else if ( t <= 500*Com->bbk ) Com->bbk = t ;
        else                          Com->bbk *= 500 ;
    }
    if ( Parm->PrintLevel )
    {
        printf ("final bbk: %e convex: %i\n", Com->bbk, convex) ;
    }
    return (PASA_OK) ;
}

/* ==========================================================================
   === pasa_update_reference ================================================
   ==========================================================================
    The Armijo line search condition is satisfied when the function value
    is small enough relative to a reference function value fR. This function
    updates fR. The following internal variable are used in the update process:

    f_min(k)   = min { fj : j <= k}
    fmaxmin(k) = max { fi : j <= i <= k, fj = f_min(k), j as small as possible}
    f_max(k)   = max { fj : max(k-M, 0) < j <= k }, M = memory of func values
    fr(k)      = fmaxmin or f_max or fr(k-1) according to the rules below

    In the first iteration of a bb cycle, we return fr while in the
    subsequent iteration, we return the current function value fk. This
    implies that the subsequent iterations are monotone while the first
    is nonmonotone.
   ========================================================================== */
PASAFLOAT pasa_update_reference
(
    PASAcom   *Com /* common variables */
)
{
    PASAparm *Parm ;

    Parm = Com->pasaparm ;

    int k, memp ;
    PASAFLOAT f, fdelete, f_max, *fmem, fr, t ;

    if ( Parm->PrintLevel > 0 )
    {
        printf ("update reference\n") ;
    }
    f = Com->f ;
    if ( Com->first == TRUE ) /* do initialization */
    {
        /* it is assumed that update_reference is called after update_bb
           so we can set first = FALSE here */
        Com->fr = f ;
        Com->f_min = f ;
        Com->fmaxmin = f ;
        Com->f_max = f ;
        /* Parm->M prior function values stored in fmem */
        Com->fmem [0] = f ;
        Com->memp = 1 ;
        for (k = 1; k < Parm->M; k++)
        {
            Com->fmem [k] = -PASAINF ; /* initialize fmem [k] = -infinity */
        }
        Com->its_since_f_min_update = 0 ; /* count iters since f_min update */
        Com->nunit_steps = 0 ;
        return (f) ;
    }

    if ( Com->alpha == PASAONE )
    {
        Com->nunit_steps++ ;
    }
    else
    {
        Com->nunit_steps = 0 ;
    }

    /* update f_min and fmaxmin */
    if ( f < Com->f_min )
    {
        Com->f_min = f ;       /* set f_min = f_{k+1} */
        Com->fmaxmin = f ;     /* set fmaxmin = f_{k+1} */
        Com->its_since_f_min_update = 0 ;  /* count its since f_min update */
    }
    else /* else f_min value remains the same */
    {
        Com->its_since_f_min_update++ ;

        if ( f > Com->fmaxmin ) /* if fmaxmin < f, update its value */
        {
            Com->fmaxmin = f ; /* update fmaxmin value */
        }
    }

    /* update f_max */
    memp = Com->memp ;
    fmem = Com->fmem ;
    f_max = Com->f_max ;

    fdelete = fmem [memp] ;  /* fdelete = value removed from fmem */
    fmem [memp] = f ;       /* store new function value in fmem vector */
    memp++ ;                /* increase fmemp value */

    /* reset memp to 0 when it reaches M */
    if ( memp == Parm->M )
    {
        memp = 0 ;
    }
    Com->memp = memp ;

    if ( f > f_max ) /* update f_max */
    {
        f_max = f ; /* set f_max = new function value f */
    }
    else if ( f_max == fdelete ) /* need to search memory for max value */
    {
        f_max = -PASAINF ;
        for (k = 0; k < Parm->M; k++)
        {
            if ( fmem [k] > f_max )
            {
                f_max = fmem [k] ;
            }
        }
    }
    Com->f_max = f_max ;

    /* Update reference function value. First check if f_min value
       has been used L times */
    fr = Com->fr ;

    if ( Com->its_since_f_min_update == Parm->L )
    {
        fr = f_max ;
        Com->its_since_f_min_update = 0 ;   /* reset f_min iters counter */
        t = Com->fmaxmin - Com->f_min ;
        if ( t > PASAZERO )
        {
            t = (fr - Com->f_min)/t ;
            if ( t > Parm->gamma1 )
            {
                fr = Com->fmaxmin ; /* update reference function value */
            }
        }
    }

    /* check if number of unit steps exceeds parameter P AND if f_max > f */
    if ( (Com->nunit_steps > Parm->P) && (f_max > f) )
    {
        if ( (fr - f)/(f_max - f) > Parm->gamma2 )
        {
            fr = f_max ;    /* update reference function value */
        }
    }

    /* check if fr differs too much from f */
    if ( fr - f > Parm->gamma3*fabs(f) )
    {
         fr = f_max ;
    }

    if ( Parm->PrintLevel > 0 )
    {
        printf ("fr: %e f_max: %e fmaxmin: %e f_min: %e\n",
        fr, f_max, Com->fmaxmin, Com->f_min) ;
    }
    if ( fr < f ) fr = f ;
    Com->fr = fr ;            /* save the new reference function value */

    if ( Com->its_in_cycle == 0 )
    {
        return (fr) ;
    }
    else
    {
        return (f) ;
    }
}

/* ==========================================================================
   === pasa_gradproj_step ===================================================
   ==========================================================================
    Project xk - stepsize*gk onto the polyhedron. The projection pk is
    stored in Com->xnew and the search direction dk = pk - xk is
    stored in Com->d. If A does not exist, then we also compute global and
    local errors.
   ========================================================================== */
int pasa_gradproj_step /* return:
                              PASA_OK
                              PASA_POLYHEDRON_INFEASIBLE
                              PASA_INVALID_LINEAR_CONSTRAINT_BOUNDS
                              PASA_OUT_OF_MEMORY_IN_PPROJ
                              PASA_SSOR_ASCENT_FAILURE_IN_PPROJ
                              PASA_SSOR_ITERATIONS_EXCEED_MAX_ITS_IN_PPROJ */
(
    PASAFLOAT const stepsize,
    PASAcom             *Com  /* common data for PASA */
)
{
    int location, PrintLevel, status ;
    PASAINT i, j, k, ncol, nrow, nf, p, *ifree, *ir ;
    PASAFLOAT E, gmax, t, *d, *g, *lo, *hi, *x, *xnew, *rhs ;
    PASAparm *Parm ;
    const int loExists = Com->loExists ;
    const int hiExists = Com->hiExists ;

    Parm = Com->pasaparm ;
    PrintLevel = Parm->PrintLevel ;

    /* get gmax if iterates are printed or error is scaled by gmax */
    int const get_gmax = (PrintLevel || Parm->scale_error_by_gmax ) ;
    nf = Com->nf ;
    g = Com->g ;
    if ( PrintLevel )
    {
        printf ("start gradproj_step (actual step: %e", stepsize) ;
    }
    Com->actual_step = stepsize ;
    if ( PrintLevel ) printf (" actual step: %e)\n", stepsize) ;

    ncol = Com->ncol ;
    d = Com->d ;
    xnew = Com->xnew ;
    x = Com->x ;
    location = Com->location ;
    Com->lambda_scale = stepsize ;

    /* there is a single linear constraint and napheap is used */
    if ( Com->use_napheap )
    {
        NAPdata *napdata ;
        napdata = Com->pasadata->napdata ;
        /* d = x - stepsize*g */
        if ( get_gmax ) /* also compute & save gmax */
        {
            Com->gmax = pasa_step_max (d, x, g, -stepsize, nf) ;
        }
        else /* make step but do not return gmax */
        {
            pasa_step (d, x, g, -stepsize, nf) ;
        }

        /* update the napdata structure */
        napdata->x   = xnew ;
        napdata->n   = Com->nf ;
        napdata->c   = d ;
        napdata->blo = Com->nap_bl ;
        napdata->bhi = Com->nap_bu ;
        /* Com->lambda stores the current estimate for the multiplier
           associated with the NLP we are trying to solve. The minus sign
           is due to the fact that the multiplier sign is flipped in napheap. */
        napdata->lambda = -Com->nap_lambda ;

        status = napheap (napdata) ;

        Com->pasastat->nproject++ ;
        /* flip sign of lambda for consistency with rest of code */
        Com->nap_lambda = -napdata->lambda ;
        pasa_step (d, xnew, x, -PASAONE, nf) ;

        /* compute derivative in the search direction */
        Com->gtd = pasa_dot (g, d, nf) ;

        /* check that d is a descent direction if PrintLevel >= 2 */
        if ( PrintLevel >= 2 )
        {
            if ( Com->gtd > PASAZERO )
            {
                printf ("search direction not a descent direction: %e\n",
                         Com->gtd);
            }
            else
            {
                printf ("gtd: %e\n", Com->gtd) ;
            }
        }
        if ( status > 0 )
        {
            return (status) ;
        }

        /* The following estimates only works in grad_proj or the base code
           where there are no active constraints or bounds. Also, when
           gtd > 0, we skip this error bound since it may be invalid due
           to infeasibility of the iterate. */
        /* if ( Com->gtd <= PASAZERO ) */
        {
            if ( (stepsize == PASAONE) && !Com->nc && !Com->nr )
            {
                Com->Egp = pasa_sup_normx (d, nf) ;
                Com->testtol = Com->switchfactor*Com->Egp ;
                Com->testtol = PASAMAX (Com->testtol, Com->grad_tol) ;
            }
            else if ( location == PASA_GRAD_PROJ ) /* => nf = ncol */
            {
                PASAFLOAT estE = pasa_sup_normx (d, ncol)/stepsize ;
                if ( PrintLevel >= 1 )
                {
                    printf ("Global Error estimate: %e\n", estE) ;
                }
                /* if stepsize <= 1, then ||d||/stepsize is an upper bound on
                   the error */
                if ( stepsize <= 1 )
                {
                    if ( estE < Com->Egp )
                    {
                        Com->Egp = estE ;
                        Com->testtol = Com->Egp*Com->switchfactor ;
                        Com->testtol = PASAMAX (Com->testtol, Com->grad_tol) ;
                    }
                }
                else /* stepsize > 1, ||d|| is upper bound on error */
                {
                    if ( stepsize*estE < Com->Egp )
                    {
                        Com->Egp = stepsize*estE ;
                        Com->testtol = Com->switchfactor * Com->Egp ;
                        Com->testtol = PASAMAX (Com->testtol, Com->grad_tol) ;
                    }
                }
            }
        }
        if ( PrintLevel >= 1 )
        {
            printf ("Global Error Egp: %e testtol: %e\n",
                     Com->Egp, Com->testtol) ;
        }
        return (PASA_OK) ;
    }

    /* If A does not exist, then column permutation of A does not enter
       into the analysis and the computation of local and global error and
       the step is simplified. */
    if ( !Com->Aexists )
    {
        PASAFLOAT e, gtd, s ;
        PASAINT nc, *bound_cols ;

        e = PASAZERO ;
        E = PASAZERO ;
        gtd = PASAZERO ;

        /* If the problem is unconstrained, the computation trivializes */
        if ( !Com->Bounds )
        {
            Com->gmax = Com->e = Com->Egp = pasa_sup_normx (g, ncol) ;
            /* line search direction = -stepsize*g */
            pasa_scale (d, g, -stepsize, ncol) ;
            gtd = pasa_dot (g, d, ncol) ;
            pasa_step (xnew, x, d, PASAONE, ncol) ;
            if ( PrintLevel >= 1 )
            {
                printf ("global error Egp: %e local error: %e\n",
                         Com->Egp, Com->e) ;
            }
            return (PASA_OK) ;
        }

        ifree = Com->ifree ;
        lo = Com->lo ;
        hi = Com->hi ;
        gmax = PASAZERO ;
        if ( (location == PASA_GRAD_PROJ) || (location == PASA_BASE_CODE) )
        {
            /* in grad proj or the base code, we make the step and compute
               both the local and global error */
            for (j = 0; j < ncol; j++)
            {
                PASAFLOAT const gj = g [j] ;
                if ( get_gmax ) if ( gmax < fabs (gj) ) gmax = fabs(gj) ;
                s = -stepsize*gj ;
                PASAFLOAT const Xj = x [j] ;
                PASAFLOAT const T = s + Xj ;

                PASAFLOAT const loj = (loExists) ? lo [j] : -PASAINF ;
                PASAFLOAT const hij = (hiExists) ? hi [j] :  PASAINF ;

                if ( T < loj )
                {
                    xnew [j] = loj ;
                    s = loj - Xj ;
                }
                else if ( T > hij )
                {
                    xnew [j] = hij ;
                    s = hij - Xj ;
                }
                else
                {
                    xnew [j] = T ;
                }
                gtd += s*gj ;
                d [j] = s ;

                /* free components enter into local error */
                if ( (Xj > loj) && (Xj < hij) )
                {
                    if ( e < fabs(gj) ) e = fabs(gj) ;
                }

                /* all components enter into global error */
                t = Xj - gj ;
                if ( t <= loj )
                {
                    s = loj - Xj ;
                }
                else if ( t >= hij )
                {
                    s = hij - Xj ;
                }
                else
                {
                    s = gj ;
                }
                if ( E < fabs (s) ) E = fabs (s) ;
            }
            Com->Egp = E ;
            E *= Com->switchfactor ;
            Com->testtol = PASAMAX (E, Com->grad_tol) ;
            Com->e = e ;
            if ( PrintLevel >= 1 )
            {
                printf ("global error Egp: %e local error: %e testtol: %e\n",
                         Com->Egp, Com->e, Com->testtol) ;
            }
        }
        /* in active_gp, only make the step, no error estimation */
        else if ( location == PASA_ACTIVE_GP )
        {
            nc = Com->nc ;
            ASSERT (nc+nf == ncol) ;
            bound_cols = Com->bound_cols ;
            for (j = 0; j < nf; j++)
            {
                PASAFLOAT const gj = g [j] ;
                if ( get_gmax && (gmax < fabs (gj)) ) gmax = fabs(gj) ;
                s = -stepsize*gj ;
                PASAFLOAT const Xj = x [j] ;
                t = s + Xj ;
                PASAFLOAT const loj = (loExists) ? lo [j] : -PASAINF ;
                PASAFLOAT const hij = (hiExists) ? hi [j] :  PASAINF ;
                if ( t < loj )
                {
                    xnew [j] = loj ;
                    s = loj - Xj ;
                    bound_cols [nc] = -(ifree [j]+1) ;
                    nc++ ;
                }
                else if ( t > hij )
                {
                    xnew [j] = hij ;
                    s = hij - Xj ;
                    bound_cols [nc] = ifree [j] ;
                    nc++ ;
                }
                else
                {
                    xnew [j] = t ;
                }
                gtd += s*gj ;
                d [j] = s ;
            }
            Com->nc_temp = nc ;
        }
        Com->gmax = gmax ;
        Com->gtd = gtd ;
        return (PASA_OK) ;
    }

    /* Now consider the case where A is not NULL.  The point
       xk - stepsize*gk is projected into the polyhedron.  This is
       equivalent to minimizing ||x - (xk - stepsize*gk)||.
       We first make the change of variables y = x - xk. The projection
       problem is to minimize ||y - (-stepsize*gk)|| subject to
       bl - A*xk <= A*y <= bu - A*xk, lo - xk <= y <= hi - xk. Since we
       have previously called pproj and its internal variables are
       present, we directly modify the bounds associated with the
       polyhedral constraint to reflect the bounds on y. First, we
       compute rhs = A*xk. */
    PPwork *W ;
    PPprob *Prob ;
    W = Com->ppcom->Work ;
    Prob = Com->ppcom->Prob ;
    rhs = W->arrayd ; /* work space from pproj */
    PASAINT   *Ap = Prob->Ap ;
    PASAINT   *Ai = Prob->Ai ;
    PASAFLOAT *Ax = Prob->Ax ;
    ir = W->ir ; /* ir [i] = 0 for active row */
    nrow = Prob->nrow ;
    ncol = Prob->ncol ; /* note the change in ncol (compressed problem) */
    PASAINT const ni = Prob->ni ;

    /* The goal is to project -stepsize*gk onto the polyhedron.
       First compute -stepsize*gk. If gmax is needed later, also
       evaluate gmax. */
    if ( get_gmax )
    {
        Com->gmax = pasa_scale_max (Com->step_g, g, -stepsize, ncol) ;
    }
    else /* do not evaluate gmax */
    {
        pasa_scale (Com->step_g, g, -stepsize, ncol) ;
    }
    pasa_step (Prob->lo, Com->lo, x, -PASAONE, ncol) ;
    pasa_step (Prob->hi, Com->hi, x, -PASAONE, ncol) ;
#ifndef NDEBUG
    PASAFLOAT *lo2 = NULL ;
    status = PASA_OK ;
    if ( loExists )
    {
        lo2 = (PASAFLOAT *) pasa_malloc (&status, ncol, sizeof (double)) ;
        pasa_copyx (lo2, Prob->lo, ncol) ;
    }
    PASAFLOAT *hi2 = NULL ;
    if ( hiExists )
    {
        hi2 = (PASAFLOAT *) pasa_malloc (&status, ncol, sizeof (double)) ;
        pasa_copyx (hi2, Prob->hi, ncol) ;
    }

    /* If the debugger is turned on, then save copies of the constraints
       in order to check later the accuracy of the solution generated by
       pproj. */
    PASAFLOAT *bl2 = (PASAFLOAT *) pasa_malloc (&status, nrow, sizeof (double));
    PASAFLOAT *bu2 = (PASAFLOAT *) pasa_malloc (&status, nrow, sizeof (double));
    Com->ppdataDEBUG->y    = Com->step_g ; /* point to be projected */
    Com->ppdataDEBUG->lo   = lo2 ;
    Com->ppdataDEBUG->hi   = hi2 ;
    Com->ppdataDEBUG->bl   = bl2 ;
    Com->ppdataDEBUG->bu   = bu2 ;
    Com->ppdataDEBUG->ncol = Com->nf ;
    Com->ppdataDEBUG->Ap   = Com->Ap ;
    Com->ppdataDEBUG->Ai   = Com->Ai ;
    Com->ppdataDEBUG->Ax   = Com->Ax ;
#endif
    /* grab more workspace from pproj, Work->arrayd+nrow */
    PASAFLOAT *absAx = rhs+nrow ;
    pasa_initx (absAx, PASAZERO, nrow) ;
    pasa_initx (rhs, PASAZERO, nrow) ;
    k = p = 0 ; /* p used to index over A and k counts inequality constraints */

    /* At one point, we tried to treat the right sides of the inequality
       constraints in slightly different ways depending on whether the
       problem has been compressed in the activeGP routine. When the problem
       is compressed, we treat the right sides of the inequality constraints
       as zero, even though they may have drifted away from zero due to
       rounding error and the fact that the project code itself only
       solves the projection problem to some specified tolerance. This did
       not work well, so it is now skipped.  */
    if ( FALSE /*Com->ProblemCompressed*/ )
    {
        PASAINT  *Anz = Prob->Anz ;
        for (j = 0; j < ncol; j++)
        {
            PASAFLOAT const Xj = x [j] ;
            PASAINT const q0 = p + Anz [j] ;
            for (; p < q0; p++)
            {
                absAx [Ai [p]] += fabs (Ax [p]*Xj) ;
#ifndef NDEBUG
                if ( ir [Ai [p]] != 0 )
                {
                    printf (" ir [%ld] = %ld != 0\n",
                           (LONG) Ai [p], (LONG) ir [Ai [p]]) ;
                    pasa_error (-1, __FILE__, __LINE__, "gradproj_step") ;
                }
#endif
            }
            PASAINT const q1 = Ap [j+1] ;
            for (; p < q1; p++)
            {
                rhs [Ai [p]] += Ax [p]*Xj ;
#ifndef NDEBUG
                if ( ir [Ai [p]] <= ni )
                {
                    printf (" ir [%ld] = %ld <= %ld\n",
                           (LONG) Ai [p], (LONG) ir [Ai [p]], (LONG) ni) ;
                    pasa_error (-1, __FILE__, __LINE__, "gradproj_step") ;
                }
#endif
            }
        }

        /* adjust the upper and lower bounds for the constraints on Ay */
        for (i = 0; i < nrow; i++) /* i corresponds to permuted constraints */
        {
            if ( ir [i] == 0 )     /* equality constraint */
            {
                Prob->b [i] = PASAZERO ; /* ignores rounding errors */
#ifndef NDEBUG
                bl2 [i] = PASAZERO ;
                bu2 [i] = PASAZERO ;
#endif
            }
            else                   /* strict inequality constraint */
            {
                k++ ;
                Prob->bl [k] = Com->bl [k] - rhs [i] ;
                Prob->bu [k] = Com->bu [k] - rhs [i] ;
#ifndef NDEBUG
                bl2 [i] = Prob->bl [k] ;
                bu2 [i] = Prob->bu [k] ;
#endif
            }
        }
    }
    else /* the problem has not yet been compressed */
    {
        for (j = 0; j < ncol; j++)
        {
            PASAFLOAT const Xj = x [j] ;
            PASAINT const q = Ap [j+1] ;
            for (; p < q; p++)
            {
                PASAFLOAT const AXj = Ax [p]*Xj ;
                PASAINT const row = Ai [p] ;
                rhs [row] += AXj ;
                /* only active rows included in absAx
                   Note: in the first iteration, if the starting guess
                   is feasible, then ir could be zero, and all rows included */
                if ( ir [row] <= ni ) absAx [row] += fabs (AXj) ;
            }
        }
        k = 1 ;
        PASAINT *ineq_row = Prob->ineq_row ;
        /* ineq_row [1], ... , ineq_row [ni] are the row numbers where
           bl_i < bu_i, while ineq_row [ni+1] = nrow */
        PASAINT next_ineq = ineq_row [k] ;
        for (i = 0; i < nrow; i++) /* i corresponds to permuted constraints */
        {
            if ( i < next_ineq )   /* equality constraint */
            {
                Prob->b [i] = Com->b [i] - rhs [i] ;
#ifndef NDEBUG
                bl2 [i] = Prob->b [i] ;
                bu2 [i] = Prob->b [i] ;
#endif
            }
            else                   /* strict inequality constraint */
            {
                Prob->bl [k] = Com->bl [k] - rhs [i] ;
                Prob->bu [k] = Com->bu [k] - rhs [i] ;
#ifndef NDEBUG
                bl2 [i] = Prob->bl [k] ;
                bu2 [i] = Prob->bu [k] ;
#endif
                k++ ;
                next_ineq = ineq_row [k] ;
            }
        }
    }
    W->absAxk = pasa_sup_normx (absAx, nrow) ;

    ASSERT (k == ni+1) ;

    if ( PrintLevel >= 2 )
    {
        if ( location == PASA_GRAD_PROJ ) gmax = pasa_sup_normx (g, ncol) ;
        else                              gmax = pasa_sup_normx (g, Com->nf) ;
        printf ("projection in pasa_gradproj_step\n") ;
        printf ("sup-norm of g: %e stepsize: %e absAxk: %e\n",
                 gmax, stepsize, W->absAxk) ;
    }
    /* project Com->step_g = -alpha*gk onto polyhedron */
    status = pasa_project (d, Com->step_g, Com) ;

    if ( PrintLevel >= 2 )
    {
        printf ("sup-norm of d: %e sup-norm of lambda: %e absAx: %e\n",
                 pasa_sup_normx (d, ncol), pasa_sup_normx (Com->lambda, nrow),
                 W->absAx) ;
    }
    /* if an error occurred, then break */
    if ( status > 1 )
    {
        return (status) ;
    }

#ifndef NDEBUG
    /* If debug is TRUE, then check that all rows in the active part of
       A are active. Also, compare solution generated by pproj to
       solution obtained with starting guess = 0 and no userI structure.

       To check that the algorithm does a fast return when the starting
       guess = computed solution, remove comments from the statements
       labelled "fast return" and change "if 1" to "if 0" where indicated
       below. */
    if ( Parm->debug == TRUE )
    {
        PASAFLOAT s, *Ax2, *comlo, *comhi, *combu, *combl, *dnew,
                 *lambdanew, *lambdaold ;
        PASAINT p0, *Ai2, *Anz, *w1, *w2 ;
        PPparm *parm ;
        PPstat *stat ;
        Anz = Prob->Anz ;
        for (j = 0; j < ncol; j++)
        {
            PASAINT const q = Ap [j] + Anz [j] ;
            for (p = Ap [j]; p < q; p++)
            {
                if ( ir [Ai [p]] > k ) /* row is dropped */
                {
                    printf ("row: %ld col: %ld in active part of A, but "
                            "row is dropped since ir = %ld > ni = %ld\n",
                            (LONG) Ai [p], (LONG) j, (LONG) ir [Ai [p]],
                            (LONG) k) ;
                    pasa_error (-1, __FILE__, __LINE__, "gradproj_step") ;
                }
            }
        }
        stat = Com->pprojstat ;
        comlo = Com->lo ;
        comhi = Com->hi ;
        combl = Com->bl ;
        combu = Com->bu ;

        /* change "if 1" to "if 0" to check !!fast return!! */
#if 1
#ifndef NOPPROJ
        Com->pprojstat = NULL ;
        Com->lo = lo2 ;
        Com->hi = hi2 ;
        Com->bl = bl2 ;
        Com->bu = bu2 ;

        /* for starting guess lambda = 0, need sorted columns */
        Ax2 = (double *) pasa_malloc (&status, Ap [ncol], sizeof (double)) ;
        Ai2 = (PASAINT *) pasa_malloc (&status, Ap [ncol], sizeof (PASAINT)) ;
        w1 = (PASAINT *) pasa_malloc (&status, nrow, sizeof (PASAINT)) ;
        w2 = (PASAINT *) pasa_malloc (&status, nrow, sizeof (PASAINT)) ;
        p = 0 ;
        for (j = 0; j < ncol; j++)
        {
            /* sort column j */
            pasa_minsorti (w1, Ai+Ap [j], w2, Ap [j+1] - Ap [j]) ;
            p0 = p ;
            PASAINT const q = Ap [j+1] ;
            for (; p < q; p++)
            {
                i = w1 [p-p0] ;
                Ax2 [p] = Ax [p0+i] ;
                Ai2 [p] = Ai [p0+i] ;
            }
        }
        Com->Ax = Ax2 ;
        Com->Ai = Ai2 ;
#endif
#endif

        parm = Com->pprojparm ;
        Com->pprojparm = NULL ;
        /* Com->pprojparm = parm ;*/ /* !!fast return!! */
        dnew = (PASAFLOAT *) pasa_malloc (&status, ncol, sizeof (PASAFLOAT)) ;
        lambdanew = (PASAFLOAT *) pasa_malloc (&status, nrow,
                                               sizeof (PASAFLOAT)) ;
        lambdaold = Com->lambda ;
        /* pasa_copyx (lambdanew, lambdaold, nrow) ; */ /*!!fast return!!*/
        Com->lambda = lambdanew ;
        /* Com->lambda = lambdaold ;*/ /* !!fast return!! */
        /* ----------------------------------------- */
        status = pasa_project (dnew, Com->step_g, Com) ;
        /* ----------------------------------------- */
        Com->pprojparm = parm ;
        Com->pprojstat = stat ;
        Com->Ax = Ax ;
        Com->Ai = Ai ;

        Com->pasastat->nproject-- ;
        /* compare dnew and d */
        s = PASAZERO ;
        t = PASAZERO ;
        for (j = 0; j < ncol; j++)
        {
            t += fabs(dnew [j] - d [j]) ;
            s += fabs(dnew [j]) ;
        }
        if ( t == PASAZERO )
        {
            printf ("\n") ;
        }
        else
        {
            printf (" rel error: %e\n", t/s) ;
        }
        /* report error if t too big */
        if ( (t > Com->GPtol*1.e6) && (t > s*Com->GPtol*1.e7) )
        {
            printf ("pasa projection error, abs error: %e rel error: %e\n",
                     t, t/s) ;
            pasa_error (-1, __FILE__, __LINE__, "projection error in pasa") ;
        }

        /* compare lambdanew and lambdaold */
        if ( (nrow > 0) && (Ap [ncol] > 0) )
        {
            s = PASAZERO ;
            t = PASAZERO ;
            for (j = 0; j < nrow; j++)
            {
                t += fabs(lambdanew [j] - lambdaold [j]) ;
                s += fabs(lambdanew [j]) ;
            }
            printf ("abs pasa multiplier error: %e", t) ;
            if ( t == PASAZERO )
            {
                printf ("\n") ;
            }
            else
            {
                printf (" rel error: %e\n", t/s) ;
            }
        }
        /* report error if t too big */
        /*if ( t > s*Com->GPtol*1.e9 )
        {
            printf ("pasa multiplier error, abs error: %e rel error: %e\n",
                     t, t/s) ;
            pasa_error (-1, __FILE__, __LINE__,
                                               "multiplier error in pasa") ;
        }*/
        /* restore Com structure */
        Com->lambda = lambdaold ;
        Com->pprojparm = parm ;
        Com->pprojstat = stat ;
        Com->lo = comlo ;
        Com->hi = comhi ;
        Com->bl = combl ;
        Com->bu = combu ;
        /* free memory */
        pasa_free (dnew) ;
        pasa_free (lambdanew) ;
        if ( Ai2 != NULL )
        {
            pasa_free (Ax2) ;
            pasa_free (Ai2) ;
        }
        if ( nrow != 0 )
        {
            pasa_free (w1) ;
            pasa_free (w2) ;
        }
    }
    if ( lo2 != NULL )
    {
        pasa_free (lo2) ;
    }
    if ( hi2 != NULL )
    {
        pasa_free (hi2) ;
    }
    pasa_free (bl2) ;
    pasa_free (bu2) ;
#endif
    /* compute derivative in the search direction */
    Com->gtd = pasa_dot (g, d, ncol) ;
    /* check that d is a descent direction if PrintLevel >= 2 */
    if ( PrintLevel >= 2 )
    {
        if ( Com->gtd > PASAZERO )
        {
            printf ("search direction not a descent direction: %e\n",
                     Com->gtd);
        }
        else
        {
            printf ("gtd: %e\n", Com->gtd) ;
        }
    }
    if ( status > 1 ) /* pproj error status */
    {
        return (status) ;
    }

    /* If there are no active rows and no bound columns, then ||d|| provides
       an error bound. If gtd > 0, then this bound is not used since the
       point being project on the convex set may be infeasible. */
    /* if ( Com->gtd <= PASAZERO )*/
    {
        if ( (stepsize == PASAONE) && !Com->nc && !Com->nr )
        {
            Com->Egp = pasa_sup_normx (d, nf) ;
            Com->testtol = Com->switchfactor * Com->Egp ;
            Com->testtol = PASAMAX (Com->testtol, Com->grad_tol) ;
            if ( PrintLevel >= 1 )
            {
                printf ("Global Error Egp: %e testtol: %e\n",
                         Com->Egp, Com->testtol) ;
            }
        }
        else if ( location == PASA_GRAD_PROJ ) /* nf = ncol in grad_proj */
        {
            /* if stepsize <= 1, then ||d||/stepsize is an upper bound on the
               error */
            PASAFLOAT const dnorm = pasa_sup_normx (d, Com->ncol) ;
            PASAFLOAT estE = dnorm/stepsize ;
            if ( PrintLevel >= 1 )
            {
                printf ("Global Error estimate: %e\n", estE) ;
            }
            if ( (stepsize < PASAONE) && (estE < Com->Egp) )
            {
                Com->Egp = estE ;
                Com->testtol = Com->switchfactor * Com->Egp ;
                Com->testtol = PASAMAX (Com->testtol, Com->grad_tol) ;
                if ( PrintLevel >= 1 )
                {
                    printf ("Reset Global error Egp to: %e testtol: %e "
                            "step < 1\n", Com->Egp, Com->testtol) ;
                }
            }
            else if ( (stepsize >= PASAONE) && (stepsize*estE < Com->Egp) )
            {
                Com->Egp = stepsize*estE ;
                Com->testtol = Com->switchfactor * Com->Egp ;
                Com->testtol = PASAMAX (Com->testtol, Com->grad_tol) ;
                if ( PrintLevel >= 1 )
                {
                    printf ("Reset Global error Egp to: %e testtol: %e "
                             "step > 1\n", Com->Egp, Com->testtol) ;
                }
            }
        }
    }
    return (status) ;
}

/* ==========================================================================
   === pasa_null_project ====================================================
   ==========================================================================
    Project the gradient g associated with the free components of the current
    iterate into the null space associated with the current active constraint
    gradients.  If B is the part of the matrix A associated with active linear
    constraints and free components of x, then the projection is given by
    p = g - B'(BB')^{-1}B*g. If B is vacuous, then p = g. If gproj = NULL,
    then the sup norm of the projected gradient is computed and stored in
    Com->e and the projected gradient is discarded.

    The computation of the projected gradient makes use of the LDL'
    factorization of BB' computed in pproj. The matrix L is unit lower
    triangular (that is, ones on the diagonal) and D is diagonal of the form

              | D+ 0 |
              | 0  0 |

    The 0s in the lower right corner correspond to either true 0s or
    element of D equal to cmm->dbound where cmm is the cholmod common
    structure. The QR factorization of B' is Q*sqrt(D)*L' so BB' = LDL'.
    If Q+ denotes the leading columns of Q corresponding to the linearly
    independent rows of B, then the minimization of ||g - B'y|| over y
    is equivalent to computing B'y = Q+ * (Q+'g), the projection of g into
    the row space of B, while g - B'y is the projection of g into the
    null space of B (the space orthogonal to the row space of B). The
    computation of B'y is displayed below:

        B'y = B' * [inv(L)' inv(sqrt(D)) inv(sqrt(D)) inv(L)] B*g

    The expression in [ ... ] is the pseudoinverse of LDL'. If B is replaced
    by L*sqrt(D)*Q', then everything cancels to give Q+ * (Q+'g).
   ========================================================================== */
void pasa_null_project
(
    PASAFLOAT     *gproj, /* the projected gradient */
    PASAFLOAT         *g, /* the gradient to be projected */
    PASAFLOAT      *gpen, /* gradient of penalty term if it exists */
    int const LocalError, /* T => compute local error */
    PASAcom         *Com  /* common data for PASA */
)
{
    int location, PrintLevel, status, *ib ;
    PASAINT ncol, nrow, i, j, k, l, *Ai, *Ap, *Anz, *ifree, *RLinkUp, *RLinkDn ;
    PASAFLOAT e, t, u, *Ax, *G, *dlambda, *lambda ;
    PPprob *Prob ;
    PPwork *W ;
/*cholmod_factor *L ;
PASAFLOAT *Lx ;*/

    PrintLevel = Com->pasaparm->PrintLevel ;
    if ( PrintLevel >= 2 )
    {
        printf ("start null project\n") ;
    }
    ncol = Com->ncol ;
    e = PASAZERO ;
    if ( Com->Aexists )
    {
        Com->lambda_scale = PASAONE ;
    }
    else /* A does not exist */
    {
        /* If Com->Aexist is FALSE, then the code was called from either CG
           or from activeGP.  In this case, simply return the sup-norm of
           the gradient since all the variables in the local problem are free.*/
        if ( LocalError == TRUE )
        {
            Com->e = pasa_sup_normx (g, Com->nf) ;
            if ( PrintLevel >= 1 )
            {
                printf ("Local Error: %e\n", Com->e) ;
            }
        }
        return ;
    }

    /* If there are no free variables, then all variables are at bounds,
       and the projected gradient is 0. */
    location = Com->location ;
    const PASAINT nf = Com->nf ;
    if ( nf == 0 )
    {
        Com->e = PASAZERO ;
        if ( PrintLevel >= 1 )
        {
            printf ("Local Error: %e\n", e) ;
        }
        return ;
    }

    /* use_ifree is TRUE if the free variables are stored in the ifree
       array, otherwise it is FALSE and all the variables are free */
    int const Bounds = Com->Bounds ;
    int const use_ifree = (Bounds && ((location == PASA_BASE_CODE) ||
                                      (location == PASA_GRAD_PROJ)))
                          ? TRUE : FALSE ;

    if ( Com->use_napheap )
    {
        PASAFLOAT *a, *lo, *hi, *x ;
        const int loExists = Com->loExists ;
        const int hiExists = Com->hiExists ;
        lo = Com->lo ;
        hi = Com->hi ;
        x  = Com->x ;
        if ( Com->nap_constraint == 0 ) /* the constraint is inactive */
        {
            if ( (location == PASA_ACTIVE_GP) || (location == PASA_CG_DESCENT) )
            {
                /* compute the sup-norm of the free components of g */
                Com->e = pasa_sup_normx (g, nf) ;
                if ( gproj != NULL )
                {
                    pasa_copyx (gproj, g, nf) ;
                }
            }
            else /* either BASE_CODE or GRAD_PROJ */
            {
                if ( Bounds == FALSE ) /* no bounds */
                {
                    e = pasa_sup_normx (g, nf) ;
                }
                else /* bounds exist, find free components */
                {
                    for (j = 0; j < nf; j++)
                    {
                        PASAFLOAT const xj = x [j] ;
                        if ( (loExists && (xj == lo [j])) ||
                             (hiExists && (xj == hi [j])) )
                        {
                            continue ;
                        }
                        if ( e < fabs (g [j]) )
                        {
                            e = fabs (g [j]) ;
                        }
                    }
                }
                Com->e = e ;
            }
            if ( PrintLevel >= 1 )
            {
                printf ("Local Error: %e\n", e) ;
            }
            return ;
        }
        if ( (location == PASA_CG_DESCENT) && (Com->use_penalty == TRUE)
                                           && (gpen != NULL) )
        {
            /* napxwork is a work area associated with napheap, it can be
               used here because it changes in each call to napheap */
            G = Com->napxwork ;
            /* store g + gpen in G when CG is used */
            pasa_step (G, g, gpen, PASAONE, nf) ;
        }
        else
        {
            G = g ;
        }
        /* Compute B*g and store in lambda. In the base code or in grad_proj,
           the matrix is not compressed, so ib must be used to extract the
           free columns of A */
        a = Com->nap_a ;
        if ( use_ifree )
        {
            ifree = Com->napiwork ;
            u = t = PASAZERO ;
            i = 0 ;
            for (j = 0; j < nf; j++)
            {
                /* skip over variables at bounds */
                if ( (loExists && (x [j] == lo [j])) ||
                     (hiExists && (x [j] == hi [j])) )
                {
                    continue ;
                }
                ifree [i] = j ;
                i++ ;
                const PASAFLOAT aj = a [j] ;
                t += G [j]*aj ;
                u += aj*aj ;
            }
            const PASAFLOAT s = (u > PASAZERO) ? t/u : PASAZERO ;
            Com->nap_lambda = s ; /* multiplier estimate = a'g/a'a */
            const PASAINT nfree = i ;
            for (i = 0; i < nfree; i++)
            {
                j = ifree [i] ;
                t = G [j] - s*a [j] ;
                if ( e < fabs (t) )
                {
                    e = fabs (t) ;
                }
            }
            Com->e = e ;
            if ( PrintLevel >= 1 )
            {
                printf ("Local Error: %e\n", e) ;
            }
            return ;
        }
        /* Bounds = FALSE or location = activeGP or CG */
        t = pasa_dot (G, a, nf) ;
        const PASAFLOAT s = t/Com->nap_a2 ;
        Com->nap_lambda = s ; /* multiplier estimate = a'g/a'a */
        if ( gproj != NULL )
        {
            pasa_step (gproj, G, a, -s, nf) ;
            if ( LocalError )
            {
                Com->e = pasa_sup_normx (gproj, nf) ;
            }
        }
        else if ( LocalError ) /* do not store gproj */
        {
            e = PASAZERO ;
            for (j = 0; j < nf; j++)
            {
                t = G [j] - s*a [j] ;
                if ( e < fabs (t) )
                {
                    e = fabs (t) ;
                }
            }
            Com->e = e ;
        }
        if ( PrintLevel >= 1 )
        {
            printf ("Local Error: %e\n", Com->e) ;
        }
        return ;
    }
    W = Com->ppcom->Work ;
    RLinkUp = W->RLinkUp ;
    RLinkDn = W->RLinkDn ;
    ib = W->ib ;
    nrow = Com->nrow ;
    /* If all rows in the matrix have dropped, then local error is the
       sup-norm of the free components of the gradient. */
    Prob = Com->ppcom->Prob ;
    if ( nrow == RLinkUp [nrow] )
    {
        if ( (location == PASA_ACTIVE_GP) || (location == PASA_CG_DESCENT) )
        {
            /* compute the sup-norm of the free components of g */
            ncol = Prob->ncol ;
            Com->e = pasa_sup_normx (g, ncol) ;
            if ( gproj != NULL )
            {
                pasa_copyx (gproj, g, ncol) ;
            }
        }
        else /* either BASE_CODE or GRAD_PROJ */
        {
            if ( Bounds == FALSE )
            {
                e = pasa_sup_normx (g, Com->ncol) ;
            }
            else
            {
                for (j = 0; j < ncol; j++)
                {
                    if ( ib [j] == 0 )
                    {
                        if ( e < fabs (g [j]) )
                        {
                            e = fabs (g [j]) ;
                        }
                    }
                }
            }
            Com->e = e ;
        }
        if ( PrintLevel >= 1 )
        {
            printf ("Local Error: %e\n", e) ;
        }
        return ;
    }

    /* A exists and use_pproj is TRUE */
    Ap = Prob->Ap ;
    Anz = Prob->Anz ;
    Ai = Prob->Ai ;
    Ax = Prob->Ax ;
    ncol = Prob->ncol ; /* above ncol = Com->ncol, now it is Prob->ncol (nf) */
    G = W->arrayd ;
    if ( (location == PASA_CG_DESCENT) && (Com->use_penalty == TRUE)
                                       && (gpen != NULL) )
    {
        /* store g + gpen in G when CG is used */
        pasa_step (G, g, gpen, PASAONE, ncol) ;
    }
    else
    {
        pasa_copyx (G, g, ncol) ;
    }

#ifndef NDEBUG
    if ( PrintLevel >= 1 )
    {
        printf ("norm g: %e\n", pasa_sup_normx (G, ncol)) ;
    }
#endif
    /* if ifree is used, then extract the free indices from the ib array */
    if ( use_ifree )
    {
        ifree = W->arrayi ;
        k = 0 ;
        for (j = 0; j < ncol; j++)
        {
            if ( ib [j] == 0 ) ifree [k++] = j ; /* the variable is free */
        }
    }
    PASAINT const Ncol = (use_ifree) ? k : ncol ; /* Ncol = # free variables */

    /* We perform the iteration g_{k+1} = g_k - A_F'*inv(A_F A_F')A_F g_k
       until g_k no longer decays. */
    PASAFLOAT E = PASAINF ;
    /* The multiplier for the linear constraint is estimated by an iterative
       process. The sum of the dlambda's in the iteration is the final lambda.*/
    int const save_lambda = TRUE ;

    if ( save_lambda )
    {
        lambda = dlambda = Com->lambda ;
    }
    else
    {
        dlambda = W->arrayd+ncol ; /* use work array of pproj */
    }

    if ( (location == PASA_CG_DESCENT) && Com->pasaparm->use_QR )
    {
        PASAFLOAT *Z = Com->Z ;
        PASAFLOAT *Zp = Z ;
        PASAFLOAT *Gproj ;
        if ( gproj != NULL ) Gproj = gproj ;
        else                 Gproj = W->arrayd+ncol ;
        pasa_initx (Gproj, PASAZERO, ncol) ;
        for (i = 0; i < Com->Zncol; i++)
        {
            t = pasa_dot (g, Zp, ncol) ;
            pasa_daxpy (Gproj, Zp, t, ncol) ;
            Zp = Zp+ncol ;
        }
        if ( LocalError ) Com->e = pasa_sup_normx (Gproj, ncol) ;

#if 0
        /* check that Gproj lies in the null space of A */
        PASAFLOAT *Ad = W->arrayd+2*ncol ;
        pasa_initx (Ad, PASAZERO, nrow) ;
        for (i = 0; i < Ncol; i++)
        {
            j = (use_ifree) ? ifree [i] : i ;
            PASAFLOAT const T = Gproj [j] ;
            if ( T != PASAZERO )
            {
                k = Ap [j] ;
                PASAINT const L = k + Anz [j] ;
                for (; k < L; k++)
                {
                    Ad [Ai [k]] += T*Ax [k] ;
                }
            }
        }
        printf ("supnorm of d(Gproj): %e supnorm of Ad: %e\n",
                pasa_sup_normx (Gproj, ncol), pasa_sup_normx (Ad, nrow)) ;

        if ( gproj != NULL )
        {
            PASAFLOAT *dGproj = W->arrayd+2*ncol ; /* use work array of pproj */
            pasa_copyx (dGproj, Gproj, ncol) ;
            pasa_daxpy (dGproj, G, -PASAONE, ncol) ;
            u = pasa_sup_normx (dGproj, ncol) ;
            printf ("absolute error in gproj: %e, relative: %e\n", u, u/t) ;
            /* use the true projected gradient in CG */
            pasa_copyx (G, Gproj, ncol) ;
        }
#endif
    }
    else
    {
        int J = 0 ; /* first iteration */
        while ( E > Com->testtol )
        {
            pasa_initx (dlambda, PASAZERO, nrow) ;
            /* compute A_F g_k and store in dlambda */
            for (i = 0; i < Ncol; i++)
            {
                j = (use_ifree) ? ifree [i] : i ;
                PASAFLOAT const T = G [j] ;
                if ( T != PASAZERO )
                {
                    k = Ap [j] ;
                    PASAINT const L = k + Anz [j] ;
                    for (; k < L; k++)
                    {
                        dlambda [Ai [k]] += T*Ax [k] ;
                    }
                }
            }
            /* compute inv(A_F A_F')dlambda and store result in dlambda */
            pproj_lsol (W->L, dlambda, RLinkUp [nrow], nrow, RLinkUp) ;
            k = RLinkUp [nrow] ;
            /* momentarily set the initial RLinkDn to -1, this simplifies
               indexing in dltsolve */
            RLinkDn [k] = -1 ;
            l = RLinkDn [nrow] ;
            pproj_dltsol (W->L, dlambda, dlambda, l, k, RLinkDn) ;
            RLinkDn [k] = nrow ; /* restore RLinkDn */
 
            /* compute g_{k+1} = g_k - A_F' dlambda */
            e = PASAZERO ;
            for (i = 0; i < Ncol; i++)
            {
                j = (use_ifree) ? ifree [i] : i ;
                t = PASAZERO ;
                k = Ap [j] ;
                PASAINT const L = k + Anz [j] ;
                for (; k < L; k++)
                {
                    t += Ax [k]*dlambda [Ai [k]] ;
                }
                PASAFLOAT const T = G [j] - t ;
                G [j] = T ;
                if ( fabs (T) > e ) e = fabs (T) ;
            }
            if ( PrintLevel >= 2 )
            {
                printf ("LocalError: %e prior estimate: %e testtol: %e\n",
                         e, E, Com->testtol) ;
            }
            if ( e > 0.5*E )
            {
                break ;
            }
            if ( J == 0 )
            {
                dlambda = W->arrayd+ncol ; /* use work array of pproj */
                J = 1 ; /* after the first iteration */
            }
            else /* J = 1 */
            {
                if ( save_lambda )
                {
                    pasa_step (lambda, lambda, dlambda, PASAONE, nrow) ;
                }
            }
            E = e ;
        }

        /* gproj_exists only in active GP or in CG, where Ncol = ncol */
        if ( gproj != NULL ) pasa_copyx (gproj, G, Ncol) ;
    
        if ( LocalError ) Com->e = e ;
    }

#ifndef NDEBUG
    /* check that the projected gradient lies in the null space */
    if ( (location == PASA_CG_DESCENT) && (gproj != NULL) )
    {
        PASAFLOAT *v ;
        status = PASA_OK ;
        v = (PASAFLOAT *) pasa_malloc (&status, nrow, sizeof (PASAFLOAT)) ;
/* printf ("matrix elements in check computation\n") ;*/
        t = CGZERO ;
        pasa_initx (v, PASAZERO, nrow) ;
        for (j = 0; j < ncol; j++)
        {
            e = gproj [j] ;
            if ( e != PASAZERO )
            {
                k = Ap [j] ;
                l = k + Anz [j] ;
                for (; k < l; k++)
                {
                    v [Ai [k]] += e*Ax [k] ;
/* printf ("i: %i j: %i Ax: %e\n", Ai [k], j, Ax [k]) ;*/
                    t = PASAMAX (t, fabs (Ax [k])) ;
                }
            }
        }
        printf ("projection error: %e maxAx: %e\n", pasa_sup_normx (v, nrow),t);
        pasa_free (v) ;
    }
#endif
}

/* ==========================================================================
   === pasa_project =========================================================
   ==========================================================================
    Project a given point y onto the polyhedren, storing the projection in x
   ========================================================================== */
int pasa_project /* return:
                            PASA_OK
                            PASA_POLYHEDRON_INFEASIBLE
                            PASA_INVALID_LINEAR_CONSTRAINT_BOUNDS
                            PASA_OUT_OF_MEMORY_IN_PPROJ
                            PASA_SSOR_ASCENT_FAILURE_IN_PPROJ
                            PASA_SSOR_ITERATIONS_EXCEED_MAX_ITS_IN_PPROJ */
(
    PASAFLOAT    *x, /* projection of y onto the polyhedron */
    PASAFLOAT    *y, /* point to project onto the polyhedron */
    PASAcom    *Com  /* common data for PASA */
)
{
    int start_guess, status ;
    PASAparm *Parm ;
    PPparm   *parm ;
    PPdata   *ppdata ;
    Com->pasastat->nproject++ ;

    if ( Com->use_napheap == TRUE )
    {
        NAPdata *napdata ;
        napdata = Com->pasadata->napdata ;

        /* update the napdata structure */
        napdata->x      = x ;
        napdata->lambda = -Com->nap_lambda ;
        napdata->n      = Com->nf ;
        napdata->c      = y ;
        napdata->blo    = Com->nap_bl ;
        napdata->bhi    = Com->nap_bu ;

        status = napheap (napdata) ;

        /* flip sign of lambda for consistency with rest of code */
        Com->nap_lambda = -napdata->lambda ;

        if ( Com->initial_projection == TRUE )
        {
            napdata->Parm->use_prior_data = TRUE ;
            Com->initial_projection = FALSE ;
        }
        /* map napheap status to pasa status */
        if ( status == NAPHEAP_STATUS_OK )
        {
            status = PASA_OK ;
        }
        return (status) ;
    }

    const int loExists = Com->loExists ;
    const int hiExists = Com->hiExists ;

    Parm = Com->pasaparm ;
    parm = Com->pprojparm ;

    /* If A does not exist, then the projection is simply trunction at bounds.*/
    if ( Com->Aexists == FALSE )
    {
        PASAINT j ;
        PASAINT const ncol = Com->nf ;
        for (j = 0; j < ncol; j++)
        {
            if ( loExists && (y [j] < Com->lo [j]) )
            {
                x [j] = Com->lo [j] ;
            }
            else if ( hiExists && (y [j] > Com->hi [j]) )
            {
                x [j] = Com->hi [j] ;
            }
            else
            {
                x [j] = y [j] ;
            }
        }
        return (PASA_OK) ;
    }

    if ( Com->initial_projection == TRUE )
    {
        /* For the initial projection, the only option is 1 if we are supposed
           to use the provided lambda, or 0 (start from initial guess = 0)
           if no initial lambda is provided. Hence, we save the value of
           start_guess in the parameter structure, reset its value to be 0 or 1,
           and then below, restore the original value of start_guess. */

        start_guess = parm->start_guess ;
        if ( Parm->use_lambda == TRUE )
        {
            parm->start_guess = 3 ;
        }
        else
        {
            parm->start_guess = 0 ;
        }
    }
#ifndef NOPPROJ

    ppdata = Com->pasadata->ppdata ;
    ppdata->x    = x ; /* projection on polyhedron */
    ppdata->y    = y ; /* point to project on polyhedron */
    ppdata->ncol = Com->nf ;

    status = pproj (ppdata) ;

#endif

    /* if the debugger is on, then check the accuracy of x */
#ifndef NDEBUG
#ifndef NOPPROJ
    {
        PPFLOAT errb, errx, absAx ;
        PPdata *ppdataDEBUG ;

        ppdataDEBUG = Com->ppdataDEBUG ;
        ppdataDEBUG->x    = x ;
        ppdataDEBUG->y    = y ;
        ppdataDEBUG->ncol = Com->nf ;
        pproj_KKTerror (&errb, &errx, &absAx, ppdataDEBUG) ;
        printf ("----------- Projection Error Statistics ------------\n") ;
        printf ("specified tolerance ..................... %e\n", Com->GPtol) ;
        printf ("rel. sup norm of dual function gradient . %e\n", errb);
        printf ("relative diff between x & dual minimizer  %e\n", errx) ;
        printf ("absAx ................................... %e\n", absAx) ;
    }
#endif
#endif
    if ( Com->initial_projection == TRUE )
    {
        Com->initial_projection = FALSE ;

        /* save the priordata in the ppdata structure */
        Com->ppcom = ppdata->priordata ;

        /* restore original value for start_guess */
        parm->start_guess = start_guess ;
    }

    /* map pproj status to pasa status */
    if ( status == PPROJ_SOLUTION_FOUND )
    {
        status = PASA_OK ;
    }
    else if ( status == PPROJ_ERROR_DECAY_STAGNATES )
    {
        if ((Com->pprojstat != NULL)&&(Com->pprojstat->errdual > Com->grad_tol))
        {
            status = PASA_POLYHEDRON_INFEASIBLE ;
        }
        else /* the weaker tolerance for the problem was satisfied, ok */
        {
            status = PASA_OK ;
        }
    }
    else if ( status == PPROJ_FACTORIZATION_FAILS_IN_CHOLMOD )
    {
        status = PASA_FACTORIZATION_FAILS_IN_CHOLMOD ;
    }
    else if ( status == PPROJ_INTEGER_OVERFLOW_IN_CHOLMOD )
    {
        status = PASA_INTEGER_OVERFLOW_IN_CHOLMOD ;
    }
    return (status) ;
}
/* ==========================================================================
   === pasa_project1 ========================================================
   ==========================================================================
    Project a given point y onto the polyhedren, storing the projection in x.
    We write x = y + dx and compute dx of minimal 2-norm such that x is
    feasible. This is the same as project2 except that project2 does not
    make the change of variables x = y + dx.
   ========================================================================== */
int pasa_project1 /* return:
                              PASA_OK
                              PASA_POLYHEDRON_INFEASIBLE
                              PASA_INVALID_LINEAR_CONSTRAINT_BOUNDS
                              PASA_OUT_OF_MEMORY_IN_PPROJ
                              PASA_SSOR_ASCENT_FAILURE_IN_PPROJ
                              PASA_SSOR_ITERATIONS_EXCEED_MAX_ITS_IN_PPROJ */
(
    PASAFLOAT       *x, /* projection of y onto the polyhedron */
    PASAFLOAT       *y, /* point to project onto the polyhedron */
    PASAcom       *Com  /* common data for PASA */
)
{
    int status ;
    PASAINT i, j, k, p ;

    int PrintLevel = Com->pasaparm->PrintLevel ;
    if ( PrintLevel >= 2 )
    {
        printf ("project1\n") ;
    }
    if ( Com->use_napheap == TRUE )
    {
        status = pasa_project (x, y, Com) ;
        return (status) ;
    }

    PPprob *Prob = Com->ppcom->Prob ;
    PASAINT const ncol = Prob->ncol ;

    /* bounds for dx are lo - x <= dx <= hi - x */
    pasa_step (Prob->lo, Com->lo, y, -PASAONE, ncol) ;
    pasa_step (Prob->hi, Com->hi, y, -PASAONE, ncol) ;

    /* linear constraints on dx are bl - A*x <= A*dx <= bu - A*x */
    PASAINT   *Ap = Prob->Ap ;
    PASAINT   *Ai = Prob->Ai ;
    PASAFLOAT *Ax = Prob->Ax ;
    PASAINT   *ir = Com->ppcom->Work->ir ; /* ir [i] = 0 for active row */
    PASAINT const nrow = Prob->nrow ;
    PASAINT const   ni = Prob->ni ;
    PASAFLOAT     *rhs = Com->ppcom->Work->arrayd ; /* work space from pproj */
    PASAFLOAT   *absAx = rhs+nrow ;

    ASSERT (ncol == Com->nf) ;

    pasa_initx (absAx, PASAZERO, nrow) ;
    pasa_initx (rhs, PASAZERO, nrow) ;
    k = p = 0 ; /* p used to index over A and k counts inequality constraints */

    /* compute A*y and store in rhs and absAx */
    for (j = 0; j < ncol; j++)
    {
        PASAFLOAT const Xj = y [j] ;
        PASAINT const q = Ap [j+1] ;
        for (; p < q; p++)
        {
            PASAFLOAT const AXj = Ax [p]*Xj ;
            PASAINT const row = Ai [p] ;
            rhs [row] += AXj ;
            /* only active rows included in absAx */
            if ( ir [row] <= ni ) absAx [row] += fabs (AXj) ;
        }
    }

    /* subtract rhs from b (equality constraints), bl, and bu */
    for (i = 0; i < nrow; i++) /* i corresponds to permuted constraints */
    {
        if ( ir [i] == 0 )     /* equality constraint */
        {
            Prob->b [i] = Com->b [i] - rhs [i] ;
        }
        else                   /* strict inequality constraint */
        {
            k++ ;
            Prob->bl [k] = Com->bl [k] - rhs [i] ;
            Prob->bu [k] = Com->bu [k] - rhs [i] ;
        }
    }
    Com->ppcom->Work->absAxk = pasa_sup_normx (absAx, nrow) ;
    ASSERT (k == ni) ;

#ifndef NDEBUG
    /* when debugging, also save the problem data in the debug structure */
    PASAFLOAT *bl2 = (PASAFLOAT *) pasa_malloc (&status, nrow, sizeof (double));
    PASAFLOAT *bu2 = (PASAFLOAT *) pasa_malloc (&status, nrow, sizeof (double));
    Com->ppdataDEBUG->bl = bl2 ;
    Com->ppdataDEBUG->bu = bu2 ;
    Com->ppdataDEBUG->lo = Prob->lo ;
    Com->ppdataDEBUG->hi = Prob->hi ;
    Com->ppdataDEBUG->Ap = Ap ;
    Com->ppdataDEBUG->Ai = Ai ;
    Com->ppdataDEBUG->Ax = Ax ;

    j = 0 ;
    for (i = 0; i < nrow; i++) /* i corresponds to the permuted constraints */
    {
        if ( ir [i] == 0 )     /* equality constraint */
        {
            bl2 [i] = Prob->b [i] ;
            bu2 [i] = Prob->b [i] ;
        }
        else                   /* strict inequality constraint */
        {
            j++ ;
            bl2 [i] = Prob->bl [j] ;
            bu2 [i] = Prob->bu [j] ;
        }
    }
#endif

    PPFLOAT *dx = Com->d ;
    /* use gnew = 0 for point to project after the change of variables */
    pasa_initx (Com->gnew, PASAZERO, ncol) ;
    /* ----------------------------------------- */
    status = pasa_project (dx, Com->gnew, Com) ;
    /* ----------------------------------------- */

    /* x = y + dx */
    pasa_step (x, y, dx, PASAONE, ncol) ;


#ifndef NDEBUG
    /* x and y should be the same to within rounding errors */
    PASAFLOAT normx = pasa_sup_normx (x, ncol) ;
    PASAFLOAT normy = pasa_sup_normx (y, ncol) ;
    PASAFLOAT normdx= pasa_sup_normx (dx, ncol) ;
    printf ("project1 norms dx: %e, x: %e, y: %e\n", normdx, normx, normy) ;
    pasa_free (bl2) ;
    pasa_free (bu2) ;
#endif
    return (status) ;
}

/* ==========================================================================
   === pasa_project2 ========================================================
   ==========================================================================
    Project a given point y onto the polyhedren, storing the projection in x.
    project2 and project are the same except that in project2, we also extract
    the bl2 and bu2 arrays needed for checking the KKT conditions. In project,
    the extract was done earlier in either the base code or in gradproj_step.
   ========================================================================== */
int pasa_project2 /* return:
                              PASA_OK
                              PASA_POLYHEDRON_INFEASIBLE
                              PASA_INVALID_LINEAR_CONSTRAINT_BOUNDS
                              PASA_OUT_OF_MEMORY_IN_PPROJ
                              PASA_SSOR_ASCENT_FAILURE_IN_PPROJ
                              PASA_SSOR_ITERATIONS_EXCEED_MAX_ITS_IN_PPROJ */
(
    PASAFLOAT       *x, /* projection of y onto the polyhedron */
    PASAFLOAT       *y, /* point to project onto the polyhedron */
    PASAcom       *Com  /* common data for PASA */
)
{
    int PrintLevel, status ;
    PASAINT i, j, *ir ;
    PASAFLOAT normx, normy, s, *b, *bl, *bu, *lo, *hi ;
    PPprob *Prob ;

    PrintLevel = Com->pasaparm->PrintLevel ;
    if ( PrintLevel >= 2 )
    {
        printf ("project2\n") ;
    }
    if ( Com->use_napheap == TRUE )
    {
        status = pasa_project (x, y, Com) ;
        return (status) ;
    }

    /* we project onto the original polyhedron, not the translated one,
       hence, we save the data for the original polyhedron, and then
       restore it later */
    Prob = Com->ppcom->Prob ;
    b  = Prob->b ;
    bl = Prob->bl ;
    bu = Prob->bu ;
    lo = Prob->lo ;
    hi = Prob->hi ;
    /* now save data for the original, untranslated polyhedron */
    Prob->lo = Com->lo ;
    Prob->hi = Com->hi ;
    Prob->bl = Com->bl ;
    Prob->bu = Com->bu ;
    Prob->b  = Com->b ;
    /* since there is no translation, set absAxk = 0 */
    Com->ppcom->Work->absAxk = PASAZERO ;

#ifndef NDEBUG
    PASAINT const nrow = Com->nrow ;
    status = PASA_OK ;
    PASAFLOAT *bl2 = (PASAFLOAT *) pasa_malloc (&status, nrow, sizeof (double));
    PASAFLOAT *bu2 = (PASAFLOAT *) pasa_malloc (&status, nrow, sizeof (double));
    Com->ppdataDEBUG->bl = bl2 ;
    Com->ppdataDEBUG->bu = bu2 ;
    Com->ppdataDEBUG->lo = Com->lo ;
    Com->ppdataDEBUG->hi = Com->hi ;
    Com->ppdataDEBUG->Ap = Com->Ap ;
    Com->ppdataDEBUG->Ai = Com->Ai ;
    Com->ppdataDEBUG->Ax = Com->Ax ;
    ir = Com->ppcom->Work->ir ;
    j = 0 ;

    for (i = 0; i < nrow; i++) /* i corresponds to the permuted constraints */
    {
        if ( ir [i] == 0 )     /* equality constraint */
        {
            bl2 [i] = Com->b [i] ;
            bu2 [i] = Com->b [i] ;
        }
        else                   /* strict inequality constraint */
        {
            j++ ;
            bl2 [i] = Com->bl [j] ;
            bu2 [i] = Com->bu [j] ;
        }
    }
#endif
    /* ----------------------------------------- */
    status = pasa_project (x, y, Com) ;
    /* ----------------------------------------- */

#ifndef NDEBUG
    /* x and y should be the same to within rounding errors */
    PASAINT const ncol = Com->ncol ;
    normx = pasa_sup_normx (x, ncol) ;
    normy = pasa_sup_normx (y, ncol) ;
    if ( (normx == PASAZERO) && (normy == PASAZERO) )
    {
        normx = PASAONE ;
    }
    s = PASAZERO ;
    for (j = 0; j < ncol; j++)
    {
        s = PASAMAX (s, fabs (x [j] - y [j])) ;
    }
    printf ("project2, norm(x-y): %e, normx: %e, normy: %e\n",
             s, normx, normy) ;
    pasa_free (bl2) ;
    pasa_free (bu2) ;
#endif
    Prob->lo = lo ;
    Prob->hi = hi ;
    Prob->bl = bl ;
    Prob->bu = bu ;
    Prob->b  = b  ;
    return (status) ;
}

/* =========================================================================
   === pasa_compress_matrix ================================================
   =========================================================================
    Compress a matrix by retaining only columns for which drop [j] = 0.
    The original matrix is overwritten by the compressed matrix. The
    number of columns in the compressed matrix is returned.
   ========================================================================= */
PASAINT pasa_compress_matrix
(
    PASAINT    *Ap, /* matrix column pointers */
    PASAINT    *Ai, /* matrix row indices */
    PASAFLOAT  *Ax, /* matrix numerical values */
    PASAINT   ncol, /* number of cols in A */
    PASAINT  *drop  /* drop columns for which drop [j] != 0 */
)
{
    PASAINT cols, j, l, p, q ;
    /* find the index of the first dropped column */
    cols = ncol ;
    for (j = 0; j < ncol; j++)
    {
        if ( drop [j] != 0 )
        {
            cols = j ;
            break ;
        }
    }
    /* if nothing drops, then return */
    if ( cols == ncol )
    {
        return (ncol) ;
    }
    /* cols stores the number of remaining columns in A */
    l = Ap [cols] ; /* start of storage for this first dropped column */
    for (j++; j < ncol; j++)
    {
        if ( drop [j] == 0 ) /* retained column */
        {
            p = Ap [j] ;
            q = Ap [j+1] ;
            Ap [cols+1] = Ap [cols] + q - p ;
            cols++ ;
            for (; p < q; p++)
            {
                Ai [l] = Ai [p] ;
                Ax [l] = Ax [p] ;
                l++ ;
            }
        }
    }
    return (cols) ;
}

/* =========================================================================
   === pasa_hadamard =======================================================
   =========================================================================
    Store the Hadamard product of x and y*c in x
   ========================================================================= */
void pasa_hadamard
(
    PASAFLOAT       *x, /* x = x.*(y*c) */
    PASAFLOAT const *y,
    PASAFLOAT const  c,
    PASAINT   const  n  /* length of vectors */
)
{
    PASAINT j, n5 ;
    n5 = n % 5 ;        /* n5 = n mod 5 */
    if ( c == PASAONE )
    {
        for (j = 0; j < n5; j++)
        {
            x [j] *= y [j] ;
        }
        for (; j < n; )
        {
            x [j] *= y [j] ; j++ ;
            x [j] *= y [j] ; j++ ;
            x [j] *= y [j] ; j++ ;
            x [j] *= y [j] ; j++ ;
            x [j] *= y [j] ; j++ ;
        }
    }
    else /* c not equal to one */
    {
        for (j = 0; j < n5; j++)
        {
            x [j] *= c*y [j] ;
        }
        for (; j < n; )
        {
            x [j] *= c*y [j] ; j++ ;
            x [j] *= c*y [j] ; j++ ;
            x [j] *= c*y [j] ; j++ ;
            x [j] *= c*y [j] ; j++ ;
            x [j] *= c*y [j] ; j++ ;
        }
    }
}

/* =========================================================================
   === pasa_convert_to_user ===============================================
   =========================================================================
   convert from pproj's order of variables to user's order
   ========================================================================= */
void pasa_convert_to_user
(
    PASAFLOAT       *x, /* x in user order */
    PASAFLOAT const *y, /* y in pproj order */
    PASAINT     *ifree, /* ifree [j] = element of x associated with y [j] */
    PASAINT   const  n  /* dimension */
)
{
    PASAINT j ;
    for (j = 0; j < n; j++)
    {
        x [ifree [j]] = y [j] ;
    }
}

/* =========================================================================
   === pasa_convert_to_pproj ==============================================
   =========================================================================
   convert from user's order of variables to pproj's order
   ========================================================================= */
void pasa_convert_to_pproj
(
    PASAFLOAT   *x, /* x in pproj order */
    PASAFLOAT   *y, /* y in user order */
    PASAINT *ifree, /* ifree [j] = element of y associated with x [j] */
    PASAINT      n  /* dimension */
)
{
    PASAINT j ;
    for (j = 0; j < n; j++)
    {
        x [j] = y [ifree [j]] ;
    }
}

/* ==========================================================================
   === pasa_evaluate ========================================================
   ==========================================================================
    Evaluate function, gradient, or both
   ========================================================================== */
int pasa_evaluate
(
    PASAFLOAT alpha_good, /* a value of alpha for which function is finite */
    PASAFLOAT     *Alpha, /* stepsize along the search direction */
    PASAcom         *Com, /* PASAcom structure */
    char           *what  /* f = function, g = gradient, fg =function+gradient*/
)
{
    PASAFLOAT alpha, df, t, *d, *f, *g, *gtemp, *Qd, *x ;
    PASAINT i, j, n, nf, *nfunc, *ngrad, *order ;
    PASAstat *Stat ;
    PASAparm *Parm ;
    int location ;
    alpha = *Alpha ;
    Parm = Com->pasaparm ;
    Stat = Com->pasastat ;
    location = Com->location ; /* PASA_BASE_CODE (main code)
                                  PASA_GRAD_PROJ (gradient projection)
                                  PASA_ACTIVE_GP (active set grad proj)
                                  PASA_CG_DESCENT (conjugate gradient) */
    order = NULL ; /* do not reorder the variables */
    if ( (location == PASA_ACTIVE_GP) || (location == PASA_CG_DESCENT) )
    {
        if ( Com->order_with_ifree == TRUE )
        {
            order = Com->ifree ;
        }
        else if ( Com->order_with_colperm == TRUE )
        {
            order = Com->colperm ;
        }
    }
    else if ( location == PASA_GRAD_PROJ )
    {
        if ( Com->order_with_colperm )
        {
            order = Com->colperm ;
        }
    }
    int const QP = Com->QP ;
    d = Com->d ;
    Qd = Com->Qd ;
    n = Com->ucol ; /* user specified number of variables */
    nf = Com->nf ;

    /* The point where the function is evaluated depends on alpha.
       alpha < 0 => evaluate at xnew
       alpha = 0 => evaluate at x (the first evaluation of the objective)
       otherwise, evaluate at xnew = x + alpha*d. If the function is
       evaluated at xnew, then the gradient and objective value are
       stored in gnew and fnew respectively. Similarly, if the function
       is evaluated at x, then the gradient and objective values are stored
       in g and f respectively. */
    if ( alpha != PASAZERO ) /* CG, GP, AGP */
    {
        x = Com->xnew ;
        g = Com->gnew ;
        if ( location == PASA_CG_DESCENT )
        {
            f = &Com->f ; /* no fnew in CG */
        }
        else
        {
            f =&Com->fnew ;
        }
    }
    else /* alpha = 0, initial function evaluation */
    {
        if ( Com->need_userx )
        {
            x = Com->userx ;
            g = Com->userg ;
        }
        else
        {
            x = Com->x ;
            g = Com->g ;
        }
        
        f =&Com->f ;
    }

    /* if alpha > 0, then compute the step xnew = x + alpha*d */
    if ( alpha > PASAZERO ) /* GP, AGP, CG but not expand step */
    {
        pasa_step (Com->xnew, Com->x, d, alpha, nf) ;
        if ( (location == PASA_CG_DESCENT) && (alpha == Com->maxbndstep) )
        {
            /* component of x hits bound */
            if ( Com->maxbndindex < 0 )
            {
                j = -(Com->maxbndindex + 1) ;
                Com->xnew [j] = Com->lo [j] ;
            }
            else
            {
                j = Com->maxbndindex - 1 ;
                Com->xnew [j] = Com->hi [j] ;
            }
        }
    }

    gtemp = g ; /* default */
    /* The very first time that the function is evaluated (that is, when
       alpha = 0), we employ the user's ordering of the variables.
       Subsequently, x in the code corresponds to pproj's ordering
       if the A matrix exists. Hence, before evaluating the function,
       we need to change from pproj's ordering to the user's ordering.
       If we come from grad_proj and there is no A,
       then there is no permutation.  */
    if ( (alpha != PASAZERO) && (QP == FALSE) )
    {
        if ( order != NULL )
        {
            for (j = 0; j < nf; j++)
            {
                Com->userx [order [j]] = x [j] ;
            }
            x = Com->userx ;
            /* if gradient is evaluated, store it's value temporarily in
               userg since we need to convert from user ordering to pproj
               ordering */
            gtemp = Com->userg ;
        }
        /* else coming from grad_proj with only bound constraints, directly
           store gradient in g */
    }

    /* Determine which counter to increment when counting the number of
       function and gradient evaluations. */
    if ( location == PASA_BASE_CODE )
    {
        nfunc = &Stat->mcnf ;
        ngrad = &Stat->mcng ;
    }
    else if ( location == PASA_GRAD_PROJ ) /* gradient projection */
    {
        nfunc = &Stat->gpnf ;
        ngrad = &Stat->gpng ;
    }
    else if ( location == PASA_ACTIVE_GP )
    {
        nfunc = &Stat->agpnf ;
        ngrad = &Stat->agpng ;
    }
    else                      /* conjugate gradients */
    {
        nfunc = &Com->cgstat->nfunc ;
        ngrad = &Com->cgstat->ngrad ;
    }

    if ( QP ) /* Hessian = Q */
    {
        (*ngrad)++ ;
        if ( alpha == PASAZERO ) /* initial evaluation of func & grad */
        {
            if ( Com->hprod_status )
            {
                pasa_builtin_hprod (g, x, NULL, n, n,
                                    Com->Hp, Com->Hi, Com->Hx) ;
            }
            else
            {
                Com->hprod (g, x, NULL, n, n) ;         /* g = Q*x */
            }
            if ( Com->QPshift ) /* g = g + QPshift*x */
            {
                pasa_step (g, g, x, Com->QPshift, n) ;
            }
            *f = 0.5*pasa_dot (g, x, n) ;               /* f = .5x'Qx */
            if ( Com->userc != NULL )
            {
                *f += pasa_dot(x, Com->userc, n) ;/* f = .5*x'*Qx + c'*x*/
                /* store gradient in userg */
                pasa_step (g, g, Com->userc, PASAONE, n) ; /* g = Qx+c */
            }
            /* else c is taken as zero so there is no effect on objective value
               or the gradient */
            return (PASA_OK) ;
        }
        if ( Com->hprod_status )
        {
            pasa_builtin_hprod (Qd, d, order, n, nf,
                                    Com->Hp, Com->Hi, Com->Hx) ;
        }
        else
        {
            Com->hprod (Qd, d, order, n, nf) ; /* Qd is in user ordering */
        }
        /* if the Hessian is regularized, adjust the gradient */
        if ( Com->QPshift )
        {
            if ( order != NULL )
            {
                for (j = 0; j < nf; j++)
                {
                    i = order [j] ;
                    Qd [i] += Com->QPshift*d [j] ;
                }
            }
            else
            {
                pasa_daxpy (Qd, Com->d, Com->QPshift, n) ;
            }
        }
        if ( alpha == PASAONE ) /* alpha = -1 when evaluation in CG */
        {
            if ( order == NULL )
            {
                pasa_step (g, Com->gtot, Qd, alpha, nf) ;
            }
            else
            {
                for (j = 0; j < nf; j++)
                {
                    i = order [j] ;
                    g [j] = Com->gtot [i] + Qd [i] ;
                }
            }
        }
        return (PASA_OK) ; /* immediately return when objective is quadratic */
    }
    else if ( !strcmp (what, "fg") ) /* compute function and grad values */
    {
        (*nfunc)++ ;
        (*ngrad)++ ;
        if ( Com->valgrad != NULL ) /* user provided valgrad */
        {
            /* evaluate func and grad together */
            Com->valgrad (f, gtemp, x, n) ;
        }
        else /* evaluate func and grad separately */
        {
            Com->value (f, x, n) ;    /* evaluate func */
            Com->grad (gtemp, x, n) ; /* evaluate grad */
        }

        /* direct return if evaluating at starting point */
        if ( alpha == PASAZERO )
        {
            return (PASA_OK) ;
        }
        /* for CG we need to compute the derivative in the search direction */
        if ( location == PASA_CG_DESCENT )
        {
            /* convert from user variables to variables in pasa */
            if ( order != NULL )
            {
                pasa_convert_to_pproj (g, gtemp, order, nf) ;
            }
            df = Com->df = pasa_dot (g, d, nf) ;
            /* if an infinite function value or an nan was encountered,
               then we attempt to find a finite function value */
            if ( (*f != *f) || (*f >= PASAINF) || (*f <= -PASAINF) ||
                 (df != df) || (df >= PASAINF) || (df <= -PASAINF) )
            {
                if ( Com->need_userx ) x = Com->userx ;
                t = Parm->infdecay ;
                for (i = 0; i < Parm->ninf_tries; i++)
                {
                    nfunc++ ;
                    ngrad++ ;
                    alpha = alpha_good + t*(alpha - alpha_good) ;
                    t *= Parm->infdecay_rate ;
                    pasa_step (Com->xnew, Com->x, d, alpha, nf) ;
                    if ( order != NULL )
                    {
                        for (j = 0; j < nf; j++)
                        {
                            Com->userx [order [j]] = Com->xnew [j] ;
                        }
                    }
                    if ( Com->valgrad != NULL )
                    {
                        Com->valgrad (f, gtemp, x, n) ;
                    }
                    else
                    {
                        Com->grad  (gtemp, x, n) ;
                        Com->value (f, x, n) ;
                    }
                    /* convert from user variables to variables in pasa */
                    if ( order != NULL )
                    {
                        pasa_convert_to_pproj (g, gtemp, order, nf) ;
                    }
                    df = Com->df = cg_dot (g, d, nf) ;
                    if ( (*f == *f) && (*f < PASAINF) && (*f > -PASAINF) &&
                         (df == df) && (df < PASAINF) && (df > -PASAINF) )
                    {
                        break ;
                    }
                }
                if ( i == Parm->ninf_tries )
                {
                    return (PASA_FUNCTION_NAN_OR_INF) ;
                }
            }
            if ( Com->use_penalty == TRUE )
            {
                Com->f_orig = *f ;
                *f += Com->fp + alpha*(Com->dp + 0.5*alpha*Com->Ad2) ;
                Com->df += Com->dp + alpha*Com->Ad2 ;
            }
        }
        else /* not in cg_descent, no need to compute directional derivative */
        {
            /* if an infinite function value or an nan was encountered,
               then we attempt to find a finite function value */
            if ( (*f != *f) || (*f >= PASAINF) || (*f <= -PASAINF) )
            {
                t = Parm->infdecay ;
                if ( Com->need_userx ) x = Com->userx ;
                for (i = 0; i < Parm->ninf_tries; i++)
                {
                    nfunc++ ;
                    ngrad++ ;
                    alpha = alpha_good + t*(alpha - alpha_good) ;
                    t *= Parm->infdecay_rate ;
                    pasa_step (Com->xnew, Com->x, d, alpha, nf) ;
                    if ( order != NULL )
                    {
                        for (j = 0; j < nf; j++)
                        {
                            Com->userx [order [j]] = Com->xnew [j] ;
                        }
                    }
                    if ( Com->valgrad != NULL )
                    {
                        Com->valgrad (f, gtemp, x, n) ;
                    }
                    else
                    {
                        Com->grad  (gtemp, x, n) ;
                        Com->value (f, x, n) ;
                    }
                    if ( (*f == *f) && (*f < PASAINF) && (*f > -PASAINF) )
                    {
                        break ;
                    }
                }
                if ( i == Parm->ninf_tries )
                {
                    return (PASA_FUNCTION_NAN_OR_INF) ;
                }
            }
            /* convert from user variables to variables in pasa */
            if ( order != NULL )
            {
                pasa_convert_to_pproj (g, gtemp, order, nf) ;
            }
        }
        *Alpha = alpha ;
        return (PASA_OK) ;
    }
    else if ( !strcmp (what, "f") )   /* only compute function value */
    {
        (*nfunc)++ ;
        Com->value (f, x, n) ;
        /* if an infinite function value or an nan was encountered,
           then we attempt to find a finite function value */
        if ( (*f != *f) || (*f >= PASAINF) || (*f <= -PASAINF) )
        {
            if ( alpha < PASAZERO ) /* expansion step gave infinite value */
            {
                *f = PASAINF ;
                 return (PASA_OK) ;
            }
            t = Parm->infdecay ;
            if ( Com->need_userx ) x = Com->userx ;
            for (i = 0; i < Parm->ninf_tries; i++)
            {
                nfunc++ ;
                alpha = alpha_good + t*(alpha - alpha_good) ;
                t *= Parm->infdecay_rate ;
                pasa_step (Com->xnew, Com->x, d, alpha, nf) ;
                if ( order != NULL )
                {
                    for (j = 0; j < nf; j++)
                    {
                        Com->userx [order [j]] = Com->xnew [j] ;
                    }
                }
                Com->value (f, x, n) ;
                if ( (*f == *f) && (*f < PASAINF) && (*f > -PASAINF) )
                {
                    break ;
                }
            }
            if ( i == Parm->ninf_tries )
            {
                return (PASA_FUNCTION_NAN_OR_INF) ;
            }
            *Alpha = alpha ;
        }

        if ( (location == PASA_CG_DESCENT) &&  (Com->use_penalty == TRUE) )
        {
            Com->f_orig = *f ;
            *f += Com->fp + alpha*(Com->dp + 0.5*alpha*Com->Ad2) ;
        }
        return (PASA_OK) ;
    }
    else                              /* only compute gradient */
    {
        (*ngrad)++ ;
        Com->grad (gtemp, x, n) ;
        /* convert from user variables to variables in pasa */
        if ( order != NULL )
        {
            pasa_convert_to_pproj (g, gtemp, order, nf) ;
        }

        /* for CG we need to compute the derivative in the search direction */
        if ( location == PASA_CG_DESCENT )
        {
            df = Com->df = pasa_dot (g, d, nf) ;
            /* if an infinite function value or an nan was encountered,
               then we attempt to find a finite function value */
            if ( (df != df) || (df >= PASAINF) || (df <= -PASAINF) )
            {
                t = Parm->infdecay ;
                for (i = 0; i < Parm->ninf_tries; i++)
                {
                    ngrad++ ;
                    alpha = alpha_good + t*(alpha - alpha_good) ;
                    t *= Parm->infdecay_rate ;
                    pasa_step (Com->xnew, Com->x, d, alpha, nf) ;
                    if ( order != NULL )
                    {
                        for (j = 0; j < nf; j++)
                        {
                            Com->userx [order [j]] = Com->xnew [j] ;
                        }
                    }
                    if ( Com->need_userx )
                    {
                         Com->grad (gtemp, Com->userx, n) ;
                    }
                    else
                    {
                         Com->grad (gtemp, Com->x, n) ;
                    }
                    /* convert from user variables to variables in pasa */
                    if ( order != NULL )
                    {
                        pasa_convert_to_pproj (g, gtemp, order, nf) ;
                    }
                    df = Com->df = cg_dot (g, d, nf) ;
                    if ( (df == df) && (df < PASAINF) && (df > -PASAINF) )
                    {
                        break ;
                    }
                }
                if ( i == Parm->ninf_tries )
                {
                    return (PASA_FUNCTION_NAN_OR_INF) ;
                }
                *Alpha = alpha ;
            }
            if ( Com->use_penalty == TRUE )
            {
                Com->df += Com->dp + alpha*Com->Ad2 ;
            }
        }
        return (PASA_OK) ;
    }
}

/* ==========================================================================
   === pasa_evaluate_x ======================================================
   ==========================================================================
    Evaluate function and gradient at given x. For a quadratic, the prior
    function value is fold and the prior iterate xold
   ========================================================================== */
void pasa_evaluate_x
(
    PASAFLOAT    *x, /* x = evaluation point when f is not quadratic, or f is
                        a quadratic outside of CG. For a quadratic in CG,
                        x stores dx = xnew - xold */
    PASAFLOAT *xold, /* for a quadratic, prior evaluation point */
    PASAFLOAT *gold, /* gradient at xold */
    PASAFLOAT  fold, /* for a quadratic, prior objective value */
    PASAINT       n, /* dimension of x */
    PASAINT  *order, /* mapping between user ordering and internal ordering */
    PASAcom    *Com  /* PASAcom structure */
)
{
    PASAINT i, j ;
    PASAFLOAT dQd, *g ;
    int const location = Com->location ;
    if ( Com->QP ) /* the objective is quadratic */
    {
        PASAFLOAT *dx ;
        if ( location == PASA_CG_DESCENT )
        {
            Com->cgstat->ngrad++ ;
            dx = x ;
        }
        else
        {
            dx = Com->d ;
            /* dx = x - xold */
            pasa_step (dx, x, xold, -PASAONE, n) ;
            if ( Com->location == PASA_ACTIVE_GP )
            {
                Com->pasastat->agpng++ ;
            }
            else /* Com->location == PASA_GRAD_PROJ */
            {
                Com->pasastat->gpng++ ;
            }
        }

        /* multiply Hessian by dx and store result in Qd */
        if ( Com->hprod_status )
        {
            pasa_builtin_hprod (Com->Qd, dx, order, Com->ncol, n,
                                Com->Hp, Com->Hi, Com->Hx) ;
        }
        else
        {
            Com->hprod (Com->Qd, dx, order, Com->ncol, n) ;
        }

        /* if the Hessian is regularized, adjust gradient */
        if ( Com->QPshift )
        {
            if ( order == NULL )
            {
                pasa_daxpy (Com->Qd, dx, Com->QPshift, n) ;
            }
            else
            {
                for (j = 0; j < n; j++)
                {
                    i = order [j] ;
                    Com->Qd [i] += Com->QPshift*dx [j] ;
                }
            }
        }

        /* dQd = pasa_dot (d, Com->Qd, n), Qd is in user's coor */
        if ( order == NULL )
        {
            dQd = pasa_dot (dx, Com->Qd, n) ;
        }
        else
        {
            dQd = PASAZERO ;
            for (j = 0; j < n; j++)
            {
                dQd += dx [j]*Com->Qd [order [j]] ;
            }
        }
        Com->f = fold + 0.5*dQd + pasa_dot (gold, dx, n) ;

        /* compute the gradient at x */
        if ( order == NULL )
        {
            pasa_step (Com->g, Com->gtot, Com->Qd, PASAONE, n) ;
        }
        else
        {
            for (j = 0; j < n; j++)
            {
                i = order [j] ;
                Com->g [j] = Com->gtot [i] + Com->Qd [i] ;
            }
        }

        /* update total gradient */
        pasa_step (Com->gtot, Com->gtot, Com->Qd, PASAONE, Com->ncol) ;
    }
    else /* evaluate general nonlinear function at x */
    {
        if ( location == PASA_ACTIVE_GP )
        {
            Com->pasastat->agpnf++ ;
            Com->pasastat->agpng++ ;
        }
        else if ( location == PASA_GRAD_PROJ )
        {
            Com->pasastat->gpnf++ ;
            Com->pasastat->gpng++ ;
        }
        else /* location is CG */
        {
            Com->cgstat->nfunc++ ;
            Com->cgstat->ngrad++ ;
        }
        if ( order == NULL )
        {
            g = Com->g ;
        }
        else
        {
            /* copy x into user coordinates */
            for (j = 0; j < n; j++)
            {
                Com->userx [order [j]] = x [j] ;
            }
            x = Com->userx ;
            g = Com->userg ;
        }
        if ( Com->valgrad != NULL ) /* valgrad provided */
        {
            /* evaluate func and grad together */
            Com->valgrad (&(Com->f), g, x, Com->ucol) ;
        }
        else /* evaluate func and grad separately */
        {
            /* evaluate function */
            Com->value (&(Com->f), x, Com->ucol) ;
            /* evaluate gradient */
            Com->grad (g, x, Com->ucol) ;
        }
        /* extract gradient at x from userg */
        if ( order != NULL )
        {
            pasa_convert_to_pproj (Com->g, g, order, n);
        }
        /* if order = NULL, then the gradient was stored directly in Com->g */
    }
}

/* ==========================================================================
   === pasa_check_feas ======================================================
   ==========================================================================
   Check feasibility of current iterate
   ========================================================================== */
void pasa_check_feas
(
    PASAcom    *Com /* PASAcom structure */
)
{
    int location, status, use_napheap, use_pproj ;
    PASAFLOAT cost, fp, fl, lo_err, hi_err, bl_err, bu_err, g_err, s, t,
              *a, *Ax, *bl, *bu, *lo, *hi, *gpen, *Gpen, *temp, *x ;
    PASAINT i, j, nrow, ncol, ucol, nf, p, q, *Ai, *Ap, *order ;
    PASAparm *Parm ;
    int const loExists = Com->loExists ;
    int const hiExists = Com->hiExists ;
    int const QP       = Com->QP ;
    use_napheap = Com->use_napheap ;
    use_pproj = Com->use_pproj ;
    Parm = Com->pasaparm ;
    ncol = Com->ncol ;
    ucol = Com->ucol ;
    nrow = Com->nrow ;
    status = PASA_OK ;
    if ( Com->need_userx )
    {
        x = Com->userx ;
        printf ("x = Com->userx\n") ;
    }
    else
    {
        x = Com->x ;
        printf ("x = Com->x\n") ;
    }
    nf = Com->nf ;
    printf ("nf: %ld ncol: %ld ucol: %ld\n",
           (LONG) nf, (LONG) ncol, (LONG) ucol) ;

    location = Com->location ;
    order = NULL ; /* do not reorder the variables */
    if ( (location == PASA_ACTIVE_GP) || (location == PASA_CG_DESCENT) )
    {
        if ( Com->order_with_ifree == TRUE )
        {
            order = Com->ifree ;
        }
        else
        {
            if ( Com->order_with_colperm == TRUE )
            {
                order = Com->colperm ;
            }
        }
    }
    if ( (location == PASA_GRAD_PROJ) || (location == PASA_GRADPROJ_LP) )
    {
        if ( Com->order_with_colperm == TRUE )
        {
            order = Com->colperm ;
        }
        else
        {
            x = Com->x ;
        }
    }

    if ( order == NULL )
    {
        printf ("order = NULL, no ordering\n") ;
    }
    else if ( order == Com->ifree )
    {
        printf ("order with ifree\n") ;
    }
    else if ( order == Com->colperm )
    {
        printf ("order with colperm\n") ;
    }
    else
    {
        printf ("invalid ordering in pasa_check_feas\n") ;
        pasa_error (-1, __FILE__, __LINE__, "stop") ;
    }

    if ( order != NULL )
    {
        for (j = 0; j < nf; j++)
        {
            x [order [j]] = Com->x [j] ;
        }
    }
#ifndef NDEBUG
    lo = Com->LO ;
    hi = Com->HI ;
#endif
#if 0
for (i = 0; i < Com->nc; i++)
{
    j = Com->bound_cols [i] ;
    if ( j < 0 ) /* lower bound */
    {
        j = -(j+1) ;
        if ( lo [j] != x [j] )
        {
            printf ("userx[%i] = %e != lo = %e\n", j, x [j], lo [j]) ;
            pasa_error (-1, __FILE__, __LINE__, "stop") ;
        }
    }
    else
    {
        if ( hi [j] != x [j] )
        {
            printf ("userx[%i] = %e != hi = %e\n", j, x [j], hi [j]) ;
            pasa_error (-1, __FILE__, __LINE__, "stop") ;
        }
    }
}
#endif
    if ( nrow > 0 )
    {
#ifndef NDEBUG
        bl = Com->BL ;
        bu = Com->BU ;
        if ( use_pproj == TRUE )
        {
            Ax = Com->AX ;
            Ai = Com->AI ;
            Ap = Com->AP ;
        }
        else if ( use_napheap == TRUE )
        {
            a = Com->nap_a ;
        }
#endif
        temp = (PASAFLOAT *) pasa_malloc (&status, nrow, sizeof (PASAFLOAT)) ;
        pasa_initx (temp, PASAZERO, nrow) ;
    }
    lo_err = PASAZERO ;
    hi_err = PASAZERO ;
    p = 0 ;
    for (j = 0; j < ucol; j++)
    {
        s = x [j] ;
        if ( loExists && (s < lo [j]) )
        {
            /* printf ("j: %i x: %e lo: %e err: %e\n",
                        j, s, lo [j], lo [j]-s) ; */
            t = lo [j] - s ;
            printf ("lo error in userx[%ld]: %e x: %e lo: %e\n",
                                                       (LONG) j, t, s, lo [j]) ;
            lo_err = PASAMAX (t, lo_err) ;
        }
        if ( hiExists && (s > hi [j]) )
        {
            /* printf ("j: %i x: %e hi: %e err: %e\n",
                        j, s, hi [j], s - hi [j]) ; */
            t = s - hi [j] ;
            printf ("hi error in userx[%ld]: %e x: %e hi: %e\n",
                                                     (LONG) j, t, s, hi [j]) ;
            hi_err = PASAMAX (t, hi_err) ;
        }
        if ( nrow > 0 )
        {
            if ( use_pproj == TRUE )
            {
                q = Ap [j+1] ;
                for (; p < q; p++)
                {
                    temp [Ai [p]] += s*Ax [p] ;
                }
            }
            else if ( use_napheap == TRUE )
            {
                temp [0] += a [j]*s ;
            }
        }
    }
    bl_err = PASAZERO ;
    bu_err = PASAZERO ;
    for (i = 0; i < nrow; i++)
    {
        s = temp [i] ;
        if ( s < bl [i] )
        {
            t = bl [i] - s ;
            /* printf ("i: %i A*x: %e bl: %e err: %e\n",
                       i, s, bl [i], bl [i] - s) ;*/
            bl_err = PASAMAX (t, bl_err) ;
        }
        if ( s > bu [i] )
        {
            t = s - bu [i] ;
            /* printf ("i: %i A*x: %e bu: %e err: %e\n",
                        i, s, bu [i], s - bu [i]) ;*/
            bu_err = PASAMAX (t, bu_err) ;
        }
    }
    printf ("feasibility error: lo: %e hi: %e bl: %e bu: %e\n",
             lo_err, hi_err, bl_err, bu_err) ;
    if ( nrow > 0 )
    {
        pasa_free (temp) ;
    }
    temp = (PASAFLOAT *) pasa_malloc (&status, ucol, sizeof (PASAFLOAT)) ;
    /* evaluate func */
    if ( QP )
    {
        if ( Com->hprod_status )
        {
            pasa_builtin_hprod (temp, x, NULL, ucol, ucol,
                                Com->Hp, Com->Hi, Com->Hx) ;
        }
        else
        {
            Com->hprod (temp, x, NULL, ucol, ucol) ; /* g = Q*x */
        }
        cost = 0.5*pasa_dot (temp, x, ucol) ;        /* f = .5*x'Qx */
        if ( Com->userc != NULL )
        {
            cost += pasa_dot(x, Com->userc, ucol) ; /*f = .5*x'*Qx + c'*x*/
            pasa_step (temp, temp, Com->userc, PASAONE, ucol) ;/* g = Qx+c */
        }
    }
    else
    {
        Com->value (temp, x, ucol) ;
        cost = temp [0] ;
        Com->grad (temp, x, ucol) ; /* evaluate grad */
    }

    fp = PASAZERO ;
    fl = PASAZERO ;
    /* add penalty term to objective if it is present */
    gpen = NULL ;
    if ( (Com->location == PASA_CG_DESCENT) && (Com->use_penalty == TRUE)
                                            && (QP == FALSE) )
    {
        PPINT *ATp, *ATi, *ir ;
        PPFLOAT c, *ATx, *b, *lambda ;
        PPwork *Work ;
        if ( use_pproj == TRUE )
        {
            Work = Com->ppcom->Work ;
            b = Com->b ;
            ATp = Work->ATp ;
            ATi = Work->ATi ;
            ATx = Work->ATx ;
            ir = Work->ir ;
        }
        gpen = (PASAFLOAT *) pasa_malloc (&status, ncol, sizeof (PASAFLOAT)) ;
        c = 0.5*Com->penalty ;
        lambda = Com->lambda_pen ;   /* lambda for penalty */
        Gpen =  Com->gpen ;          /* gradient of penalty term */
        pasa_copyx (gpen, Gpen, nf) ;
        for (i = 0; i < nrow; i++)
        {
            if ( use_pproj == TRUE )
            {
                if ( ir [i] == 0 )
                {
                    s = CGZERO ;
                    q = ATp [i+1] ;
                    for (p = ATp [i]; p < q; p++)
                    {
                        s += ATx [p]*Com->x [ATi [p]] ;
                    }
                    t = b [i] - s ;
                    fl += lambda [i]*t ;
/*printf ("i: %i t: %25.15e lambda: %25.15e\n", i, t, lambda [i]) ;*/
                    fp += c*t*t ;
                    s = -(t*Com->penalty + lambda [i]) ;
                    for (p = ATp [i]; p < q; p++)
                    {
                        gpen [ATi [p]] -= ATx [p]*s ;
                    }
                }
            }
            else if ( use_napheap == TRUE )
            {
                s = pasa_dot (a, Com->x, nf) ;
                t = Com->nap_bl - s ;
                fl += lambda [i]*t ;
                fp += c*t*t ;
                s = -(t*Com->penalty + lambda [i]) ;
                for (j = 0; j < nf; j++)
                {
                    gpen [j] -= a [j]*s ;
                }
            }
        }
        cost += fp + fl ;
        printf ("penalty error: %e penalty: %e\n", Com->fp - fp - fl, fp + fl) ;
        printf ("sup-norm penalty grad error: %e, sup norm grad penalty: %e\n",
                 pasa_sup_normx (gpen, nf), pasa_sup_normx (Gpen, nf)) ;
    }
    printf ("function value error: %e function value: %25.15e "
            "penalty: %25.15e lambda term: %25.15e\n",
             cost - Com->f, cost, fp, fl) ;

    /* gradient error */
    g_err = PASAZERO ;
    if ( order == NULL )
    {
        for (j = 0; j < Com->nf; j++)
        {
            t = fabs (Com->g [j] - temp [j]) ;
            g_err = PASAMAX (t, g_err) ;
        }
    }
    else
    {
        for (j = 0; j < Com->nf; j++)
        {
            t = fabs (Com->g [j] - temp [order [j]]) ;
            g_err = PASAMAX (t, g_err) ;
        }
    }
    printf ("gradient error: %e\n", g_err) ;

    /* in CG check that d lies in null space of current active constraints */
    if ( (Com->location == PASA_CG_DESCENT) && (Com->Aexists == TRUE) &&
         (use_pproj == TRUE) )
    {
        PPwork *Work ;
        PPINT *ATp, *ATi, *ir ;
        PPFLOAT *ATx, *d ;
        Work = Com->ppcom->Work ;
        ir = Work->ir ;
        ATp = Work->ATp ;
        ATi = Work->ATi ;
        ATx = Work->ATx ;
        d = Com->d ;
        printf ("sup norm of d: %e\n", pasa_sup_normx (d, Com->nf)) ;
        for (i = 0; i < nrow; i++)
        {
            if ( ir [i] == 0 )
            {
                s = PASAZERO ;
                t = PASAZERO ;
                q = ATp [i+1] ;
                for (p = ATp [i]; p < q; p++)
                {
                    t = PASAMAX (t, fabs(ATx [p]*d [ATi [p]])) ;
                    s += ATx [p]*d [ATi [p]] ;
                }
                if ( t > 1.e-10 )
                {
                    if ( fabs (s/t) > Parm->debugtol )
                    {
                        printf ("d not in null space of active constraints\n") ;
                        printf ("row: %ld A*d: %e abs(A*d): %e\n",
                               (LONG) i, s, t) ;
                        /* pasa_error (-1, __FILE__, __LINE__, "STOP") ;*/
                    }
                }
            }
        }
    }
    else if ( (Com->location == PASA_CG_DESCENT) && (Com->Aexists == TRUE) &&
              (use_napheap == TRUE) )
    {
        PASAFLOAT *d ;
        d = Com->d ;
        s = PASAZERO ;
        t = PASAZERO ;
        for (j = 0; j < nf; j++)
        {
            t = PASAMAX (t, fabs (a [j]*d [j])) ;
            s += a [j]*d [j] ;
        }
        if ( t > 1.e-10 )
        {
            if ( fabs (s/t) > Parm->debugtol )
            {
                printf ("d not in null space of active constraints\n") ;
                printf ("row: %ld A*d: %e abs(A*d): %e\n",
                       (LONG) 0, s, t) ;
            }
        }
    }

    if ( gpen != NULL )
    {
        pasa_free (gpen) ;
    }
    pasa_free (temp) ;
}

void check_zero
(
    PASAFLOAT *x,
    PASAINT    n,
    int        l
)
{
    PASAINT j ;
    for (j = 0; j < n; j++)
    {
        if ( x [j] != PASAZERO )
        {
             printf ("location: %ld component %ld value: %e\n",
                      (LONG) l, (LONG) j, x [j]) ;
             pproj_error (-1, __FILE__, __LINE__, "stop\n") ;
        }
    }
}
/* ==========================================================================
   === pasa_checkA ==========================================================
   ==========================================================================
   Check that Prob->A and Prob->AT corresponds to the current compressed
   problem where the columns associated with bound variables have been removed.
   ========================================================================== */
void pasa_checkA
(
    PASAcom    *Com /* PASAcom structure */
)
{
    PASAINT i, j, k, p, q ;

    int        location = Com->location ;
    int              sI = sizeof (PASAINT) ;
    int              sF = sizeof (PASAFLOAT) ;
    PASAcopy      *Copy = Com->Copy ;           /* copy of original prob */
    PPprob        *Prob = Com->ppcom->Prob ;    /* problem in PPROJ */
    PPwork        *Work = Com->ppcom->Work ;   /* work structure in PPROJ */

    /* arrays in Prob structure */
    PASAINT         *Ap = Prob->Ap ;
    PASAINT         *Ai = Prob->Ai ;
    PASAINT        *Anz = Prob->Anz ;
    PASAFLOAT       *Ax = Prob->Ax ;
    PASAINT      *ifree = Com->ifree ;      /* current free cols, user coor */
    PASAINT          nf = Com->nf ;         /* number of indices in ifree */
    PASAINT          nc = Com->nc ;         /* number of bound columns */
    PASAINT *bound_cols = Com->bound_cols ; /* cols in user coor */
    PASAINT          nr = Com->nr ;         /* number of bound inequalities */
    PASAINT          ni = Com->ni ;         /* # strict inequal original prob */
    PASAINT *bound_rows = Com->bound_rows ;
    PASAINT    *colperm = Com->colperm ;    /* user col for pproj col */
    PASAINT        *ATp = Work->ATp ;       /* transpose stored in Work struc */
    PASAINT        *ATi = Work->ATi ;
    PASAFLOAT      *ATx = Work->ATx ;
    int             *ib = Work->ib ;        /* ib = 0 for free, +-1 for bound */
    PASAINT         *ir = Work->ir ;
    PASAINT    *RLinkUp = Work->RLinkUp ;
    PASAINT     prob_ni = Prob->ni ;        /* ir <= prob_ni for active rows */
    PASAINT          nz = Ap [nf] ;         /* nonzeros in A */


    /* original matrix dimensions */
    PASAINT        nrow = Com->nrow ;
    PASAINT        ncol = Com->ncol ;
    PASAINT        ucol = Com->ucol ;
    PASAINT        *CTp = Copy->ATp ;
    PASAINT        *CTi = Copy->ATi ;
    PASAFLOAT      *CTx = Copy->ATx ;

    /* form a compressed version of CT by mapping new columns to their
       new locations and discarding the bound columns */
    PASAINT *mapcols = (PASAINT *) malloc (ucol*sI) ;
    pasa_initi (mapcols, EMPTY, ucol) ;
    ASSERT (ncol == nf + nc ) ;
    if ( (Com->location != PASA_PRE_GRAD_PROJ) && (Com->Bounds == TRUE) )
    {
        for (k = 0; k < nf; k++)
        {
            mapcols [ifree [k]] = k ;
        }
        for (k = 0; k < nc; k++)
        {
            j = bound_cols [k] ;
            if ( j < 0 ) j = -(j+1) ; /* lower bound cols stored as -(j+1) */
            ASSERT (mapcols [j] == EMPTY) ;
            mapcols [j]-- ;
        }

        /* reset bound indices in mapcols to EMPTY */
        for (k = 0; k < nc; k++)
        {
            j = bound_cols [k] ;
            if ( j < 0 ) j = -(j+1) ; /* lower bound cols stored as -(j+1) */
            mapcols [j] = EMPTY ;
        }

        /* check that Work->AT agrees with the compressed version of copy AT */
        ASSERT (ATp [0] == 0) ;
        for (i = 0; i < nrow; i++)
        {
            q = ATp [i] ;
            for (p = CTp [i]; p < CTp [i+1]; p++)
            {
                j = mapcols [colperm [CTi [p]]] ;
                if ( j == EMPTY ) continue ;
                ASSERT (j == ATi [q]) ;
                ASSERT (CTx [p] == ATx [q]) ;
                q++ ;
            }
            ASSERT (q == ATp [i+1]) ;
        }
    }
    /* check that the nz's in A and AT match */
    ASSERT (nz == ATp [nrow]) ;
    /* Both the indices and numerical entries in AT match those in copy AT */

    /* Check that the transpose of A, stored in B, is equal to AT */
    PASAINT   *Bp = (PASAINT *) malloc ((nrow+1)*sI) ;
    PASAINT   *Bi = (PASAINT *) malloc (nz*sI) ;
    PASAFLOAT *Bx = (PASAFLOAT *) malloc (nz*sF) ;
    PASAINT    *W = (PASAINT *) malloc (nrow*sI) ;
    sopt_transpose (Bp, Bi, Bx, Ap, Ai, Ax, nrow, nf, W) ;

    for (i = 0; i <= nrow; i++)
    {
        ASSERT (Bp [i] == ATp [i]) ;
    }
    for (k = 0; k < nz; k++)
    {
        ASSERT (Bi [k] == ATi [k]) ;
        ASSERT (Bx [k] == ATx [k]) ;
    }

    /* Check that the Anz entries in A are in increasing order and that
       the rows are all active. Also check that the entries beneath to
       top Anz entries represent inactive rows. */
    for (j = 0; j < nf; j++)
    {
        p = Ap [j] ;
        q = Ap [j] + Anz [j] ;
        for (k = p; k < q; k++)
        {
            if ( location == PASA_PRE_GRAD_PROJ )
            {
                ASSERT (ir [Ai [k]] <= ni) ;
            }
            else
            {
                ASSERT (ir [Ai [k]] == 0) ;          /* equality constraint */
            }
            if ( k > p )
            {
                ASSERT (Ai [k-1] < Ai [k]) ;     /* column is increasing */
            }
        }
        for (; k < Ap [j+1]; k++)
        {
            i = ir [Ai [k]] ;
            if ( ir [Ai [k]] <= prob_ni )
            {
                printf ("ir: %ld prob_ni: %ld\n",
                       (LONG) ir [Ai [k]], (LONG) prob_ni) ;
                fflush(stdout) ;
                pproj_error (-1, __FILE__, __LINE__, "stop\n") ;
            }
        }
    }

    /* Check that all nf entries in ib are zero */
    if ( location != PASA_PRE_GRAD_PROJ )
    {
        for (j = 0; j < nf; j++)
        {
            ASSERT (ib [j] == 0) ;
        }
    }

    /* Check that RLinkUp corresponds to the active rows */
    if ( location != PASA_PRE_GRAD_PROJ )
    {
        for (i = 0; i < nrow; i++)
        {
            if ( ir [i] == 0 ) break ; /* first active row */
        }
        if ( i == nrow ) /* no active rows */
        {
            ASSERT (RLinkUp [nrow] == nrow) ;
        }
        else
        {
            ASSERT (RLinkUp [nrow] == i) ;
            k = RLinkUp [i] ;
            for (i++; i < nrow; i++)
            {
                if ( i < k )
                {
                    ASSERT (ir [i] > prob_ni) ;
                }
                else
                {
                    ASSERT (ir [i] == 0) ;
                    ASSERT (RLinkUp [k] > k) ;
                    k = RLinkUp [k] ;
                }
            }
            ASSERT (k == nrow) ;
        }
    }

    for (k = 0; k < nr; k++)
    {
        i = bound_rows [k] ;
        if ( i > 0 ) i-- ;
        else         i = -(i+1) ;
        ASSERT (ir [i] == 0) ;
    }
    free (Bp) ;
    free (Bi) ;
    free (Bx) ;
    free (W) ;
    free (mapcols) ;
}

/* compute orthonormal basis for space perpindicular to the columns of a
   given matrix A, return the number of columns in Z */
PASAINT pasa_null
(
    PASAcom *Com,
    PASAFLOAT *A,  /* matrix of size nrow = Com->nf by ncol */
    PASAINT ncol
)
{
    PASAINT i, j, k, nrow ;
    PASAFLOAT s, t, *Ap, *Aq, *Hp, *Hq, *Z, *Zp ;
    Z = Com->Z ;
    nrow = Com->nf ;
    PASAFLOAT *colnorm = Z ; Z = Z+ncol ;
    PASAFLOAT *H = Z ;
    PASAFLOAT mindiag = PASAINF ;
    /* when the max norm of the column squared exceeds colmax_recompute,
       recompute the square norms of the columns */
    PASAFLOAT colmax = -1 ; PASAFLOAT colmax_recompute = 0 ;
    Hp = H ;
    PASAINT const minrowcol = PASAMIN (nrow, ncol) ;
    PASAFLOAT C = Com->pasaparm->QRcutoff ;
    for (j = 0; j < minrowcol; j++)
    {
        if ( colmax <= colmax_recompute )
        {
            /* Compute the column norms squared */
            colmax = -1 ;
            Ap = A+(j*nrow) ;
            for (i = j; i < minrowcol; i++)
            {
                t = pasa_dot (Ap+j, Ap+j, nrow-j) ;
                colnorm [i] = t ;
                if ( t > colmax )
                {
                    colmax = t ;
                    k = i ;
                }
                Ap = Ap+nrow ;
            }
            colmax_recompute = 1.e-5 * colmax ;
        }
        if ( sqrt(colmax) <= C ) /* done */
        {
            printf ("break at col %ld due to small colmax: %e cutoff: %e\n",
                     (LONG) j, sqrt (colmax), C) ;
            break ;
        }
        /* swap colnorm for column k and j */
        colnorm [k] = colnorm [j] ;
        colnorm [j] = colmax ;
        Ap = A+nrow*j ; /* start of column j */
        Aq = A+nrow*k ; /* start of column k */
        /* exchange columns j and k */
        for (i = j; i < nrow; i++)
        {
            t = Ap [i] ;
            Ap [i] = Aq [i] ;
            Aq [i] = t ;
        }
        /* compute Householder matrix for column j */
        t = pasa_dot (Ap+j, Ap+j, nrow-j) ;
        if ( t == PASAZERO )
        {
            printf ("remainder of columns are zero: %ld\n", (LONG) j) ;
            break ;
        }
        s = sqrt(t) ;
        if ( s < mindiag ) mindiag = s ;
        t = 1/sqrt(2*s*(s+fabs(Ap [j]))) ;
        if ( Ap [j] < PASAZERO )
        {
            Hp [0] = (Ap [j] - s)*t ;
            Ap [j] = s ;
        }
        else
        {
            Hp [0] = (Ap [j] + s)*t ;
            Ap [j] = -s ;
        }
        pasa_scale (Hp+1, Ap+j+1, t, nrow-j-1) ;
        /* apply Household to A and update column norms */
        colmax = -1 ;
        for (i = j+1; i < minrowcol; i++)
        {
            Ap = Ap+nrow ;
            t = 2*pasa_dot (Hp, Ap+j, nrow-j) ;
            pasa_daxpy (Ap+j, Hp, -t, nrow-j) ;
            colnorm [i] -= Ap [j]*Ap [j] ;
            if ( colnorm [i] > colmax )
            {
                colmax = colnorm [i] ;
                k = i ;
            }
        }
        Hp = Hp+nrow-j ;
    }
    printf ("min diagonal of R: %e\n", mindiag) ;
    ncol = nrow - j ; /* number of columns in Z */
    if ( (ncol == nrow) || (ncol == 0) ) return (ncol) ;
    Z = Hp ;
    Com->Z = Z ;
    Zp = Z ;
    j-- ;
    Hq = Hp-nrow+j ; /* Hq now points to the start of last Householder vec */
    for (i = 0; i < ncol; i++)
    {
        Hp = Hq ;
        /* compute initial column vector */
        t = 2*Hp [i+1] ;

        pasa_initx (Zp, PASAZERO, nrow) ;
        pasa_scale (Zp+j, Hp, -t, nrow-j) ;
        Zp [j+i+1] += PASAONE ;
        for (k = j-1; k >= 0; k--)
        {
            Hp = Hp-nrow+k ;
            t = 2*pasa_dot (Zp+k, Hp, nrow-k) ;
            pasa_daxpy (Zp+k, Hp, -t, nrow-k) ;
        }
        Zp = Zp+nrow ;
    }
    return (ncol) ;
}

/* ==========================================================================
   === pasa_builtin_hprod ===================================================
   ==========================================================================
   If the use provides the Hessian of a QP in sparse matrix format, then
   builtin_hprod computes Hd = H(:, ifree)*d
   ========================================================================== */
void pasa_builtin_hprod
(
    PASAFLOAT          *Hd, /* Hd = H*d */
    PASAFLOAT           *d, /* length n */
    PASAINT   const *ifree, /* length n, column ifree [i] in H * d [i] */
    PASAINT   const      m, /* H is m by m and Hd has length m */
    PASAINT   const      n, /* number of entries in both d and ifree */
    PASAINT   const    *Hp, /* column pointers */
    PASAINT   const    *Hi, /* row indices */
    PASAFLOAT const    *Hx  /* numerical entries */
)
{
    PASAINT j, k, p ;
    pasa_initx (Hd, PASAZERO, m) ;
    if ( ifree != NULL )
    {
        for (j = 0; j < n; j++)
        {
            PASAFLOAT const t = d [j] ;
            if ( t ) /* if t != 0 */
            {
                k = ifree [j] ;
                PASAINT const q = Hp [k+1] ;
                for (p = Hp [k]; p < q; p++)
                {
                    Hd [Hi [p]] += t*Hx [p] ;
                }
            }
        }
    }
    else
    {
        for (j = 0; j < n; j++)
        {
            PASAFLOAT const t = d [j] ;
            if ( t ) /* if t != 0 */
            {
                PASAINT const q = Hp [j+1] ;
                for (p = Hp [j]; p < q; p++)
                {
                    Hd [Hi [p]] += t*Hx [p] ;
                }
            }
        }
    }
}
