function make()
    global print_info ;
    % Option for bypassing compilation of CHOLMOD objects (for debugging)
    bypass_cholmod = false ; % Default should be false for regular installation
    % Option for bypassing compilation of PPROJ objects (for debugging)
    bypass_pproj = false ; % Default should be false for regular installation
    % Option for bypassing cleanup of objects (for debugging)
    bypass_cleanup = false ; % Default value should be false

    %% ------------------ Compile the CHOLMOD object files ------------------ %%
    % Initialize details variable to 0 
    details = 0 ;
    % location of SuiteSparse directory
    SuiteSparsePath = '../../SuiteSparseX/' ;
    MATLABpath = strcat(SuiteSparsePath, 'CHOLMOD/MATLAB/') ;
    
    v = version ;
    try
        % ispc does not appear in MATLAB 5.3
        pc = ispc ;
        mac = ismac ;
    catch                                                                       %#ok
        % if ispc fails, assume we are on a Windows PC if it's not unix
        pc = ~isunix ;
        mac = 0 ;
    end
    
    flags = '' ;
    is64 = ~isempty (strfind (computer, '64')) ;
    if (is64)
        % 64-bit MATLAB
        flags = '-largeArrayDims' ;
    end

    % MATLAB 8.3.0 now has a -silent option to keep 'mex' from burbling too much
    if (~verLessThan ('matlab', '8.3.0'))
        flags = ['-silent ' flags] ;
    end
    
    %% ----- Set include string to all directories to be included ----- %%
    % list of subdirectories of SuiteSparse to include in mex compilation of pasa
    include_dir = {'AMD/Include/', ...
                   'COLAMD/Include/', ...
                   'CCOLAMD/Include/', ...
                   'CAMD/Include/', ...
                   'CHOLMOD/Include/', ...
                   'SuiteSparse_config/'} ;
    
    include = ['-I"' MATLABpath '"'] ;
    for f = include_dir
        ff = strcat(SuiteSparsePath, f {1}) ;
        include = [include ' -I"' ff '"'] ;
    end
    
    if (verLessThan ('matlab', '7.0'))
        % do not attempt to compile CHOLMOD with large file support
        include = [include ' -DNLARGEFILE'] ;
    elseif (~pc)
        % Linux/Unix require these flags for large file support
        include = [include ' -D_FILE_OFFSET_BITS=64 -D_LARGEFILE64_SOURCE'] ;
    end
    
    if (verLessThan ('matlab', '6.5'))
        % logical class does not exist in MATLAB 6.1 or earlie
        include = [include ' -DMATLAB6p1_OR_EARLIER'] ;
    end
    
    % Determine if METIS is available
    metis_path = strcat(SuiteSparsePath, 'metis-5.1.0') ;
    have_metis = exist (metis_path, 'dir') ;
    
    % If METIS is available add necessary include directories
    if (have_metis)
        fprintf ('Compiling CHOLMOD with METIS 5.1.0 for MATLAB Version %s\n', v) ;
        include = [include ' -I' metis_path '/include'] ;
        include = [include ' -I' metis_path '/GKlib'] ;
        include = [include ' -I' metis_path '/libmetis'] ;
    else
        fprintf ('Compiling CHOLMOD without METIS for MATLAB Version %s\n', v) ;
        include = ['-DNPARTITION ' include] ;
    end
    
    %---------------------------------------------------------------------------
    % BLAS option
    %---------------------------------------------------------------------------
    
    % This is exceedingly ugly. The MATLAB mex command needs to be told where to
    % find the LAPACK and BLAS libraries, which is a real portability nightmare.
    
    if (pc)
        if (verLessThan ('matlab', '7.5'))
            lapack = 'libmwlapack.lib' ;
        elseif ( verLessThan ('matlab', '9.5') )
            lapack = 'libmwlapack.lib libmwblas.lib' ;
        else % version = 9.5 or newer
            lapack = '-lmwlapack -lmwblas' ;
        end
    else
        if (verLessThan ('matlab', '7.5'))
            lapack = '-lmwlapack' ;
        else
            lapack = '-lmwlapack -lmwblas' ;
        end
    end
    
    if (is64 && ~verLessThan ('matlab', '7.8'))
        % versions 7.8 and later on 64-bit platforms use a 64-bit BLAS
        fprintf ('with 64-bit BLAS\n') ;
        flags = [flags ' -DBLAS64'] ;
    end
    
    if (~(pc || mac))
        % for POSIX timing routine
        lapack = [lapack ' -lrt'] ;
    end
    
    
    %---------------------------------------------------------------------------
    config_path = strcat(SuiteSparsePath, ...
                                      'SuiteSparse_config/SuiteSparse_config') ;
    config_src = { config_path } ;
    
    ordering_src = { ...
        'AMD/Source/amd_1', ...
        'AMD/Source/amd_2', ...
        'AMD/Source/amd_aat', ...
        'AMD/Source/amd_control', ...
        'AMD/Source/amd_defaults', ...
        'AMD/Source/amd_dump', ...
        'AMD/Source/amd_global', ...
        'AMD/Source/amd_info', ...
        'AMD/Source/amd_order', ...
        'AMD/Source/amd_postorder', ...
        'AMD/Source/amd_post_tree', ...
        'AMD/Source/amd_preprocess', ...
        'AMD/Source/amd_valid', ...
        'CAMD/Source/camd_1', ...
        'CAMD/Source/camd_2', ...
        'CAMD/Source/camd_aat', ...
        'CAMD/Source/camd_control', ...
        'CAMD/Source/camd_defaults', ...
        'CAMD/Source/camd_dump', ...
        'CAMD/Source/camd_global', ...
        'CAMD/Source/camd_info', ...
        'CAMD/Source/camd_order', ...
        'CAMD/Source/camd_postorder', ...
        'CAMD/Source/camd_preprocess', ...
        'CAMD/Source/camd_valid', ...
        'COLAMD/Source/colamd', ...
        'CCOLAMD/Source/ccolamd'} ;

    % Update ordering source to be located in SuiteSparse directory
    for k = 1:length(ordering_src)
        ordering_src {k} = strcat(SuiteSparsePath, ordering_src {k}) ;
    end
    
    if (have_metis)
        metis_src = {
            'GKlib/b64', ...
            'GKlib/blas', ...
            'GKlib/csr', ...
            'GKlib/error', ...
            'GKlib/evaluate', ...
            'GKlib/fkvkselect', ...
            'GKlib/fs', ...
            'GKlib/getopt', ...
            'GKlib/gkregex', ...
            'GKlib/graph', ...
            'GKlib/htable', ...
            'GKlib/io', ...
            'GKlib/itemsets', ...
            'GKlib/mcore', ...
            'GKlib/memory', ...
            'GKlib/omp', ...
            'GKlib/pdb', ...
            'GKlib/pqueue', ...
            'GKlib/random', ...
            'GKlib/rw', ...
            'GKlib/seq', ...
            'GKlib/sort', ...
            'GKlib/string', ...
            'GKlib/timers', ...
            'GKlib/tokenizer', ...
            'GKlib/util', ...
            'libmetis/auxapi', ...
            'libmetis/balance', ...
            'libmetis/bucketsort', ...
            'libmetis/checkgraph', ...
            'libmetis/coarsen', ...
            'libmetis/compress', ...
            'libmetis/contig', ...
            'libmetis/debug', ...
            'libmetis/fm', ...
            'libmetis/fortran', ...
            'libmetis/frename', ...
            'libmetis/gklib', ...
            'libmetis/graph', ...
            'libmetis/initpart', ...
            'libmetis/kmetis', ...
            'libmetis/kwayfm', ...
            'libmetis/kwayrefine', ...
            'libmetis/mcutil', ...
            'libmetis/mesh', ...
            'libmetis/meshpart', ...
            'libmetis/minconn', ...
            'libmetis/mincover', ...
            'libmetis/mmd', ...
            'libmetis/ometis', ...
            'libmetis/options', ...
            'libmetis/parmetis', ...
            'libmetis/pmetis', ...
            'libmetis/refine', ...
            'libmetis/separator', ...
            'libmetis/sfm', ...
            'libmetis/srefine', ...
            'libmetis/stat', ...
            'libmetis/timing', ...
            'libmetis/util', ...
            'libmetis/wspace'} ;
    
        for i = 1:length (metis_src)
            metis_src {i} = [metis_path '/' metis_src{i}] ;
        end
    end
    
    cholmod_matlab = { [MATLABpath 'cholmod_matlab'] } ;
    
    cholmod_src = {
        '../Core/cholmod_aat', ...
        '../Core/cholmod_add', ...
        '../Core/cholmod_band', ...
        '../Core/cholmod_change_factor', ...
        '../Core/cholmod_common', ...
        '../Core/cholmod_complex', ...
        '../Core/cholmod_copy', ...
        '../Core/cholmod_dense', ...
        '../Core/cholmod_error', ...
        '../Core/cholmod_factor', ...
        '../Core/cholmod_memory', ...
        '../Core/cholmod_sparse', ...
        '../Core/cholmod_transpose', ...
        '../Core/cholmod_triplet', ...
        '../Check/cholmod_check', ...
        '../Check/cholmod_read', ...
        '../Check/cholmod_write', ...
        '../Cholesky/cholmod_amd', ...
        '../Cholesky/cholmod_analyze', ...
        '../Cholesky/cholmod_colamd', ...
        '../Cholesky/cholmod_etree', ...
        '../Cholesky/cholmod_factorize', ...
        '../Cholesky/cholmod_postorder', ...
        '../Cholesky/cholmod_rcond', ...
        '../Cholesky/cholmod_resymbol', ...
        '../Cholesky/cholmod_rowcolcounts', ...
        '../Cholesky/cholmod_rowfac', ...
        '../Cholesky/cholmod_solve', ...
        '../Cholesky/cholmod_spsolve', ...
        '../MatrixOps/cholmod_drop', ...
        '../MatrixOps/cholmod_horzcat', ...
        '../MatrixOps/cholmod_norm', ...
        '../MatrixOps/cholmod_scale', ...
        '../MatrixOps/cholmod_sdmult', ...
        '../MatrixOps/cholmod_ssmult', ...
        '../MatrixOps/cholmod_submatrix', ...
        '../MatrixOps/cholmod_vertcat', ...
        '../MatrixOps/cholmod_symmetry', ...
        '../Modify/cholmod_rowadd', ...
        '../Modify/cholmod_rowdel', ...
        '../Modify/cholmod_updown', ...
        '../Supernodal/cholmod_super_numeric', ...
        '../Supernodal/cholmod_super_solve', ...
        '../Supernodal/cholmod_super_symbolic', ...
        '../Partition/cholmod_ccolamd', ...
        '../Partition/cholmod_csymamd', ...
        '../Partition/cholmod_camd', ...
        '../Partition/cholmod_metis', ...
        '../Partition/cholmod_nesdis'} ;
    % Update cholmod source to be located in CHOLMOD directory
    for k = 1:length(cholmod_src)
        cholmod_src {k} = strcat(MATLABpath, cholmod_src {k}) ;
    end
    
    if (pc)
        % Windows does not have drand48 and srand48, required by METIS.  Use
        % drand48 and srand48 in CHOLMOD/MATLAB/Windows/rand48.c instead.
        % Also provide Windows with an empty <strings.h> include file.
        obj_extension = '.obj' ;
        Windows_path = strcat(MATLABpath, 'Windows/') ;
        Windows_rand48 = strcat(MATLABpath, 'Windows/rand48') ;
        cholmod_matlab = [cholmod_matlab {Windows_rand48}] ;
        include = [include ' -I' Windows_path] ;
    else
        obj_extension = '.o' ;
    end
    
    % compile each library source file
    cholmod_obj = '' ;
    
    source = [ordering_src config_src cholmod_src cholmod_matlab] ;
    if (have_metis)
        source = [metis_src source] ;
    end
    
    details = 0 ;
    kk = 0 ;
    
    %% -------------- Compile CHOLMOD objects from source code -------------- %%
    for f = source
        ff = f {1} ;
        if (isequal (ff, [metis_path '/GKlib/util']))
            % special case, since a file with the same name also exists in libmetis
            copyfile ([ff '.c'], 'GKlib_util.c', 'f') ;
            ff = 'GKlib_util' ;
            o = 'GKlib_util' ;
        elseif (isequal (ff, [metis_path '/GKlib/graph']))
            % special case, since a file with the same name also exist in libmetis
            copyfile ([ff '.c'], 'GKlib_graph.c', 'f') ;
            ff = 'GKlib_graph' ;
            o = 'GKlib_graph' ;
        else
            slash = strfind (ff, '/') ;
            if (isempty (slash))
                slash = 1 ;
            else
                slash = slash (end) + 1 ;
            end
            o = ff (slash:end) ;
        end
        cholmod_obj = [cholmod_obj  ' ' o obj_extension] ;
        if (bypass_cholmod == false)
            fprintf('\n\n  Making object file %s.o\n\n', o) ;
            s = sprintf ('mex %s -DDLONG -O %s -c %s.c', flags, include, ff) ;
            kk = do_cmd (s, kk, details) ;
        end
    end

    % Indicate whether or not cholmod has been compiled successfully
    if (bypass_cholmod == false)
        fprintf ('\nCHOLMOD successfully compiled\n') ;
    else
        fprintf ('\nWarning: Compilation of CHOLMOD bypassed\n') ;
    end

    %% ---------- Compile the PPROJ object and mex files ---------- %%
    % Used for printing details when compiling
    details = 0 ;
    % Used during compilation of objects
    kk = 0 ;
    
    %% ---------- Indicate include directories and source code ---------- %%
    % expand cholmod include directories to incorporate those of pproj
    include = [include ' -I. -I../Include -I../../SuiteOPTconfig'] ;
    % Add directories containing prototypes for SUITEOPT matlab functions
    include = [include ' -I../../MATLAB/Include'] ;

    % list of suiteopt source files to include in mex compilation
    suiteopt_matlab_src = {'../../SuiteOPTconfig/sopt', ...
             '../../MATLAB/Source/suiteopt_matlab', ...
             '../../MATLAB/Source/pproj_matlab'} ;
    
    % list of source files to include in mex compilation of pproj
    pproj_check_src = {'../Check/pproj_checkA', ...
                       '../Check/pproj_checkb', ...
                       '../Check/pproj_check_AFT', ...
                       '../Check/pproj_check_diag3', ...
                       '../Check/pproj_check_line', ...
                       '../Check/pproj_check_order', ...
                       '../Check/pproj_checkc', ...
                       '../Check/pproj_check_eqn5', ...
                       '../Check/pproj_check_diag', ...
                       '../Check/pproj_check_link', ...
                       '../Check/pproj_check_AT', ...
                       '../Check/pproj_check_const', ...
                       '../Check/pproj_checkF', ...
                       '../Check/pproj_check_minheap', ...
                       '../Check/pproj_check_back', ...
                       '../Check/pproj_checkD', ...
                       '../Check/pproj_check_dual', ...
                       '../Check/pproj_check_forward', ...
                       '../Check/pproj_check_deriv', ...
                       '../Check/pproj_check_modlist'} ;
    
    pproj_src = {'../Source/pproj_init', ...
                 '../Source/pproj_phase1', ...
                 '../Source/pproj_hotchol', ...
                 '../Source/pproj_coor_ascent', ...
                 '../Source/pproj_dasa', ...
                 '../Source/pproj_sparsa', ...
                 '../Source/pproj_ssor0', ...
                 '../Source/pproj_ssor1', ...
                 '../Source/pproj_print', ...
                 '../Source/pproj_default', ...
                 '../Source/pproj'} ;
    
    %% ------------------- Required flags ------------------- %%
    % These flags are REQUIRED for the PPROJ mex function to compile
    % add '-DPPROJ_MATLAB' to indicate PPROJ
    flags = [flags ' -DPPROJ_MATLAB'] ;

    % add '-DSUITEOPT_MATLAB_BOUND_CONSTRAINTS' to indicate bound constraints
    flags = [flags ' -DSUITEOPT_MATLAB_BOUND_CONSTRAINTS'] ;

    % add '-DSUITEOPT_MATLAB_POLYHEDRAL_CONSTRAINTS' indicate polyhedral constr
    flags = [flags ' -DSUITEOPT_MATLAB_POLYHEDRAL_CONSTRAINTS'] ;

    %% ------------------- Debugging flags ------------------- %%
    % add '-g' flag for debugging           
    % flags = [flags ' -g'] ; 
    
    % add '-v' flag for verbose mode           
    % flags = [flags ' -v'] ; 
    
    % add '-DNDEBUG' to turn off debugging
    flags = [flags ' -DNDEBUG'] ;
     
    % add '-DDEBUG_SUITEOPT_MEX' to turn on debugging for SuiteOPT matlab funcs
    %flags = [flags ' -DDEBUG_SUITEOPT_MEX'] ;

    % add '-DDEBUG_PPROJ_MEX' to turn on debugging for PPROJ MEX funcs
    %flags = [flags ' -DDEBUG_PPROJ_MEX'] ;
    
    %% ---------------- Compile objects from source code ---------------- %%
    source = [suiteopt_matlab_src pproj_check_src pproj_src] ;
    
    % Keep track of all object files compiled for later cleanup of directory
    pproj_obj = '' ;
    
    % compile object files 
    for f = source 
        ff = f {1} ;
        slash = strfind (ff, '/') ;
        if (isempty (slash))
            slash = 1 ;
        else
            slash = slash (end) + 1 ;
        end
        o = ff (slash:end) ;
        pproj_obj = [pproj_obj ' ' o obj_extension] ;
        if (bypass_pproj == false)
            sprintf('Compiling %s.o', o) 
            s = sprintf('mex %s -DDLONG -O %s -c %s.c', flags, include, ff) ;
            kk = do_cmd (s, kk, details) ;
        end
    end
    
    %% ---------------- Compile mex function(s) ---------------- %%
    % list of mex functions required in mex compilation of pproj
    pproj_mex_src = {'pproj'} ;
    
    % compile each mexFunction
    for f = pproj_mex_src
    %    sprintf('Compiling %s.c', f{1}) 
        s = sprintf('mex %s -DDLONG %s %s_mex.c %s %s %s -output %s', ...
                    flags, include, f{1}, cholmod_obj, pproj_obj, lapack, f{1}) ;
        kk = do_cmd (s, kk, details) ;
    end
    
    %% ---------------- Clean up and exit ---------------- %%
    % clean up object files created during installation
    if (bypass_cleanup == false)
        % remove CHOLMOD files, if they exist
        s = ['delete ' cholmod_obj] ;
        do_cmd (s, kk, details) ;
        % remove PPROJ files, if they exist
        s = ['delete ' pproj_obj] ;
        do_cmd (s, kk, details) ;
        % remove the renamed METIS files, if they exist
        if (exist ('GKlib_util.c', 'file'))
            delete ('GKlib_util.c') ;
        end
        if (exist ('GKlib_graph.c', 'file'))
            delete ('GKlib_graph.c') ;
        end
    end
    fprintf ('\nPPROJ successfully compiled\n') ;

    % Call pproj function to print information about pproj if
    % print_info is not defined
    if ( isempty(print_info) == 1 )
        pproj ;
    end
    
    %---------------------------------------------------------------------------
    % Function used in installation routine
    function kk = do_cmd (s, kk, details)
        %DO_CMD: evaluate a command, and either print it or print a "."
        if (details)
            fprintf ('%s\n', s) ;
        else
            if (mod (kk, 60) == 0)
                fprintf ('\n') ;
            end
            kk = kk + 1 ;
            fprintf ('.') ;
        end
        eval (s) ; 
    end

end
